"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const client_wafv2_1 = require("@aws-sdk/client-wafv2"); // eslint-disable-line
const DELIMITER = ":";
const SCOPE = "CLOUDFRONT";
const client = new client_wafv2_1.WAFV2({
    region: "us-east-1",
    customUserAgent: "aws-pdk/static-website/waf",
});
const MAX_CREATE_RETRY = 10;
const RETRY_INTERVAL = 2000;
/**
 * Handler for creating a WAF V2 ACL in US-EAST-1.
 */
exports.onEvent = async (event) => {
    const { ID, MANAGED_RULES, CIDR_ALLOW_LIST } = event.ResourceProperties;
    const [WEB_ACL_ID, IP_SET_ID] = event.PhysicalResourceId
        ? event.PhysicalResourceId.split(DELIMITER)
        : [];
    let response = {};
    switch (event.RequestType) {
        case "Create":
            response = await createWaf(ID, MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Update":
            response = await updateWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID), MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Delete":
            response = await deleteWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID));
            break;
        default:
            throw new Error(`Invalid RequestType: ${event.RequestType}`);
    }
    return response;
};
/**
 * Generates the name of the IP Set.
 *
 * @param id param passed in.
 * @returns name of IP Set.
 */
const getIpSetName = (id) => `${id}-IPSet`;
/**
 * Returns a set of rules to apply.
 *
 * @param ipSetArn ip set arn
 * @param ipSetName  ip set name
 * @param managedRules  managed rules
 * @param cidrAllowList cidr allow list
 * @returns set of rules to apply.
 */
const getWafRules = (ipSetArn, ipSetName, managedRules, cidrAllowList) => {
    const rules = [];
    if (cidrAllowList) {
        rules.push({
            Name: ipSetName,
            Priority: 1,
            VisibilityConfig: {
                MetricName: ipSetName,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
            Action: {
                Block: {},
            },
            Statement: {
                NotStatement: {
                    Statement: {
                        IPSetReferenceStatement: {
                            ARN: ipSetArn,
                        },
                    },
                },
            },
        });
    }
    if (managedRules) {
        rules.push(...managedRules
            .map((r) => ({ VendorName: r.vendor, Name: r.name }))
            .map((rule, Priority) => ({
            Name: `${rule.VendorName}-${rule.Name}`,
            Priority,
            Statement: { ManagedRuleGroupStatement: rule },
            OverrideAction: { None: {} },
            VisibilityConfig: {
                MetricName: `${rule.VendorName}-${rule.Name}`,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
        })));
    }
    return rules;
};
const createWaf = async (id, managedRules, cidrAllowList) => {
    const ipSetName = getIpSetName(id);
    const createIpSetResponse = await client.createIPSet({
        Name: ipSetName,
        Scope: SCOPE,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        IPAddressVersion: cidrAllowList?.cidrType ?? "IPV4",
    });
    const createWebAclResponse = await createWafAcl(id, ipSetName, createIpSetResponse, managedRules, cidrAllowList);
    return {
        PhysicalResourceId: `${createWebAclResponse.Summary?.Id}${DELIMITER}${createIpSetResponse.Summary?.Id}`,
        Data: {
            WebAclArn: createWebAclResponse.Summary?.ARN,
            WebAclId: createWebAclResponse.Summary?.Id,
            IPSetArn: createIpSetResponse.Summary?.ARN,
            IPSetId: createIpSetResponse.Summary?.Id,
        },
    };
};
const createWafAcl = async (id, ipSetName, createIpSetResponse, managedRules, cidrAllowList) => {
    let counter = 0;
    while (true) {
        try {
            const createWebAclResponse = await client.createWebACL({
                Name: id,
                DefaultAction: { Allow: {} },
                Scope: SCOPE,
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: id,
                    SampledRequestsEnabled: true,
                },
                Rules: getWafRules(createIpSetResponse.Summary.ARN, ipSetName, managedRules, cidrAllowList),
            });
            return createWebAclResponse;
        }
        catch (e) {
            if (e instanceof client_wafv2_1.WAFUnavailableEntityException &&
                counter < MAX_CREATE_RETRY) {
                counter++;
                console.log(`Received error: ${e.message}; Waiting for retrying ${counter}`);
                await sleep(RETRY_INTERVAL);
                continue;
            }
            throw e;
        }
    }
};
const updateWaf = async (webAclId, ipSetId, id, ipSetName, managedRules, cidrAllowList) => {
    const getIpSetResponse = await client.getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    });
    await client.updateIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    });
    const getWebAclResponse = await client.getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    });
    await client.updateWebACL({
        Name: id,
        DefaultAction: { Allow: {} },
        Scope: SCOPE,
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: id,
            SampledRequestsEnabled: true,
        },
        Rules: getWafRules(getIpSetResponse.IPSet?.ARN, ipSetName, managedRules, cidrAllowList),
        Id: getWebAclResponse.WebACL?.Id,
        LockToken: getWebAclResponse.LockToken,
    });
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
const deleteWaf = async (webAclId, ipSetId, id, ipSetName) => {
    const getWebAclResponse = await client.getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    });
    await client.deleteWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
        LockToken: getWebAclResponse.LockToken,
    });
    const getIpSetResponse = await client.getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    });
    await client.deleteIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    });
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
const sleep = async (duration) => {
    return new Promise((resolve) => setTimeout(resolve, duration));
};
//# sourceMappingURL=data:application/json;base64,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