"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const crypto = require("crypto");
const client_s3_1 = require("@aws-sdk/client-s3");
const prepare_spec_1 = require("./prepare-spec");
const s3 = new client_s3_1.S3Client({
    customUserAgent: `aws-pdk/type-safe-api/prepare-spec`,
});
/**
 * Prepare the api spec for API Gateway
 * @param inputSpecLocation location of the specification to prepare
 * @param outputSpecLocation location to write the prepared spec to
 * @param options integrations, authorizers etc to apply
 * @return the output location of the prepared spec
 */
const prepare = async ({ inputSpecLocation, outputSpecLocation, ...options }) => {
    // Read the spec from the s3 input location
    const inputSpec = JSON.parse(await (await s3.send(new client_s3_1.GetObjectCommand({
        Bucket: inputSpecLocation.bucket,
        Key: inputSpecLocation.key,
    }))).Body.transformToString("utf-8"));
    // Prepare the spec
    const preparedSpec = (0, prepare_spec_1.prepareApiSpec)(inputSpec, options);
    const preparedSpecHash = crypto
        .createHash("sha256")
        .update(JSON.stringify(preparedSpec))
        .digest("hex");
    const outputLocation = {
        bucket: outputSpecLocation.bucket,
        key: `${outputSpecLocation.key}/${preparedSpecHash}.json`,
    };
    // Write the spec to the s3 output location
    await s3.send(new client_s3_1.PutObjectCommand({
        Bucket: outputLocation.bucket,
        Key: outputLocation.key,
        Body: JSON.stringify(preparedSpec),
    }));
    return outputLocation;
};
exports.handler = async (event) => {
    switch (event.RequestType) {
        case "Create":
        case "Update":
            // Prepare the spec on create
            const outputLocation = await prepare(event.ResourceProperties);
            return {
                PhysicalResourceId: outputLocation.key,
                Status: "SUCCESS",
                Data: {
                    outputSpecKey: outputLocation.key,
                },
            };
        case "Delete":
        // Nothing to do for delete
        default:
            break;
    }
    return {
        PhysicalResourceId: event.PhysicalResourceId,
        Status: "SUCCESS",
    };
};
//# sourceMappingURL=data:application/json;base64,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