"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.validatePathItem = exports.generateCorsResponseParameters = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
const concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
exports.concatMethodAndPath = concatMethodAndPath;
/**
 * Return an array of security scheme references including the api key one if required
 */
const apiKeySecurityReference = (options) => options?.apiKeyRequired ? [{ [constants_1.DefaultAuthorizerIds.API_KEY]: [] }] : [];
/**
 * Generate a "no auth" spec snippet
 */
const noAuthSpecSnippet = (options) => ({
    security: apiKeySecurityReference(options),
    "x-amazon-apigateway-auth": {
        type: "NONE",
    },
});
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 * @param options api integration options
 */
const applyMethodAuthorizer = (methodAuthorizer, options) => {
    if (methodAuthorizer || options) {
        if (methodAuthorizer?.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return noAuthSpecSnippet(options);
        }
        else {
            return {
                security: [
                    ...(methodAuthorizer
                        ? [
                            {
                                [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                            },
                        ]
                        : []),
                    ...apiKeySecurityReference(options),
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions, apiKeyOptions, defaultAuthorizerReference, }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    let { methodAuthorizer, integration, options } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    let methodApiKeyOptions = options;
    // When no API key options are present on the method, require the API key if it's
    // required by default
    if (!methodApiKeyOptions && apiKeyOptions?.requiredByDefault) {
        methodApiKeyOptions = { apiKeyRequired: true };
    }
    // Can only "require" an API key if it's in a header, since we only define the security
    // scheme we'd reference in this case.
    if (apiKeyOptions?.source !== "HEADER" &&
        methodApiKeyOptions?.apiKeyRequired) {
        throw new Error(`Cannot require an API Key when API Key source is not HEADER: ${operationName} (${method} ${path})`);
    }
    // Apply the default authorizer unless a method authorizer is defined
    methodAuthorizer = methodAuthorizer ?? defaultAuthorizerReference;
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": integration,
        ...applyMethodAuthorizer(methodAuthorizer, methodApiKeyOptions),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
exports.generateCorsResponseParameters = generateCorsResponseParameters;
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: (0, exports.generateCorsResponseParameters)(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
            // No auth for CORS options requests
            ...noAuthSpecSnippet(),
        },
    };
};
const validatePathItem = (path, pathItem) => {
    const supportedPathItemKeys = new Set([
        // https://spec.openapis.org/oas/v3.0.3#path-item-object
        ...Object.values(constants_1.HttpMethods),
        "summary",
        "description",
        "parameters",
        "servers",
        // All $refs should be resolved already, so we'll error if one remains somehow
    ]);
    const unsupportedMethodsInSpec = Object.keys(pathItem).filter((method) => !supportedPathItemKeys.has(method));
    if (unsupportedMethodsInSpec.length > 0) {
        throw new Error(`Path ${path} contains unsupported method${unsupportedMethodsInSpec.length > 1 ? "s" : ""} ${unsupportedMethodsInSpec.join(", ")}. Supported methods are ${Object.values(constants_1.HttpMethods).join(", ")}.`);
    }
};
exports.validatePathItem = validatePathItem;
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    (0, exports.validatePathItem)(path, pathItem);
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Find all unique header parameters used in operations
 */
const findHeaderParameters = (spec) => {
    const allHeaderParameters = Object.values(spec.paths).flatMap((pathDetails) => Object.values(constants_1.HttpMethods).flatMap((method) => (pathDetails?.[method]?.parameters ?? []).flatMap((parameter) => "in" in parameter && parameter.in === "header" ? [parameter.name] : [])));
    const headerParameterSet = new Set();
    return allHeaderParameters.filter((p) => {
        const seen = headerParameterSet.has(p);
        headerParameterSet.add(p);
        return !seen;
    });
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
const prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        (0, exports.concatMethodAndPath)(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[(0, exports.concatMethodAndPath)(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    // If there are cors options, add any header parameters defined in the spec as allowed headers to
    // save users from having to manually specify these (or face cors issues!)
    const corsOptions = options.corsOptions
        ? {
            ...options.corsOptions,
            allowHeaders: [
                ...options.corsOptions.allowHeaders,
                ...findHeaderParameters(spec),
            ],
        }
        : undefined;
    const updatedOptions = {
        ...options,
        corsOptions,
    };
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(corsOptions
                    ? {
                        responseParameters: (0, exports.generateCorsResponseParameters)(corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, updatedOptions, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...updatedOptions.securitySchemes,
            },
        },
        ...(updatedOptions.apiKeyOptions
            ? {
                // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-api-key-source.html
                "x-amazon-apigateway-api-key-source": updatedOptions.apiKeyOptions.source,
            }
            : {}),
    };
};
exports.prepareApiSpec = prepareApiSpec;
//# sourceMappingURL=data:application/json;base64,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