"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const client_apigatewayv2_1 = require("@aws-sdk/client-apigatewayv2");
const client_s3_1 = require("@aws-sdk/client-s3");
const websocket_schema_1 = require("./websocket-schema");
const BATCH_SIZE = 10;
const s3 = new client_s3_1.S3({
    customUserAgent: `aws-pdk/type-safe-api/ws-schema`,
});
const apiGw = new client_apigatewayv2_1.ApiGatewayV2({
    customUserAgent: `aws-pdk/type-safe-api/ws-schema`,
});
/**
 * Chunk an array into sub-arrays of the given size
 */
const chunk = (items, size = BATCH_SIZE) => {
    const chunks = [];
    for (let i = 0; i < items.length; i += size) {
        chunks.push(items.slice(i, i + size));
    }
    return chunks;
};
/**
 * Delete a batch of models
 */
const batchDeleteModels = async (apiId, routes, models) => {
    for (const batch of chunk(models)) {
        await Promise.all(batch.map(async (m) => {
            // If there's a route for this model, and it's associated with this model, we need to first remove the association,
            // since cloudformation will delete the route later, and we're not allowed to delete a model that's still referenced
            // by a route
            if (routes[m.Name] &&
                routes[m.Name].RequestModels?.model === m.Name) {
                await apiGw.updateRoute({
                    ...routes[m.Name],
                    ApiId: apiId,
                    RouteId: routes[m.Name].RouteId,
                    RequestModels: {
                        model: "",
                    },
                    ModelSelectionExpression: undefined,
                });
            }
            await apiGw.deleteModel({
                ApiId: apiId,
                ModelId: m.ModelId,
            });
        }));
    }
};
/**
 * Retrieve all models which already exist on the api
 */
const getAllModelsByRouteKey = async (apiId) => {
    let nextToken = undefined;
    const models = [];
    do {
        const res = await apiGw.getModels({
            ApiId: apiId,
            NextToken: nextToken,
        });
        nextToken = res.NextToken;
        models.push(...(res.Items ?? []));
    } while (nextToken);
    // Models are named with the route key
    return Object.fromEntries(models.map((m) => [m.Name, m]));
};
const getAllRoutesByRouteKey = async (apiId) => {
    let nextToken = undefined;
    const routes = [];
    do {
        const res = await apiGw.getRoutes({
            ApiId: apiId,
            NextToken: nextToken,
        });
        nextToken = res.NextToken;
        routes.push(...(res.Items ?? []));
    } while (nextToken);
    return Object.fromEntries(routes.map((r) => [r.RouteKey, r]));
};
/**
 * Wrap the schema from the spec in our protocol-specific schema
 */
const wrapSchema = (schema) => ({
    type: "object",
    properties: {
        // All inputs must have a "route" which is our route selector
        route: {
            type: "string",
        },
        // Payload references the definition
        payload: {
            $ref: "#/definitions/Payload",
        },
    },
    // When we don't have a schema, the payload can be anything, including not specified
    required: ["route", ...(schema ? ["payload"] : [])],
    definitions: {
        ...schema?.definitions,
        // The payload is of the operation schema type, or {} which means "any"
        Payload: schema?.schema ?? {},
    },
});
/**
 * Create a batch of models with the appropriate schema
 */
const batchCreateModels = async (apiId, routeKeys, schemas) => {
    const results = [];
    for (const batch of chunk(routeKeys)) {
        results.push(...(await Promise.all(batch.map(async (routeKey) => apiGw.createModel({
            ApiId: apiId,
            Name: routeKey,
            ContentType: "application/json",
            Schema: JSON.stringify(wrapSchema(schemas[routeKey])),
        })))));
    }
    return Object.fromEntries(results.map((r) => [r.Name, r.ModelId]));
};
/**
 * Update a batch of models with the new schema
 */
const batchUpdateModels = async (apiId, models, schemas) => {
    const results = [];
    for (const batch of chunk(models)) {
        results.push(...(await Promise.all(batch.map(async (model) => apiGw.updateModel({
            ApiId: apiId,
            ModelId: model.ModelId,
            ContentType: "application/json",
            Schema: JSON.stringify(wrapSchema(schemas[model.Name])),
        })))));
    }
    return Object.fromEntries(results.map((r) => [r.Name, r.ModelId]));
};
/**
 * Create or update the models
 */
const createOrUpdateModels = async (properties, routes) => {
    const modelsByRouteKey = await getAllModelsByRouteKey(properties.apiId);
    const existingRouteKeys = new Set(Object.keys(modelsByRouteKey));
    const newRouteKeys = new Set(Object.keys(properties.serverOperationPaths));
    const deletedRouteKeys = [...existingRouteKeys].filter((id) => !newRouteKeys.has(id));
    console.log("Operations to delete", deletedRouteKeys);
    const addedRouteKeys = [...newRouteKeys].filter((id) => !existingRouteKeys.has(id));
    console.log("Operations to add", addedRouteKeys);
    const updateRouteKeys = [...newRouteKeys].filter((id) => existingRouteKeys.has(id));
    console.log("Operations to update", updateRouteKeys);
    // Delete all the models to delete
    await batchDeleteModels(properties.apiId, routes, deletedRouteKeys.map((id) => modelsByRouteKey[id]));
    // Load the spec
    const spec = JSON.parse(await (await s3.getObject({
        Bucket: properties.inputSpecLocation.bucket,
        Key: properties.inputSpecLocation.key,
    })).Body.transformToString("utf-8"));
    // Extract the schemas from the spec
    const schemas = (0, websocket_schema_1.extractWebSocketSchemas)([...addedRouteKeys, ...updateRouteKeys], properties.serverOperationPaths, spec);
    // Create/update the relevant models
    return {
        ...(await batchCreateModels(properties.apiId, addedRouteKeys.filter((id) => schemas[id]), schemas)),
        ...(await batchUpdateModels(properties.apiId, updateRouteKeys
            .filter((id) => schemas[id])
            .map((id) => modelsByRouteKey[id]), schemas)),
    };
};
/**
 * Delete all models
 */
const deleteModels = async (properties, routes) => {
    const modelsByRouteKey = await getAllModelsByRouteKey(properties.apiId);
    await batchDeleteModels(properties.apiId, routes, Object.values(modelsByRouteKey));
};
/**
 * Handler for creating websocket schemas
 */
exports.handler = async (event) => {
    const PhysicalResourceId = event.PhysicalResourceId ?? `${event.ResourceProperties.apiId}-models`;
    const routes = await getAllRoutesByRouteKey(event.ResourceProperties.apiId);
    switch (event.RequestType) {
        case "Create":
        case "Update":
            await createOrUpdateModels(event.ResourceProperties, routes);
            break;
        case "Delete":
            await deleteModels(event.ResourceProperties, routes);
            break;
        default:
            break;
    }
    return {
        PhysicalResourceId,
    };
};
//# sourceMappingURL=data:application/json;base64,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