"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractWebSocketSchemas = exports.extractWebSocketSchema = void 0;
const constants_1 = require("./constants");
const prepare_spec_1 = require("./prepare-spec");
const _get = (object, paths) => {
    if (typeof object === "undefined") {
        return undefined;
    }
    if (paths.length === 0) {
        return object;
    }
    return _get(object[paths[0]], paths.slice(1));
};
/**
 * Return whether or not the given OpenAPI object is a reference
 */
const isRef = (obj) => !!obj && typeof obj === "object" && "$ref" in obj;
/**
 * Return whether a given OpenAPI object is a schema object
 */
const isSchemaObj = (obj) => !!obj &&
    typeof obj === "object" &&
    ("type" in obj ||
        "allOf" in obj ||
        "oneOf" in obj ||
        "anyOf" in obj ||
        "not" in obj);
/**
 * Split a reference into its component parts
 * eg: #/components/schemas/Foo -> ["components", "schemas", "Foo"]
 */
const splitRef = (ref) => ref
    .slice(2)
    .split("/")
    .map((p) => p.replace(/~0/g, "~").replace(/~1/g, "/"));
/**
 * Resolve the given reference in the spec
 */
const resolveRef = (spec, ref) => {
    const refParts = splitRef(ref);
    const resolved = _get(spec, refParts);
    if (!resolved) {
        throw new Error(`Unable to resolve ref ${ref} in spec`);
    }
    return resolved;
};
/**
 * Get the id of a reference to be used in the "definitions" section
 */
const getRefId = (ref) => splitRef(ref).join("_");
/**
 * Rewrite a reference to an API gateway model supported format
 * eg #/components/schemas/Foo -> #/definitions/components_schemas_Foo
 */
const rewriteRef = (ref) => `#/definitions/${getRefId(ref)}`;
/**
 * Map the given function over all refs in an OpenAPI object
 */
const mapRefs = (obj, fn) => {
    // Use JSON.stringify's replacement function to simplify traversing a spec
    return JSON.parse(JSON.stringify(obj, (key, value) => {
        if (key === "$ref") {
            return fn(value);
        }
        return value;
    }));
};
/**
 * Find all references recursively starting at the given schema
 */
const findAllReferences = (schema, spec, seenRefs = new Set()) => {
    const newRefs = new Set();
    mapRefs(schema, (ref) => newRefs.add(ref));
    const refsToSearch = [...newRefs].filter((ref) => !seenRefs.has(ref));
    const newSeenRefs = new Set([...newRefs, ...seenRefs]);
    return new Set([
        ...newSeenRefs,
        ...refsToSearch.flatMap((ref) => [
            ...findAllReferences(resolveRef(spec, ref), spec, newSeenRefs),
        ]),
    ]);
};
/**
 * Rewrite all references in the schema to instead reference #/definitions/xxx, and collect any other schemas recursively referenced by the schema
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/models-mappings-models.html#api-gateway-request-validation-model-more-complex
 */
const rewriteSchemaReferences = (schema, spec) => {
    // Rewrite the schema and spec, replacing references with #/definitions/xxx as per the API Gateway model format
    const rewrittenSchema = mapRefs(schema, rewriteRef);
    const rewrittenSpec = mapRefs(spec, rewriteRef);
    // Set of definitions that must be included for the given schema
    const definitions = {};
    // Find all recursive references from the schema, and add their (rewritten) schema to definitions
    [...findAllReferences(schema, spec)].forEach((ref) => {
        definitions[getRefId(ref)] = resolveRef(rewrittenSpec, ref);
    });
    return { schema: rewrittenSchema, definitions };
};
const extractWebSocketSchema = (operationId, requestBody, spec) => {
    // Resolve the body reference, if any
    const body = isRef(requestBody)
        ? resolveRef(spec, requestBody.$ref)
        : requestBody;
    const candidateSchema = body?.content?.["application/json"]?.schema;
    if (!candidateSchema) {
        // No schema found
        return undefined;
    }
    // Resolve the top level schema reference, if any
    const rawSchema = isRef(candidateSchema)
        ? resolveRef(spec, candidateSchema.$ref)
        : candidateSchema;
    if (!isSchemaObj(rawSchema)) {
        throw new Error(`Invalid OpenAPI specification: request body for operation ${operationId} is not a valid schema`);
    }
    // Rewrite schema references to a format accepted by API Gateway
    return rewriteSchemaReferences(rawSchema, spec);
};
exports.extractWebSocketSchema = extractWebSocketSchema;
const extractWebSocketSchemas = (operationIds, serverOperationPaths, spec) => {
    const schemasByOperationId = {};
    for (const operationId of operationIds) {
        const path = serverOperationPaths[operationId];
        const pathItem = spec.paths[path];
        (0, prepare_spec_1.validatePathItem)(path, pathItem);
        // Exactly 1 operation must be defined for each path in a websocket OpenAPI spec
        const operations = Object.values(constants_1.HttpMethods).flatMap((method) => pathItem[method] ? [{ ...pathItem[method], method }] : []);
        if (operations.length !== 1) {
            throw new Error(`Each path must have a single method for websocket apis. Found ${operations.map((o) => o.method).join(", ") || "no methods"}`);
        }
        // Extract the schema for the websocket input validation model
        if (operations[0]?.requestBody) {
            const schema = (0, exports.extractWebSocketSchema)(operationId, operations[0].requestBody, spec);
            if (schema) {
                schemasByOperationId[operationId] = schema;
            }
        }
    }
    return schemasByOperationId;
};
exports.extractWebSocketSchemas = extractWebSocketSchemas;
//# sourceMappingURL=data:application/json;base64,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