"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayWebAcl = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("../../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const constructs_1 = require("constructs");
/**
 * Associate an AWS WAF v2 Web ACL with the given api
 */
class OpenApiGatewayWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const aclName = `${pdk_nag_1.PDKNag.getStackPrefix(aws_cdk_lib_1.Stack.of(this))
            .split("/")
            .join("-")}${id}-${this.node.addr.slice(-8)}`;
        const ipSetName = `${aclName}-IPSet`;
        // Create the IP Set if requested
        this.ipSet = props.cidrAllowList
            ? new aws_wafv2_1.CfnIPSet(this, "ApiIPSet", {
                name: ipSetName,
                addresses: props.cidrAllowList.cidrRanges,
                ipAddressVersion: props.cidrAllowList.cidrType,
                scope: "REGIONAL",
            })
            : undefined;
        // TODO: vendor property is deprecated, to be removed in the future iterations
        // and vendorName will be required
        const anyMissingVendor = props.managedRules?.some((q) => !q.vendorName && !q.vendor);
        if (anyMissingVendor) {
            throw new Error("The provided managed rules need to define either the vendor or vendorName (preferred) property");
        }
        const managedRules = props.managedRules ?? [
            { vendorName: "AWS", name: "AWSManagedRulesCommonRuleSet" },
        ];
        const rules = [
            // Add a rule for the IP Set if specified
            ...(this.ipSet
                ? [
                    {
                        name: ipSetName,
                        priority: 1,
                        visibilityConfig: {
                            metricName: ipSetName,
                            cloudWatchMetricsEnabled: true,
                            sampledRequestsEnabled: true,
                        },
                        action: {
                            block: {},
                        },
                        statement: {
                            notStatement: {
                                statement: {
                                    ipSetReferenceStatement: {
                                        arn: this.ipSet.attrArn,
                                    },
                                },
                            },
                        },
                    },
                ]
                : []),
            // Add the managed rules
            ...managedRules.map(({ vendor, vendorName, name, ...others }, i) => {
                // TODO: the usage of `vendor` it's for backward compatibility
                // it will be removed in the next PDK versions
                const vendorNameToUser = (vendor || vendorName);
                return {
                    name: `${vendorNameToUser}-${name}`,
                    priority: i + 2,
                    statement: {
                        managedRuleGroupStatement: {
                            ...others,
                            vendorName: vendorNameToUser,
                            name,
                        },
                    },
                    overrideAction: { none: {} },
                    visibilityConfig: {
                        metricName: `${aclName}-${vendorNameToUser}-${name}`,
                        cloudWatchMetricsEnabled: true,
                        sampledRequestsEnabled: true,
                    },
                };
            }),
        ];
        this.webAcl = new aws_wafv2_1.CfnWebACL(this, "ApiWebACL", {
            name: aclName,
            defaultAction: {
                // Allow by default, and use rules to deny unwanted requests
                allow: {},
            },
            scope: "REGIONAL",
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: aclName,
            },
            rules,
        });
        this.webAclAssociation = new aws_wafv2_1.CfnWebACLAssociation(this, "WebACLAssociation", {
            resourceArn: props.apiDeploymentStageArn,
            webAclArn: this.webAcl.attrArn,
        });
    }
}
exports.OpenApiGatewayWebAcl = OpenApiGatewayWebAcl;
//# sourceMappingURL=data:application/json;base64,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