"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiToolsJsonFile = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const node_assert_1 = require("node:assert");
const path_1 = require("path");
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
/**
 * Represents an openapitools.json file
 * @see https://github.com/OpenAPITools/openapi-generator-cli#configuration
 */
class OpenApiToolsJsonFile extends projen_1.JsonFile {
    /**
     * Retrieves an instance of OpenApiToolsJsonFile if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => monorepo_1.ProjectUtils.isNamedInstanceOf(c, OpenApiToolsJsonFile));
    }
    /**
     * Retrieves an instance of OpenApiToolsJsonFile if one is associated to the given project,
     * otherwise creates a OpenApiToolsJsonFile instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project) {
        return (OpenApiToolsJsonFile.of(project) || new OpenApiToolsJsonFile(project));
    }
    constructor(project) {
        if (OpenApiToolsJsonFile.of(project)) {
            throw new Error(`Project ${project.name} already has associated OpenApiToolsJsonFile component.`);
        }
        const localFilePath = "openapitools.json";
        const dynamicFilePath = path_1.posix.join(".pdk/dynamic-files", localFilePath);
        super(project, dynamicFilePath, {
            obj: {
                // Schema is located in node_modules when generator cli is installed in tmp dir
                $schema: "node_modules/@openapitools/openapi-generator-cli/config.schema.json",
                spaces: 2,
                "generator-cli": () => this.config,
            },
        });
        /**
         * Configuration for OpenAPI Generator CLI
         * @private
         */
        this.config = {
            version: "6.3.0",
            storageDir: "~/.open-api-generator-cli",
        };
        // Ignore the location that gets copied to
        project.addGitIgnore(`/${localFilePath}`);
        // Create the task, but don't attach it anywhere yet. This is because of wanting
        // to do the generation as part of the "generate" tasks, which may not yet exist.
        this.createTask = project.addTask(`create-openapitools.json`, {
            steps: [{ exec: `cp -f ${dynamicFilePath} ${localFilePath}` }],
        });
    }
    preSynthesize() {
        const generateTask = this.project.tasks.tryFind("generate");
        (0, node_assert_1.ok)(generateTask);
        generateTask.prependSpawn(this.createTask);
    }
    /**
     * Update the OpenAPI Generator config. Shallow-merges with any existing default config
     */
    addOpenApiGeneratorCliConfig(config) {
        this.config = {
            ...this.config,
            ...config,
        };
    }
}
exports.OpenApiToolsJsonFile = OpenApiToolsJsonFile;
//# sourceMappingURL=data:application/json;base64,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