"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHandlersProjectVendorExtensions = exports.buildInvokeMockDataGeneratorCommand = exports.buildCleanOpenApiGeneratedCodeCommand = exports.buildInvokeOpenApiGeneratorCommandArgs = exports.buildTypeSafeApiExecCommand = exports.getTypeSafeApiTaskEnvironment = exports.TypeSafeApiScript = exports.OtherGenerators = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../../../monorepo");
const runtime_version_utils_1 = require("../runtime-version-utils");
/**
 * Enum for generator directories for non-runtime generators
 */
var OtherGenerators;
(function (OtherGenerators) {
    OtherGenerators["DOCS"] = "docs";
    // Infrastructure
    OtherGenerators["TYPESCRIPT_CDK_INFRASTRUCTURE"] = "typescript-cdk-infrastructure";
    OtherGenerators["PYTHON_CDK_INFRASTRUCTURE"] = "python-cdk-infrastructure";
    OtherGenerators["JAVA_CDK_INFRASTRUCTURE"] = "java-cdk-infrastructure";
    OtherGenerators["TYPESCRIPT_ASYNC_CDK_INFRASTRUCTURE"] = "typescript-async-cdk-infrastructure";
    OtherGenerators["PYTHON_ASYNC_CDK_INFRASTRUCTURE"] = "python-async-cdk-infrastructure";
    OtherGenerators["JAVA_ASYNC_CDK_INFRASTRUCTURE"] = "java-async-cdk-infrastructure";
    // Handlers
    OtherGenerators["TYPESCRIPT_LAMBDA_HANDLERS"] = "typescript-lambda-handlers";
    OtherGenerators["PYTHON_LAMBDA_HANDLERS"] = "python-lambda-handlers";
    OtherGenerators["JAVA_LAMBDA_HANDLERS"] = "java-lambda-handlers";
    OtherGenerators["TYPESCRIPT_ASYNC_LAMBDA_HANDLERS"] = "typescript-async-lambda-handlers";
    OtherGenerators["PYTHON_ASYNC_LAMBDA_HANDLERS"] = "python-async-lambda-handlers";
    OtherGenerators["JAVA_ASYNC_LAMBDA_HANDLERS"] = "java-async-lambda-handlers";
    // Async runtime
    OtherGenerators["TYPESCRIPT_ASYNC_RUNTIME"] = "typescript-async-runtime";
    OtherGenerators["PYTHON_ASYNC_RUNTIME"] = "python-async-runtime";
    OtherGenerators["JAVA_ASYNC_RUNTIME"] = "java-async-runtime";
})(OtherGenerators || (exports.OtherGenerators = OtherGenerators = {}));
/**
 * Built in scripts.
 * If adding a script here, ensure you map it in TypeSafeApiProject (in /projenrc/projects)
 */
var TypeSafeApiScript;
(function (TypeSafeApiScript) {
    TypeSafeApiScript["PARSE_OPENAPI_SPEC"] = "type-safe-api.parse-openapi-spec";
    TypeSafeApiScript["GENERATE"] = "type-safe-api.generate";
    TypeSafeApiScript["GENERATE_MOCK_DATA"] = "type-safe-api.generate-mock-data";
    TypeSafeApiScript["GENERATE_HTML_REDOC_DOCS"] = "type-safe-api.generate-html-redoc-docs";
    TypeSafeApiScript["GENERATE_ASYNCAPI_HTML_DOCS"] = "type-safe-api.generate-asyncapi-html-docs";
    TypeSafeApiScript["GENERATE_ASYNCAPI_MARKDOWN_DOCS"] = "type-safe-api.generate-asyncapi-markdown-docs";
    TypeSafeApiScript["CLEAN_OPENAPI_GENERATED_CODE"] = "type-safe-api.clean-openapi-generated-code";
    TypeSafeApiScript["COPY_GRADLE_WRAPPER"] = "type-safe-api.copy-gradle-wrapper";
    TypeSafeApiScript["COPY_ASYNC_SMITHY_TRANSFORMER"] = "type-safe-api.copy-async-smithy-transformer";
    TypeSafeApiScript["GENERATE_ASYNCAPI_SPEC"] = "type-safe-api.generate-asyncapi-spec";
})(TypeSafeApiScript || (exports.TypeSafeApiScript = TypeSafeApiScript = {}));
/**
 * Return the environment that should be used for executing type safe api commands
 */
const getTypeSafeApiTaskEnvironment = () => ({
    AWS_PDK_VERSION: monorepo_1.ProjectUtils.getPdkVersion(),
});
exports.getTypeSafeApiTaskEnvironment = getTypeSafeApiTaskEnvironment;
/**
 * Build a command for running a script from this project's bin
 */
const buildTypeSafeApiExecCommand = (script, args) => {
    return `npx --yes -p @aws/pdk@$AWS_PDK_VERSION ${script}${args ? ` ${args}` : ""}`;
};
exports.buildTypeSafeApiExecCommand = buildTypeSafeApiExecCommand;
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Generate code or docs by invoking the root generate script
 */
const buildInvokeOpenApiGeneratorCommandArgs = (options) => {
    const srcDir = options.srcDir ?? "src";
    const tstDir = options.tstDir ?? "test";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    const normalizers = options.normalizers
        ? ` --openapi-normalizer "${serializeProperties(Object.fromEntries(Object.entries(options.normalizers).map(([k, v]) => [k, `${v}`])))}"`
        : "";
    const extensions = options.extraVendorExtensions
        ? ` --extra-vendor-extensions '${JSON.stringify(options.extraVendorExtensions)}'`
        : "";
    const generateAliasAsModel = options.generateAliasAsModel ?? true ? " --generate-alias-as-model" : "";
    const specPath = options.specPath;
    const outputPath = ".";
    return `--generator ${options.generator} --spec-path ${specPath} --output-path ${outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir} --tst-dir ${tstDir}${additionalProperties}${normalizers}${extensions}${generateAliasAsModel}`;
};
exports.buildInvokeOpenApiGeneratorCommandArgs = buildInvokeOpenApiGeneratorCommandArgs;
/**
 * Builds a command to clean up files which were previously generated by openapi generator
 */
const buildCleanOpenApiGeneratedCodeCommand = () => {
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.CLEAN_OPENAPI_GENERATED_CODE, `--code-path .`);
};
exports.buildCleanOpenApiGeneratedCodeCommand = buildCleanOpenApiGeneratedCodeCommand;
/**
 * Invoke the mock data generator script
 */
const buildInvokeMockDataGeneratorCommand = (options) => {
    const outputPath = options.outputSubDir ?? ".";
    const locale = options.locale ? ` --locale ${options.locale}` : "";
    const maxArrayLength = options.maxArrayLength !== undefined
        ? ` --max-array-length ${options.maxArrayLength}`
        : "";
    const seed = options.seed !== undefined ? ` --seed ${options.seed}` : "";
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.GENERATE_MOCK_DATA, `--spec-path ${options.specPath} --output-path ${outputPath}${locale}${maxArrayLength}${seed}`);
};
exports.buildInvokeMockDataGeneratorCommand = buildInvokeMockDataGeneratorCommand;
/**
 * Return vendor extensions containing details about the handler projects
 */
const getHandlersProjectVendorExtensions = (targetProject, { java, python, typescript }) => ({
    "x-handlers-python-module": python?.moduleName ?? "",
    "x-handlers-java-package": java?.packageName ?? "",
    "x-handlers-typescript-asset-path": typescript
        ? path.join(path.relative(targetProject.outdir, typescript.outdir), "dist", "lambda")
        : "",
    "x-handlers-python-asset-path": python
        ? path.join(path.relative(targetProject.outdir, python.outdir), "dist", "lambda")
        : "",
    "x-handlers-java-asset-path": java
        ? path.join(path.relative(targetProject.outdir, java.outdir), java.distdir, ...java.pom.groupId.split("."), java.pom.artifactId, java.pom.version, `${java.pom.artifactId}-${java.pom.version}.jar`)
        : "",
    "x-handlers-node-lambda-runtime-version": typescript
        ? runtime_version_utils_1.RuntimeVersionUtils.NODE.getLambdaRuntime(typescript.runtimeVersion)
        : "",
    "x-handlers-python-lambda-runtime-version": python
        ? runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getLambdaRuntime(python.runtimeVersion)
        : "",
    "x-handlers-java-lambda-runtime-version": java
        ? runtime_version_utils_1.RuntimeVersionUtils.JAVA.getLambdaRuntime(java.runtimeVersion)
        : "",
});
exports.getHandlersProjectVendorExtensions = getHandlersProjectVendorExtensions;
//# sourceMappingURL=data:application/json;base64,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