"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateDocsProjects = exports.generateAsyncLibraryProjects = exports.generateLibraryProjects = exports.generateAsyncRuntimeProjects = exports.generateRuntimeProjects = exports.generateAsyncHandlersProjects = exports.generateHandlersProjects = exports._generateInfraProject = exports.generateAsyncInfraProject = exports.generateInfraProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const generated_asyncapi_html_documentation_project_1 = require("./documentation/generated-asyncapi-html-documentation-project");
const generated_asyncapi_markdown_documentation_project_1 = require("./documentation/generated-asyncapi-markdown-documentation-project");
const generated_html_redoc_documentation_project_1 = require("./documentation/generated-html-redoc-documentation-project");
const generated_html2_documentation_project_1 = require("./documentation/generated-html2-documentation-project");
const generated_markdown_documentation_project_1 = require("./documentation/generated-markdown-documentation-project");
const generated_plantuml_documentation_project_1 = require("./documentation/generated-plantuml-documentation-project");
const generated_java_async_handlers_project_1 = require("./handlers/generated-java-async-handlers-project");
const generated_java_handlers_project_1 = require("./handlers/generated-java-handlers-project");
const generated_python_async_handlers_project_1 = require("./handlers/generated-python-async-handlers-project");
const generated_python_handlers_project_1 = require("./handlers/generated-python-handlers-project");
const generated_typescript_async_handlers_project_1 = require("./handlers/generated-typescript-async-handlers-project");
const generated_typescript_handlers_project_1 = require("./handlers/generated-typescript-handlers-project");
const generated_java_async_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-java-async-cdk-infrastructure-project");
const generated_java_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-java-cdk-infrastructure-project");
const generated_python_async_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-python-async-cdk-infrastructure-project");
const generated_python_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-python-cdk-infrastructure-project");
const generated_typescript_async_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-typescript-async-cdk-infrastructure-project");
const generated_typescript_cdk_infrastructure_project_1 = require("./infrastructure/cdk/generated-typescript-cdk-infrastructure-project");
const typescript_react_query_hooks_library_1 = require("./library/typescript-react-query-hooks-library");
const typescript_websocket_client_library_1 = require("./library/typescript-websocket-client-library");
const typescript_websocket_hooks_library_1 = require("./library/typescript-websocket-hooks-library");
const generated_java_async_runtime_project_1 = require("./runtime/generated-java-async-runtime-project");
const generated_java_runtime_project_1 = require("./runtime/generated-java-runtime-project");
const generated_python_async_runtime_project_1 = require("./runtime/generated-python-async-runtime-project");
const generated_python_runtime_project_1 = require("./runtime/generated-python-runtime-project");
const generated_typescript_async_runtime_project_1 = require("./runtime/generated-typescript-async-runtime-project");
const generated_typescript_runtime_project_1 = require("./runtime/generated-typescript-runtime-project");
const languages_1 = require("../languages");
const logger = (0, log4js_1.getLogger)();
// No dashes or underscores since this is used in the package name in imports
const sanitiseJavaProjectName = (name) => name.replace(/@/g, "").replace(/[\-/_]/g, "");
// kebab-case for java artifact ids
const sanitiseJavaArtifactId = (name) => name.replace(/@/g, "").replace(/[/_]/g, "-");
// kebab-case for typescript packages
const sanitiseTypescriptPackageName = (name) => name.replace(/_/g, "-");
// snake_case for python modules
const sanitisePythonModuleName = (name) => name.replace(/@/g, "").replace(/[\-/]/g, "_");
// Use dashes in project name since distributable's PKG-INFO always converts _ to -
// https://stackoverflow.com/questions/36300788/python-package-wheel-pkg-info-name
const sanitisePythonPackageName = (name) => name.replace(/@/g, "").replace(/[_/]/g, "-");
const generateLanguageProject = (suffix, language, projectConstructors, options) => {
    const packageName = `${options.parentPackageName}-${language}-${suffix}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, language),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (language) {
        case languages_1.Language.TYPESCRIPT: {
            logger.trace(`Attempting to generate TYPESCRIPT ${suffix} project.`);
            return new projectConstructors[language]({
                ...options,
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.Language.PYTHON: {
            logger.trace(`Attempting to generate PYTHON ${suffix} project.`);
            return new projectConstructors[language]({
                ...options,
                ...commonOptions,
                name: sanitisePythonPackageName(packageName),
                moduleName: sanitisePythonModuleName(packageName),
                ...options.pythonOptions,
            });
        }
        case languages_1.Language.JAVA: {
            logger.trace(`Attempting to generate JAVA ${suffix} project.`);
            return new projectConstructors[language]({
                ...options,
                ...commonOptions,
                name: sanitiseJavaProjectName(packageName),
                artifactId: sanitiseJavaArtifactId(packageName),
                groupId: "com.generated.api",
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown ${suffix} language ${language}`);
    }
};
const generateLanguageProjects = (suffix, languages, projectConstructors, options) => {
    const projects = {};
    languages.forEach((language) => {
        const project = generateLanguageProject(suffix, language, projectConstructors, options);
        if (project != null) {
            projects[language] = project;
        }
    });
    return projects;
};
/**
 * Returns a generated infrastructure project for the given language
 */
const generateInfraProject = (language, options) => {
    return (0, exports._generateInfraProject)(language, {
        [languages_1.Language.JAVA]: generated_java_cdk_infrastructure_project_1.GeneratedJavaCdkInfrastructureProject,
        [languages_1.Language.PYTHON]: generated_python_cdk_infrastructure_project_1.GeneratedPythonCdkInfrastructureProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_cdk_infrastructure_project_1.GeneratedTypescriptCdkInfrastructureProject,
    }, options);
};
exports.generateInfraProject = generateInfraProject;
/**
 * Returns a generated async infrastructure project for the given language
 */
const generateAsyncInfraProject = (language, options) => {
    return (0, exports._generateInfraProject)(language, {
        [languages_1.Language.JAVA]: generated_java_async_cdk_infrastructure_project_1.GeneratedJavaAsyncCdkInfrastructureProject,
        [languages_1.Language.PYTHON]: generated_python_async_cdk_infrastructure_project_1.GeneratedPythonAsyncCdkInfrastructureProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_async_cdk_infrastructure_project_1.GeneratedTypescriptAsyncCdkInfrastructureProject,
    }, options);
};
exports.generateAsyncInfraProject = generateAsyncInfraProject;
const _generateInfraProject = (language, projectConstructors, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Infrastructure",
            "",
            "This directory contains a generated type-safe CDK construct which can provision the API gateway infrastructure for an API based on your model.",
        ],
        readonly: true,
    });
    return generateLanguageProject("infra", language, projectConstructors, {
        ...options,
        typescriptOptions: {
            ...options.typescriptOptions,
            generatedTypescriptTypes: options.generatedRuntimes.typescript,
        },
        pythonOptions: {
            ...options.pythonOptions,
            generatedPythonTypes: options.generatedRuntimes.python,
        },
        javaOptions: {
            ...options.javaOptions,
            generatedJavaTypes: options.generatedRuntimes.java,
        },
    });
};
exports._generateInfraProject = _generateInfraProject;
/**
 * Create handlers projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateHandlersProjects = (languages, options) => {
    return _generateHandlersProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_handlers_project_1.GeneratedJavaHandlersProject,
        [languages_1.Language.PYTHON]: generated_python_handlers_project_1.GeneratedPythonHandlersProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_handlers_project_1.GeneratedTypescriptHandlersProject,
    }, options);
};
exports.generateHandlersProjects = generateHandlersProjects;
/**
 * Create async handlers projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateAsyncHandlersProjects = (languages, options) => {
    return _generateHandlersProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_async_handlers_project_1.GeneratedJavaAsyncHandlersProject,
        [languages_1.Language.PYTHON]: generated_python_async_handlers_project_1.GeneratedPythonAsyncHandlersProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_async_handlers_project_1.GeneratedTypescriptAsyncHandlersProject,
    }, options);
};
exports.generateAsyncHandlersProjects = generateAsyncHandlersProjects;
const _generateHandlersProjects = (languages, projectConstructors, options) => {
    if (languages.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Handlers",
                "",
                "This directory contains lambda handlers for implementing your API.",
                "",
                "Whenever an operation is annotated with the `@handler` trait in Smithy (or the `x-handler` vendor extension in OpenAPI), a stub handler implementation will be generated for you, which you are free to modify.",
            ],
            readonly: true,
        });
    }
    return generateLanguageProjects("handlers", languages, projectConstructors, {
        ...options,
        typescriptOptions: {
            ...options.typescriptOptions,
            generatedTypescriptTypes: options.generatedRuntimes.typescript,
        },
        pythonOptions: {
            ...options.pythonOptions,
            generatedPythonTypes: options.generatedRuntimes.python,
        },
        javaOptions: {
            ...options.javaOptions,
            generatedJavaTypes: options.generatedRuntimes.java,
        },
    });
};
/**
 * Create runtime projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateRuntimeProjects = (languages, options) => {
    return _generateRuntimeProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_runtime_project_1.GeneratedJavaRuntimeProject,
        [languages_1.Language.PYTHON]: generated_python_runtime_project_1.GeneratedPythonRuntimeProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_runtime_project_1.GeneratedTypescriptRuntimeProject,
    }, options);
};
exports.generateRuntimeProjects = generateRuntimeProjects;
/**
 * Create async runtime projects in the given languages
 * @param languages the languages to generate for
 * @param options options for the projects to be created
 */
const generateAsyncRuntimeProjects = (languages, options) => {
    return _generateRuntimeProjects(languages, {
        [languages_1.Language.JAVA]: generated_java_async_runtime_project_1.GeneratedJavaAsyncRuntimeProject,
        [languages_1.Language.PYTHON]: generated_python_async_runtime_project_1.GeneratedPythonAsyncRuntimeProject,
        [languages_1.Language.TYPESCRIPT]: generated_typescript_async_runtime_project_1.GeneratedTypescriptAsyncRuntimeProject,
    }, options);
};
exports.generateAsyncRuntimeProjects = generateAsyncRuntimeProjects;
const _generateRuntimeProjects = (languages, projectConstructors, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Runtimes",
            "",
            "This directory contains generated runtime projects based on your API model.",
            "",
            "Each runtime project includes types from your API model, as well as type-safe client and server code.",
        ],
        readonly: true,
    });
    return generateLanguageProjects("runtime", languages, projectConstructors, options);
};
/**
 * Returns a generated client project for the given language
 */
const generateLibraryProject = (library, options) => {
    const packageName = `${options.parentPackageName}-${library}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, library),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (library) {
        case languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS: {
            return new typescript_react_query_hooks_library_1.TypescriptReactQueryHooksLibrary({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptReactQueryHooksOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        default:
            throw new Error(`Unknown library ${library}`);
    }
};
/**
 * Create library projects
 * @param libraries the libraries to generate for
 * @param options options for the projects to be created
 */
const generateLibraryProjects = (libraries, options) => {
    if (libraries.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Libraries",
                "",
                "This directory contains generated libraries based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedLibraries = {};
    libraries.forEach((library) => {
        const project = generateLibraryProject(library, options);
        if (project != null) {
            generatedLibraries[library] = project;
        }
    });
    return generatedLibraries;
};
exports.generateLibraryProjects = generateLibraryProjects;
/**
 * Returns a generated client project for the given language
 */
const generateAsyncLibraryProject = (library, options) => {
    const packageName = `${options.parentPackageName}-${library}`;
    const commonOptions = {
        outdir: path.join(options.generatedCodeDir, library),
        specPath: options.parsedSpecPath,
        parent: options.parent,
    };
    switch (library) {
        case languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT: {
            return new typescript_websocket_client_library_1.TypescriptWebsocketClientLibrary({
                ...commonOptions,
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptWebSocketClientOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        case languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS: {
            return new typescript_websocket_hooks_library_1.TypescriptWebsocketHooksLibrary({
                ...commonOptions,
                clientPackageName: options.typescriptWebSocketHooksOptions.clientPackageName ??
                    sanitiseTypescriptPackageName(`${options.parentPackageName}-${languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT}`),
                name: sanitiseTypescriptPackageName(packageName),
                ...options.typescriptWebSocketClientOptions,
                isWithinMonorepo: options.isWithinMonorepo,
            });
        }
        default:
            throw new Error(`Unknown library ${library}`);
    }
};
/**
 * Create library projects
 * @param libraries the libraries to generate for
 * @param options options for the projects to be created
 */
const generateAsyncLibraryProjects = (libraries, options) => {
    if (libraries.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Libraries",
                "",
                "This directory contains generated libraries based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedLibraries = {};
    libraries.forEach((library) => {
        const project = generateAsyncLibraryProject(library, options);
        if (project != null) {
            generatedLibraries[library] = project;
        }
    });
    return generatedLibraries;
};
exports.generateAsyncLibraryProjects = generateAsyncLibraryProjects;
const generateDocsProject = (format, options) => {
    const commonProps = {
        name: `${options.parentPackageName}-documentation-${format.replace(/_/g, "-")}`,
        parent: options.parent,
        outdir: path.join(options.generatedDocsDir, format),
        specPath: options.parsedSpecPath,
    };
    switch (format) {
        case languages_1.DocumentationFormat.HTML2: {
            return new generated_html2_documentation_project_1.GeneratedHtml2DocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.html2,
            });
        }
        case languages_1.DocumentationFormat.HTML_REDOC: {
            return new generated_html_redoc_documentation_project_1.GeneratedHtmlRedocDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.htmlRedoc,
            });
        }
        case languages_1.DocumentationFormat.MARKDOWN: {
            return new generated_markdown_documentation_project_1.GeneratedMarkdownDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.markdown,
            });
        }
        case languages_1.DocumentationFormat.PLANTUML: {
            return new generated_plantuml_documentation_project_1.GeneratedPlantumlDocumentationProject({
                ...commonProps,
                ...options.documentationOptions?.plantuml,
            });
        }
        case languages_1.WebSocketDocumentationFormat.HTML: {
            return new generated_asyncapi_html_documentation_project_1.GeneratedAsyncApiHtmlDocumentationProject({
                ...commonProps,
                ...options.asyncDocumentationOptions?.html,
            });
        }
        case languages_1.WebSocketDocumentationFormat.MARKDOWN: {
            return new generated_asyncapi_markdown_documentation_project_1.GeneratedAsyncApiMarkdownDocumentationProject({
                ...commonProps,
                ...options.asyncDocumentationOptions?.markdown,
            });
        }
        default:
            throw new Error(`Unknown documentation format ${format}`);
    }
};
const generateDocsProjects = (formats, options) => {
    if (formats.length > 0) {
        new projen_1.TextFile(options.parent, path.join(options.generatedDocsDir, "README.md"), {
            lines: [
                "## Generated Documentation",
                "",
                "This directory contains generated documentation based on your API model.",
            ],
            readonly: true,
        });
    }
    const generatedClients = {};
    formats.forEach((format) => {
        const project = generateDocsProject(format, options);
        if (project != null) {
            generatedClients[format] = project;
        }
    });
    return generatedClients;
};
exports.generateDocsProjects = generateDocsProjects;
//# sourceMappingURL=data:application/json;base64,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