"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonHandlersBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const languages_1 = require("../../languages");
const types_1 = require("../../types");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedPythonHandlersBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            pytest: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
            ...options,
            sample: false,
            poetry: true,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.PythonVersion.PYTHON_3_11;
        this.tstDir = "test";
        if (options.pytest ?? true) {
            // Pytest fails with exit code 5 when there are no tests.
            // We want to allow users to delete all their tests, or to upgrade an existing project without breaking their build
            // See: https://github.com/pytest-dev/pytest/issues/2393
            this.testTask.reset(`pytest ${this.tstDir}/ || ([ $? = 5 ] && exit 0 || exit $?)`);
        }
        [
            runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPythonDependencyVersion(this.runtimeVersion),
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.moduleName}/__all_handlers.py`, `!${this.tstDir}/__all_tests.py`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(".openapi-generator");
        // Write __init__.py as sample code
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        // Package task to build a distributable which can be deployed to lambda
        this.packageTask.exec("mkdir -p dist/lambda && rm -rf dist/lambda/*");
        this.packageTask.exec(`cp -r ${this.moduleName} dist/lambda/${this.moduleName}`);
        this.packageTask.exec("poetry export --without-hashes --format=requirements.txt > dist/lambda/requirements.txt");
        // Select the platform based on the specified architecture, defaulting to x86_64
        // See: https://docs.aws.amazon.com/lambda/latest/dg/python-package.html#python-package-native-libraries
        const platform = options.architecture === types_1.Architecture.ARM_64
            ? "manylinux2014_aarch64"
            : "manylinux2014_x86_64";
        this.packageTask.exec(`pip install -r dist/lambda/requirements.txt --target dist/lambda --upgrade --platform ${platform} --only-binary :all: --python-version ${runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPipPackagingPythonVersion(this.runtimeVersion)}`);
    }
}
exports.GeneratedPythonHandlersBaseProject = GeneratedPythonHandlersBaseProject;
//# sourceMappingURL=data:application/json;base64,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