"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptHandlersBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const languages_1 = require("../../languages");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedTypescriptHandlersBaseProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.NodeVersion.NODE_18;
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the runtime can be
            // resolved
            options.isWithinMonorepo
                ? `${options.generatedTypescriptTypes.package.packageName}@${options.generatedTypescriptTypes.package.manifest.version}`
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        this.addDevDeps("esbuild", "@types/aws-lambda");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Ignore everything for the regular open api generator pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.srcdir}/__all_handlers.ts`, `!${this.testdir}/__all_tests.ts`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the openapi generator metadata
        this.gitignore.addPatterns(".openapi-generator");
        // Create a separate lambda bundle for each handler as part of the package task.
        // Note that every typescript file directly in src is bundled by default, but users may specify their own
        // entry point globs if they prefer a different directory structure.
        this.packageTask.exec(`mkdir -p dist/lambda && rm -rf dist/lambda/*`);
        this.packageTask.exec(`esbuild --bundle ${(options.handlerEntryPoints ?? [`${this.srcdir}/*.ts`]).join(" ")} --platform=node --outdir=dist/lambda --target=${runtime_version_utils_1.RuntimeVersionUtils.NODE.getEsbuildNodeTarget(this.runtimeVersion)}`);
        // Move each bundled file into a separate directory
        this.packageTask.exec("for f in $(ls dist/lambda); do mkdir dist/lambda/$(basename $f .js) && mv dist/lambda/$f dist/lambda/$(basename $f .js)/index.js; done");
        // Create an empty index.ts sample on synth so that tsc is happy if the handlers project is configured
        // but no operations have @handler(language: "typescript")
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                "index.ts": "",
            },
        });
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptHandlersBaseProject = GeneratedTypescriptHandlersBaseProject;
//# sourceMappingURL=data:application/json;base64,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