"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaCdkInfrastructureBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../../../../monorepo");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedJavaCdkInfrastructureBaseProject extends java_1.JavaProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            junit: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.packageName = `${this.pom.groupId}.${this.name}.infra`;
        this.srcDir = path.join("src", "main", "java", ...this.packageName.split("."));
        [
            `software.aws/pdk@${monorepo_1.ProjectUtils.getPdkVersion()}`,
            "software.constructs/constructs@10.3.0",
            "software.amazon.awscdk/aws-cdk-lib@2.133.0",
            "io.github.cdklabs/cdknag@2.28.60",
            "org.projectlombok/lombok@1.18.30",
            "com.fasterxml.jackson.core/jackson-databind@2.17.0",
            `io.github.cdklabs/projen@0.80.10`,
            `${options.generatedJavaTypes.pom.groupId}/${options.generatedJavaTypes.pom.artifactId}@${options.generatedJavaTypes.pom.version}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Pin constructs version
        this.deps.removeDependency("software.constructs/constructs", projen_1.DependencyType.BUILD);
        this.addDependency("software.constructs/constructs@10.3.0");
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Add a dependency on the generated java types repository
        this.pom.addRepository({
            url: `file://${path.relative(this.outdir, options.generatedJavaTypes.outdir)}/dist/java`,
            id: `${options.generatedJavaTypes.pom.groupId}-${options.generatedJavaTypes.pom.artifactId}-repo`,
        });
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore = openapiGeneratorIgnore;
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.srcDir}/Api.java`, `!${this.srcDir}/ApiProps.java`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.srcDir}/__functions.java`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        this.generateTask = generateTask;
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        // Copy the parsed spec into the resources directory so that it's included in the jar
        generateTask.exec("mkdir -p src/main/resources");
        generateTask.exec(`cp -f ${this.options.specPath} src/main/resources/.api.json`);
        // Absolute path of this project is required for determining the path to the handlers jar,
        // since java executes from the jar which could be anywhere in the filesystem (eg the .m2 directory).
        // While ugly, since this is written at build time and not checked in it remains portable.
        generateTask.exec("echo $(pwd) > src/main/resources/project-absolute-path.txt");
        this.preCompileTask.spawn(generateTask);
    }
}
exports.GeneratedJavaCdkInfrastructureBaseProject = GeneratedJavaCdkInfrastructureBaseProject;
//# sourceMappingURL=data:application/json;base64,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