"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptCdkInfrastructureBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedTypescriptCdkInfrastructureBaseProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            jest: false,
            eslint: false,
            prettier: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        /**
         * Path to the packaged copy of the openapi specification
         */
        this.packagedSpecPath = "assets/api.json";
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.addDevDeps("@types/aws-lambda");
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the types can be
            // resolved
            options.isWithinMonorepo
                ? `${options.generatedTypescriptTypes.package.packageName}@${options.generatedTypescriptTypes.package.manifest.version}`
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        const devAndPeerDeps = ["@aws/pdk", "constructs", "aws-cdk-lib", "cdk-nag"];
        this.addDevDeps(...devAndPeerDeps.filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.BUILD)));
        this.addPeerDeps(...devAndPeerDeps.filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.PEER)));
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore = openapiGeneratorIgnore;
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.srcdir}/index.ts`, `!${this.srcdir}/api.ts`, `!${this.srcdir}/mock-integrations.ts`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.srcdir}/__functions.ts`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        this.generateTask = generateTask;
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        // Copy the api spec to within the package
        generateTask.exec(`mkdir -p ${path.dirname(this.packagedSpecPath)}`);
        generateTask.exec(`cp -f ${this.options.specPath} ${this.packagedSpecPath}`);
        this.gitignore.addPatterns(`/${this.packagedSpecPath}`);
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(this.srcdir, ".openapi-generator", "mocks");
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptCdkInfrastructureBaseProject = GeneratedTypescriptCdkInfrastructureBaseProject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2VuZXJhdGVkLXR5cGVzY3JpcHQtY2RrLWluZnJhc3RydWN0dXJlLWJhc2UtcHJvamVjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImdlbmVyYXRlZC10eXBlc2NyaXB0LWNkay1pbmZyYXN0cnVjdHVyZS1iYXNlLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7c0NBQ3NDO0FBQ3RDLDZCQUE2QjtBQUM3QixtQ0FBMEQ7QUFDMUQsc0RBQTJEO0FBQzNELHNEQUcrQjtBQUsvQiwwSEFBa0g7QUFDbEgsb0dBQTZGO0FBQzdGLHdGQUFpRjtBQUNqRiwwR0FBbUc7QUFDbkcsa0RBTWdDO0FBMkJoQyxNQUFzQiwrQ0FBZ0QsU0FBUSw4QkFBaUI7SUFjN0YsWUFBWSxPQUErRDtRQUN6RSxLQUFLLENBQUM7WUFDSixHQUFJLE9BQWU7WUFDbkIsVUFBVSxFQUFFLEtBQUs7WUFDakIsSUFBSSxFQUFFLEtBQUs7WUFDWCxNQUFNLEVBQUUsS0FBSztZQUNiLFFBQVEsRUFBRSxLQUFLO1lBQ2YsUUFBUSxFQUFFO2dCQUNSLGVBQWUsRUFBRTtvQkFDZixHQUFHLEVBQUUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDO29CQUN0QixvRUFBb0U7b0JBQ3BFLGNBQWMsRUFBRSxLQUFLO29CQUNyQixrQkFBa0IsRUFBRSxLQUFLO29CQUN6QixZQUFZLEVBQUUsSUFBSTtvQkFDbEIsR0FBRyxPQUFPLEVBQUUsUUFBUSxFQUFFLGVBQWU7aUJBQ3RDO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRSxLQUFLO1NBQ3hCLENBQUMsQ0FBQztRQTFCTDs7V0FFRztRQUNnQixxQkFBZ0IsR0FBRyxpQkFBaUIsQ0FBQztRQXFJakQsNkJBQXdCLEdBQUcsR0FBRyxFQUFFO1lBQ3JDLE9BQU8sSUFBQSw4Q0FBc0MsRUFDM0MsSUFBSSxDQUFDLDRCQUE0QixFQUFFLENBQ3BDLENBQUM7UUFDSixDQUFDLENBQUM7UUFqSEEsaUVBQTZCLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1FBRXZCLElBQUksQ0FBQyxVQUFVLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsT0FBTyxDQUNWLEdBQUc7WUFDRCw4R0FBOEc7WUFDOUcsV0FBVztZQUNYLE9BQU8sQ0FBQyxnQkFBZ0I7Z0JBQ3RCLENBQUMsQ0FBQyxHQUFHLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPLENBQUMsV0FBVyxJQUFJLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtnQkFDeEgsQ0FBQyxDQUFDLEdBQ0UsT0FBTyxDQUFDLHdCQUF3QixDQUFDLE9BQU8sQ0FBQyxXQUMzQyxTQUFTLElBQUksQ0FBQyxRQUFRLENBQ3BCLElBQUksQ0FBQyxNQUFNLEVBQ1gsT0FBTyxDQUFDLHdCQUF3QixDQUFDLE1BQU0sQ0FDeEMsRUFBRTtTQUNSLENBQUMsTUFBTSxDQUNOLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLHVCQUFjLENBQUMsT0FBTyxDQUFDLENBQ2xFLENBQ0YsQ0FBQztRQUVGLE1BQU0sY0FBYyxHQUFHLENBQUMsVUFBVSxFQUFFLFlBQVksRUFBRSxhQUFhLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDNUUsSUFBSSxDQUFDLFVBQVUsQ0FDYixHQUFHLGNBQWMsQ0FBQyxNQUFNLENBQ3RCLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLHVCQUFjLENBQUMsS0FBSyxDQUFDLENBQ2hFLENBQ0YsQ0FBQztRQUNGLElBQUksQ0FBQyxXQUFXLENBQ2QsR0FBRyxjQUFjLENBQUMsTUFBTSxDQUN0QixDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsRUFBRSx1QkFBYyxDQUFDLElBQUksQ0FBQyxDQUMvRCxDQUNGLENBQUM7UUFFRiwwREFBMEQ7UUFDMUQsTUFBTSxTQUFTLEdBQUcsSUFBSSxtQkFBVSxDQUFDLElBQUksRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNyRCxTQUFTLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFcEQseUNBQXlDO1FBQ3pDLE1BQU0sc0JBQXNCLEdBQUcsSUFBSSwyREFBMEIsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsc0JBQXNCLEdBQUcsc0JBQXNCLENBQUM7UUFDckQsc0JBQXNCLENBQUMsV0FBVyxDQUNoQyxJQUFJLEVBQ0osTUFBTSxFQUNOLEdBQUcsRUFDSCxJQUFJLElBQUksQ0FBQyxNQUFNLFdBQVcsRUFDMUIsSUFBSSxJQUFJLENBQUMsTUFBTSxTQUFTLEVBQ3hCLElBQUksSUFBSSxDQUFDLE1BQU0sdUJBQXVCLENBQ3ZDLENBQUM7UUFFRixNQUFNLGdDQUFnQyxHQUNwQyxJQUFJLGdGQUFvQyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pELGdDQUFnQyxDQUFDLFdBQVcsQ0FDMUMsSUFBSSxFQUNKLE1BQU0sRUFDTixHQUFHLEVBQ0gsSUFBSSxJQUFJLENBQUMsTUFBTSxpQkFBaUIsQ0FDakMsQ0FBQztRQUVGLDBDQUEwQztRQUMxQywrQ0FBb0IsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsNEJBQTRCLENBQzVELE9BQU8sQ0FBQyx5QkFBeUIsQ0FDbEMsQ0FBQztRQUVGLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFZLENBQUM7UUFDakMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFBLDZDQUFxQyxHQUFFLENBQUMsQ0FBQztRQUMzRCxZQUFZLENBQUMsSUFBSSxDQUNmLElBQUEsbUNBQTJCLEVBQ3pCLHlCQUFpQixDQUFDLFFBQVEsRUFDMUIsSUFBSSxDQUFDLHdCQUF3QixFQUFFLENBQ2hDLENBQ0YsQ0FBQztRQUVGLDBDQUEwQztRQUMxQyxZQUFZLENBQUMsSUFBSSxDQUFDLFlBQVksSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDckUsWUFBWSxDQUFDLElBQUksQ0FDZixTQUFTLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUMxRCxDQUFDO1FBQ0YsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBRXhELElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRXhDLDRCQUE0QjtRQUM1QixJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLG9CQUFvQixFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXZFLGlIQUFpSDtRQUNqSCxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixFQUFFO1lBQzdCLFFBQVEsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUU7Z0JBQ25DLEtBQUssK0JBQWtCLENBQUMsSUFBSTtvQkFDMUIsSUFBSSxDQUFDLEtBQUs7eUJBQ1AsT0FBTyxDQUFDLFNBQVMsQ0FBQzt3QkFDbkIsRUFBRSxXQUFXLENBQ1gsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsVUFBVSxJQUFJLENBQUMsUUFBUSxDQUNuRCxJQUFJLENBQUMsTUFBTSxFQUNYLElBQUksQ0FBQyxPQUFPLENBQUMsd0JBQXdCLENBQUMsTUFBTSxDQUM3QyxFQUFFLENBQ0osQ0FBQztvQkFDSixNQUFNO2dCQUNSO29CQUNFLElBQUksQ0FBQyxLQUFLO3lCQUNQLE9BQU8sQ0FBQyxTQUFTLENBQUM7d0JBQ25CLEVBQUUsV0FBVyxDQUNYLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLFNBQVMsSUFBSSxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLENBQ25HLENBQUM7b0JBQ0osTUFBTTthQUNUO1NBQ0Y7SUFDSCxDQUFDO0NBU0Y7QUFySkQsMEdBcUpDIiwic291cmNlc0NvbnRlbnQiOlsiLyohIENvcHlyaWdodCBbQW1hem9uLmNvbV0oaHR0cDovL2FtYXpvbi5jb20vKSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wICovXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBEZXBlbmRlbmN5VHlwZSwgSWdub3JlRmlsZSwgVGFzayB9IGZyb20gXCJwcm9qZW5cIjtcbmltcG9ydCB7IE5vZGVQYWNrYWdlTWFuYWdlciB9IGZyb20gXCJwcm9qZW4vbGliL2phdmFzY3JpcHRcIjtcbmltcG9ydCB7XG4gIFR5cGVTY3JpcHRQcm9qZWN0LFxuICBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMsXG59IGZyb20gXCJwcm9qZW4vbGliL3R5cGVzY3JpcHRcIjtcbmltcG9ydCB7XG4gIENvZGVHZW5lcmF0aW9uU291cmNlT3B0aW9ucyxcbiAgR2VuZXJhdGVkV2l0aE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zLFxufSBmcm9tIFwiLi4vLi4vLi4vdHlwZXNcIjtcbmltcG9ydCB7IE9wZW5BcGlHZW5lcmF0b3JIYW5kbGViYXJzSWdub3JlRmlsZSB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL29wZW4tYXBpLWdlbmVyYXRvci1oYW5kbGViYXJzLWlnbm9yZS1maWxlXCI7XG5pbXBvcnQgeyBPcGVuQXBpR2VuZXJhdG9ySWdub3JlRmlsZSB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL29wZW4tYXBpLWdlbmVyYXRvci1pZ25vcmUtZmlsZVwiO1xuaW1wb3J0IHsgT3BlbkFwaVRvb2xzSnNvbkZpbGUgfSBmcm9tIFwiLi4vLi4vY29tcG9uZW50cy9vcGVuLWFwaS10b29scy1qc29uLWZpbGVcIjtcbmltcG9ydCB7IFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50IH0gZnJvbSBcIi4uLy4uL2NvbXBvbmVudHMvdHlwZS1zYWZlLWFwaS1jb21tYW5kLWVudmlyb25tZW50XCI7XG5pbXBvcnQge1xuICBidWlsZENsZWFuT3BlbkFwaUdlbmVyYXRlZENvZGVDb21tYW5kLFxuICBidWlsZEludm9rZU9wZW5BcGlHZW5lcmF0b3JDb21tYW5kQXJncyxcbiAgYnVpbGRUeXBlU2FmZUFwaUV4ZWNDb21tYW5kLFxuICBHZW5lcmF0aW9uT3B0aW9ucyxcbiAgVHlwZVNhZmVBcGlTY3JpcHQsXG59IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL3V0aWxzXCI7XG5pbXBvcnQgeyBHZW5lcmF0ZWRIYW5kbGVyc1Byb2plY3RzIH0gZnJvbSBcIi4uLy4uL2dlbmVyYXRlXCI7XG5pbXBvcnQgeyBHZW5lcmF0ZWRUeXBlc2NyaXB0UnVudGltZUJhc2VQcm9qZWN0IH0gZnJvbSBcIi4uLy4uL3J1bnRpbWUvZ2VuZXJhdGVkLXR5cGVzY3JpcHQtcnVudGltZS1iYXNlLXByb2plY3RcIjtcblxuZXhwb3J0IGludGVyZmFjZSBHZW5lcmF0ZWRUeXBlc2NyaXB0SW5mcmFzdHJ1Y3R1cmVCYXNlT3B0aW9uc1xuICBleHRlbmRzIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyxcbiAgICBHZW5lcmF0ZWRXaXRoT3BlbkFwaUdlbmVyYXRvck9wdGlvbnMge31cblxuZXhwb3J0IGludGVyZmFjZSBHZW5lcmF0ZWRUeXBlc2NyaXB0Q2RrSW5mcmFzdHJ1Y3R1cmVCYXNlUHJvamVjdE9wdGlvbnNcbiAgZXh0ZW5kcyBHZW5lcmF0ZWRUeXBlc2NyaXB0SW5mcmFzdHJ1Y3R1cmVCYXNlT3B0aW9ucyxcbiAgICBDb2RlR2VuZXJhdGlvblNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogR2VuZXJhdGVkIHR5cGVzY3JpcHQgdHlwZXMgcHJvamVjdFxuICAgKi9cbiAgcmVhZG9ubHkgZ2VuZXJhdGVkVHlwZXNjcmlwdFR5cGVzOiBHZW5lcmF0ZWRUeXBlc2NyaXB0UnVudGltZUJhc2VQcm9qZWN0O1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZWQgaGFuZGxlcnMgcHJvamVjdHNcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRlZEhhbmRsZXJzOiBHZW5lcmF0ZWRIYW5kbGVyc1Byb2plY3RzO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBpbmZyYXN0cnVjdHVyZSBhbmQgY2xpZW50IHByb2plY3RzIGFyZSBwYXJlbnRlZCBieSBhbiBtb25vcmVwbyBvciBub3RcbiAgICovXG4gIHJlYWRvbmx5IGlzV2l0aGluTW9ub3JlcG8/OiBib29sZWFuO1xufVxuXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgR2VuZXJhdGVkVHlwZXNjcmlwdENka0luZnJhc3RydWN0dXJlQmFzZVByb2plY3QgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdCB7XG4gIC8qKlxuICAgKiBPcHRpb25zIGNvbmZpZ3VyZWQgZm9yIHRoZSBwcm9qZWN0XG4gICAqL1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgb3B0aW9uczogR2VuZXJhdGVkVHlwZXNjcmlwdENka0luZnJhc3RydWN0dXJlQmFzZVByb2plY3RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBQYXRoIHRvIHRoZSBwYWNrYWdlZCBjb3B5IG9mIHRoZSBvcGVuYXBpIHNwZWNpZmljYXRpb25cbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBwYWNrYWdlZFNwZWNQYXRoID0gXCJhc3NldHMvYXBpLmpzb25cIjtcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgZ2VuZXJhdGVUYXNrOiBUYXNrO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgb3BlbmFwaUdlbmVyYXRvcklnbm9yZTogT3BlbkFwaUdlbmVyYXRvcklnbm9yZUZpbGU7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogR2VuZXJhdGVkVHlwZXNjcmlwdENka0luZnJhc3RydWN0dXJlQmFzZVByb2plY3RPcHRpb25zKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4uKG9wdGlvbnMgYXMgYW55KSxcbiAgICAgIHNhbXBsZUNvZGU6IGZhbHNlLFxuICAgICAgamVzdDogZmFsc2UsXG4gICAgICBlc2xpbnQ6IGZhbHNlLFxuICAgICAgcHJldHRpZXI6IGZhbHNlLFxuICAgICAgdHNjb25maWc6IHtcbiAgICAgICAgY29tcGlsZXJPcHRpb25zOiB7XG4gICAgICAgICAgbGliOiBbXCJkb21cIiwgXCJlczIwMTlcIl0sXG4gICAgICAgICAgLy8gR2VuZXJhdGVkIGNvZGUgaW1wb3J0cyBhbGwgbW9kZWxzLCBhbmQgbWF5IG5vdCByZWZlcmVuY2UgdGhlbSBhbGxcbiAgICAgICAgICBub1VudXNlZExvY2FsczogZmFsc2UsXG4gICAgICAgICAgbm9VbnVzZWRQYXJhbWV0ZXJzOiBmYWxzZSxcbiAgICAgICAgICBza2lwTGliQ2hlY2s6IHRydWUsXG4gICAgICAgICAgLi4ub3B0aW9ucz8udHNjb25maWc/LmNvbXBpbGVyT3B0aW9ucyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBucG1pZ25vcmVFbmFibGVkOiBmYWxzZSxcbiAgICB9KTtcbiAgICBUeXBlU2FmZUFwaUNvbW1hbmRFbnZpcm9ubWVudC5lbnN1cmUodGhpcyk7XG4gICAgdGhpcy5vcHRpb25zID0gb3B0aW9ucztcblxuICAgIHRoaXMuYWRkRGV2RGVwcyhcIkB0eXBlcy9hd3MtbGFtYmRhXCIpO1xuICAgIHRoaXMuYWRkRGVwcyhcbiAgICAgIC4uLltcbiAgICAgICAgLy8gSWYgd2l0aGluIGEgbW9ub3JlcG8sIGFkZCBhIHJlZ3VsYXIgZGVwZW5kZW5jeS4gT3RoZXJ3aXNlLCB1c2UgYSBmaWxlIGRlcGVuZGVuY3kgdG8gZW5zdXJlIHRoZSB0eXBlcyBjYW4gYmVcbiAgICAgICAgLy8gcmVzb2x2ZWRcbiAgICAgICAgb3B0aW9ucy5pc1dpdGhpbk1vbm9yZXBvXG4gICAgICAgICAgPyBgJHtvcHRpb25zLmdlbmVyYXRlZFR5cGVzY3JpcHRUeXBlcy5wYWNrYWdlLnBhY2thZ2VOYW1lfUAke29wdGlvbnMuZ2VuZXJhdGVkVHlwZXNjcmlwdFR5cGVzLnBhY2thZ2UubWFuaWZlc3QudmVyc2lvbn1gXG4gICAgICAgICAgOiBgJHtcbiAgICAgICAgICAgICAgb3B0aW9ucy5nZW5lcmF0ZWRUeXBlc2NyaXB0VHlwZXMucGFja2FnZS5wYWNrYWdlTmFtZVxuICAgICAgICAgICAgfUBmaWxlOiR7cGF0aC5yZWxhdGl2ZShcbiAgICAgICAgICAgICAgdGhpcy5vdXRkaXIsXG4gICAgICAgICAgICAgIG9wdGlvbnMuZ2VuZXJhdGVkVHlwZXNjcmlwdFR5cGVzLm91dGRpclxuICAgICAgICAgICAgKX1gLFxuICAgICAgXS5maWx0ZXIoXG4gICAgICAgIChkZXApID0+ICF0aGlzLmRlcHMudHJ5R2V0RGVwZW5kZW5jeShkZXAsIERlcGVuZGVuY3lUeXBlLlJVTlRJTUUpXG4gICAgICApXG4gICAgKTtcblxuICAgIGNvbnN0IGRldkFuZFBlZXJEZXBzID0gW1wiQGF3cy9wZGtcIiwgXCJjb25zdHJ1Y3RzXCIsIFwiYXdzLWNkay1saWJcIiwgXCJjZGstbmFnXCJdO1xuICAgIHRoaXMuYWRkRGV2RGVwcyhcbiAgICAgIC4uLmRldkFuZFBlZXJEZXBzLmZpbHRlcihcbiAgICAgICAgKGRlcCkgPT4gIXRoaXMuZGVwcy50cnlHZXREZXBlbmRlbmN5KGRlcCwgRGVwZW5kZW5jeVR5cGUuQlVJTEQpXG4gICAgICApXG4gICAgKTtcbiAgICB0aGlzLmFkZFBlZXJEZXBzKFxuICAgICAgLi4uZGV2QW5kUGVlckRlcHMuZmlsdGVyKFxuICAgICAgICAoZGVwKSA9PiAhdGhpcy5kZXBzLnRyeUdldERlcGVuZGVuY3koZGVwLCBEZXBlbmRlbmN5VHlwZS5QRUVSKVxuICAgICAgKVxuICAgICk7XG5cbiAgICAvLyBNaW5pbWFsIC5ucG1pZ25vcmUgdG8gYXZvaWQgaW1wYWN0aW5nIE9wZW5BUEkgR2VuZXJhdG9yXG4gICAgY29uc3QgbnBtaWdub3JlID0gbmV3IElnbm9yZUZpbGUodGhpcywgXCIubnBtaWdub3JlXCIpO1xuICAgIG5wbWlnbm9yZS5hZGRQYXR0ZXJucyhcIi8ucHJvamVuL1wiLCBcIi9zcmNcIiwgXCIvZGlzdFwiKTtcblxuICAgIC8vIElnbm9yZSBldmVyeXRoaW5nIGJ1dCB0aGUgdGFyZ2V0IGZpbGVzXG4gICAgY29uc3Qgb3BlbmFwaUdlbmVyYXRvcklnbm9yZSA9IG5ldyBPcGVuQXBpR2VuZXJhdG9ySWdub3JlRmlsZSh0aGlzKTtcbiAgICB0aGlzLm9wZW5hcGlHZW5lcmF0b3JJZ25vcmUgPSBvcGVuYXBpR2VuZXJhdG9ySWdub3JlO1xuICAgIG9wZW5hcGlHZW5lcmF0b3JJZ25vcmUuYWRkUGF0dGVybnMoXG4gICAgICBcIi8qXCIsXG4gICAgICBcIioqLypcIixcbiAgICAgIFwiKlwiLFxuICAgICAgYCEke3RoaXMuc3JjZGlyfS9pbmRleC50c2AsXG4gICAgICBgISR7dGhpcy5zcmNkaXJ9L2FwaS50c2AsXG4gICAgICBgISR7dGhpcy5zcmNkaXJ9L21vY2staW50ZWdyYXRpb25zLnRzYFxuICAgICk7XG5cbiAgICBjb25zdCBvcGVuYXBpR2VuZXJhdG9ySGFuZGxlYmFyc0lnbm9yZSA9XG4gICAgICBuZXcgT3BlbkFwaUdlbmVyYXRvckhhbmRsZWJhcnNJZ25vcmVGaWxlKHRoaXMpO1xuICAgIG9wZW5hcGlHZW5lcmF0b3JIYW5kbGViYXJzSWdub3JlLmFkZFBhdHRlcm5zKFxuICAgICAgXCIvKlwiLFxuICAgICAgXCIqKi8qXCIsXG4gICAgICBcIipcIixcbiAgICAgIGAhJHt0aGlzLnNyY2Rpcn0vX19mdW5jdGlvbnMudHNgXG4gICAgKTtcblxuICAgIC8vIEFkZCBPcGVuQVBJIEdlbmVyYXRvciBjbGkgY29uZmlndXJhdGlvblxuICAgIE9wZW5BcGlUb29sc0pzb25GaWxlLmVuc3VyZSh0aGlzKS5hZGRPcGVuQXBpR2VuZXJhdG9yQ2xpQ29uZmlnKFxuICAgICAgb3B0aW9ucy5vcGVuQXBpR2VuZXJhdG9yQ2xpQ29uZmlnXG4gICAgKTtcblxuICAgIGNvbnN0IGdlbmVyYXRlVGFzayA9IHRoaXMuYWRkVGFzayhcImdlbmVyYXRlXCIpO1xuICAgIHRoaXMuZ2VuZXJhdGVUYXNrID0gZ2VuZXJhdGVUYXNrO1xuICAgIGdlbmVyYXRlVGFzay5leGVjKGJ1aWxkQ2xlYW5PcGVuQXBpR2VuZXJhdGVkQ29kZUNvbW1hbmQoKSk7XG4gICAgZ2VuZXJhdGVUYXNrLmV4ZWMoXG4gICAgICBidWlsZFR5cGVTYWZlQXBpRXhlY0NvbW1hbmQoXG4gICAgICAgIFR5cGVTYWZlQXBpU2NyaXB0LkdFTkVSQVRFLFxuICAgICAgICB0aGlzLmJ1aWxkR2VuZXJhdGVDb21tYW5kQXJncygpXG4gICAgICApXG4gICAgKTtcblxuICAgIC8vIENvcHkgdGhlIGFwaSBzcGVjIHRvIHdpdGhpbiB0aGUgcGFja2FnZVxuICAgIGdlbmVyYXRlVGFzay5leGVjKGBta2RpciAtcCAke3BhdGguZGlybmFtZSh0aGlzLnBhY2thZ2VkU3BlY1BhdGgpfWApO1xuICAgIGdlbmVyYXRlVGFzay5leGVjKFxuICAgICAgYGNwIC1mICR7dGhpcy5vcHRpb25zLnNwZWNQYXRofSAke3RoaXMucGFja2FnZWRTcGVjUGF0aH1gXG4gICAgKTtcbiAgICB0aGlzLmdpdGlnbm9yZS5hZGRQYXR0ZXJucyhgLyR7dGhpcy5wYWNrYWdlZFNwZWNQYXRofWApO1xuXG4gICAgdGhpcy5wcmVDb21waWxlVGFzay5zcGF3bihnZW5lcmF0ZVRhc2spO1xuXG4gICAgLy8gSWdub3JlIHRoZSBnZW5lcmF0ZWQgY29kZVxuICAgIHRoaXMuZ2l0aWdub3JlLmFkZFBhdHRlcm5zKHRoaXMuc3JjZGlyLCBcIi5vcGVuYXBpLWdlbmVyYXRvclwiLCBcIm1vY2tzXCIpO1xuXG4gICAgLy8gSWYgd2UncmUgbm90IGluIGEgbW9ub3JlcG8sIHdlIG5lZWQgdG8gbGluayB0aGUgZ2VuZXJhdGVkIHR5cGVzIHN1Y2ggdGhhdCB0aGUgbG9jYWwgZGVwZW5kZW5jeSBjYW4gYmUgcmVzb2x2ZWRcbiAgICBpZiAoIW9wdGlvbnMuaXNXaXRoaW5Nb25vcmVwbykge1xuICAgICAgc3dpdGNoICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIpIHtcbiAgICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTTpcbiAgICAgICAgICB0aGlzLnRhc2tzXG4gICAgICAgICAgICAudHJ5RmluZChcImluc3RhbGxcIilcbiAgICAgICAgICAgID8ucHJlcGVuZEV4ZWMoXG4gICAgICAgICAgICAgIGAke3RoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcn0gbGluayAvJHtwYXRoLnJlbGF0aXZlKFxuICAgICAgICAgICAgICAgIHRoaXMub3V0ZGlyLFxuICAgICAgICAgICAgICAgIHRoaXMub3B0aW9ucy5nZW5lcmF0ZWRUeXBlc2NyaXB0VHlwZXMub3V0ZGlyXG4gICAgICAgICAgICAgICl9YFxuICAgICAgICAgICAgKTtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICB0aGlzLnRhc2tzXG4gICAgICAgICAgICAudHJ5RmluZChcImluc3RhbGxcIilcbiAgICAgICAgICAgID8ucHJlcGVuZEV4ZWMoXG4gICAgICAgICAgICAgIGAke3RoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcn0gbGluayAke3RoaXMub3B0aW9ucy5nZW5lcmF0ZWRUeXBlc2NyaXB0VHlwZXMucGFja2FnZS5wYWNrYWdlTmFtZX1gXG4gICAgICAgICAgICApO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBidWlsZEdlbmVyYXRlQ29tbWFuZEFyZ3MgPSAoKSA9PiB7XG4gICAgcmV0dXJuIGJ1aWxkSW52b2tlT3BlbkFwaUdlbmVyYXRvckNvbW1hbmRBcmdzKFxuICAgICAgdGhpcy5idWlsZE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zKClcbiAgICApO1xuICB9O1xuXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBidWlsZE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zKCk6IEdlbmVyYXRpb25PcHRpb25zO1xufVxuIl19