"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptLibraryProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Typescript generated library project
 */
class GeneratedTypescriptLibraryProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("strict-peer-dependencies", "false");
        }
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        this.openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore.addPatterns(...GeneratedTypescriptLibraryProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns("src", ".npmignore", "README.md", ".openapi-generator");
        // If we're not in a monorepo, we need to link the generated client such that any local dependency on it can be
        // resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    // Nothing to do for pnpm, since the pnpm link command handles both the dependant and dependee
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.exec(monorepo_1.NodePackageUtils.command.cmd(this.package.packageManager, "link"));
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptLibraryProject = GeneratedTypescriptLibraryProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedTypescriptLibraryProject.openApiIgnorePatterns = [
    "package.json",
    "tsconfig.json",
    "tsconfig.esm.json",
    ".npmignore",
];
//# sourceMappingURL=data:application/json;base64,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