"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonRuntimeBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const python_1 = require("projen/lib/python");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Python project containing types generated using OpenAPI Generator CLI
 */
class GeneratedPythonRuntimeBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Add dependencies required by the client
        [
            "python-dateutil@~2.8.2",
            "pydantic@^2.5.2",
            "aenum@^3.1.11",
            "urllib3@~1.26.7",
            `aws-lambda-powertools@{extras=["tracer", "aws-sdk"],version="^2.28.0"}`,
            "python@^3.9",
        ].forEach((dep) => this.addDependency(dep));
        this.openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore.addPatterns(...GeneratedPythonRuntimeBaseProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "7.1.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns(this.moduleName, "docs", "README.md", ".openapi-generator");
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py as our first install step to keep poetry happy until the generator overwrites
        // it.
        ["install", "install:ci"].forEach((t) => this.tasks
            .tryFind(t)
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py`));
    }
}
exports.GeneratedPythonRuntimeBaseProject = GeneratedPythonRuntimeBaseProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedPythonRuntimeBaseProject.openApiIgnorePatterns = [
    "test",
    "test/*",
    "test/**/*",
    ".github",
    ".github/*",
    ".github/**/*",
    ".gitlab-ci.yml",
    ".travis.yml",
    "git_push.sh",
    "tox.ini",
    "requirements.txt",
    "test-requirements.txt",
    "setup.py",
    "setup.cfg",
    "pyproject.toml",
];
//# sourceMappingURL=data:application/json;base64,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