"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptRuntimeBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Typescript project containing types generated using OpenAPI Generator CLI
 */
class GeneratedTypescriptRuntimeBaseProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("strict-peer-dependencies", "false");
        }
        // For event and context types
        this.addDeps("@types/aws-lambda", "@aws-lambda-powertools/tracer", "@aws-lambda-powertools/logger", "@aws-lambda-powertools/metrics");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", `/${this.srcdir}`, "/dist");
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore = openapiGeneratorIgnore;
        openapiGeneratorIgnore.addPatterns(...GeneratedTypescriptRuntimeBaseProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore all the generated code
        this.gitignore.addPatterns(this.srcdir, ".npmignore", "README.md", ".openapi-generator");
        // If we're not in a monorepo, we need to link the generated client such that any local dependency on it can be
        // resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    // Nothing to do for pnpm, since the pnpm link command handles both the dependant and dependee
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.exec(monorepo_1.NodePackageUtils.command.cmd(this.package.packageManager, "link"));
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptRuntimeBaseProject = GeneratedTypescriptRuntimeBaseProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedTypescriptRuntimeBaseProject.openApiIgnorePatterns = [
    "package.json",
    "tsconfig.json",
    "tsconfig.esm.json",
    ".npmignore",
];
//# sourceMappingURL=data:application/json;base64,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