"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyProjectDefinition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const smithy_aws_pdk_prelude_1 = require("./components/smithy-aws-pdk-prelude");
const smithy_build_gradle_file_1 = require("./components/smithy-build-gradle-file");
const smithy_settings_gradle_file_1 = require("./components/smithy-settings-gradle-file");
const version_1 = require("./version");
const utils_1 = require("../../codegen/components/utils");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyProjectDefinition extends projen_1.Component {
    constructor(project, options) {
        super(project);
        const { smithyOptions } = options;
        // Ignore gradle wrapper by default
        if (smithyOptions.ignoreGradleWrapper ?? true) {
            project.gitignore.addPatterns("gradle");
            project.gitignore.addPatterns("gradlew");
            project.gitignore.addPatterns("gradlew.bat");
        }
        // Always ignore the .gradle dir which the wrapper downloads gradle into
        project.gitignore.addPatterns(".gradle");
        this.gradleProjectName = project.name.replace(/[\/\\:<>"?\*|]/g, "-");
        // Add settings.gradle
        new smithy_settings_gradle_file_1.SmithySettingsGradleFile(project, {
            gradleProjectName: this.gradleProjectName,
        });
        const modelDir = "src/main/smithy";
        this.modelDir = modelDir;
        // Always add the following required dependencies
        const requiredDependencies = [
            "software.amazon.smithy:smithy-cli",
            "software.amazon.smithy:smithy-model",
            "software.amazon.smithy:smithy-openapi",
            "software.amazon.smithy:smithy-aws-traits",
        ];
        // Ensure dependencies always include the required dependencies, allowing users to customise the version
        const userSpecifiedDependencies = smithyOptions.smithyBuildOptions?.maven?.dependencies ?? [];
        const userSpecifiedDependencySet = new Set(userSpecifiedDependencies.map((dep) => dep.split(":").slice(0, -1).join(":")));
        const dependencies = [
            ...requiredDependencies
                .filter((requiredDep) => !userSpecifiedDependencySet.has(requiredDep))
                .map((dep) => `${dep}:${version_1.DEFAULT_SMITHY_VERSION}`),
            ...userSpecifiedDependencies,
        ];
        // Add build.gradle
        this.smithyBuildGradleFile = new smithy_build_gradle_file_1.SmithyBuildGradleFile(project, {
            modelDir,
            dependencies,
            repositoryUrls: smithyOptions.smithyBuildOptions?.maven?.repositoryUrls,
        });
        const { namespace: serviceNamespace, serviceName } = smithyOptions.serviceName;
        // Create the smithy build json file
        this.smithyBuild = new smithy_build_1.SmithyBuild(project, {
            version: "2.0",
            ...smithyOptions.smithyBuildOptions,
            sources: [
                modelDir,
                ...this.asRelativePathsToProject(smithyOptions.smithyBuildOptions?.additionalSources ?? []),
            ],
            projections: {
                ...smithyOptions.smithyBuildOptions?.projections,
                openapi: {
                    ...smithyOptions.smithyBuildOptions?.projections?.openapi,
                    plugins: {
                        openapi: {
                            service: `${serviceNamespace}#${serviceName}`,
                            // By default, preserve tags in the generated spec, but allow users to explicitly overwrite this
                            tags: true,
                            // By default, use integer types as this is more intuitive when smithy distinguishes between Integers and Doubles.
                            // Users may also override this.
                            useIntegerType: true,
                            ...smithyOptions.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
            maven: {
                // Filter out any file dependencies since these aren't supported in smithy-build.json
                dependencies: dependencies.filter((dep) => !dep.startsWith(smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix)),
                repositories: (smithyOptions.smithyBuildOptions?.maven?.repositoryUrls ?? [
                    "https://repo.maven.apache.org/maven2/",
                    "file://~/.m2/repository",
                ]).map((url) => ({ url })),
            },
        });
        // Add the smithy prelude (managed by aws-pdk)
        this.generatedModelDir = path.join("generated", "main", "smithy");
        new smithy_aws_pdk_prelude_1.SmithyAwsPdkPrelude(project, {
            generatedModelDir: this.generatedModelDir,
            serviceNamespace,
            handlerLanguages: options.handlerLanguages,
        });
        this.addSources(this.generatedModelDir);
        const projectionOutputPath = path.join("build", "smithyprojections", this.gradleProjectName, "openapi");
        this.openApiSpecificationPath = path.join(projectionOutputPath, "openapi", `${serviceName}.openapi.json`);
        this.smithyJsonModelPath = path.join(projectionOutputPath, "model", "model.json");
        // Copy the gradle files during build if they don't exist. We don't overwrite to allow users to BYO gradle wrapper
        // and set `ignoreGradleWrapper: false`.
        project.generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.COPY_GRADLE_WRAPPER));
        // Build with gradle to generate smithy projections, and any other tasks
        project.generateTask.exec("./gradlew build");
        if (smithyOptions.ignoreSmithyBuildOutput ?? true) {
            // Ignore the build directory, and smithy-output which was the old build directory for the cli-based generation
            project.gitignore.addPatterns("build", "smithy-output");
        }
    }
    /**
     * Add maven-style or local file dependencies to the smithy model project
     * @param deps dependencies to add, eg "software.amazon.smithy:smithy-validation-model:1.27.2" or "file://../some/path/build/lib/my-shapes.jar
     */
    addDeps(...deps) {
        this.smithyBuildGradleFile.addDeps(...deps);
        this.smithyBuild.addMavenDependencies(...deps.filter((dep) => !dep.startsWith(smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix)));
    }
    /**
     * Add dependencies on other smithy models, such that their shapes can be imported in this project
     * @param deps smithy definitions to depend on
     */
    addSmithyDeps(...deps) {
        this.addDeps(...deps.map((dep) => `${smithy_build_gradle_file_1.SmithyBuildGradleFile.fileDependencyPrefix}${path.join(path.relative(this.project.outdir, dep.project.outdir), "build", "libs", `${dep.gradleProjectName}.jar`)}`));
    }
    /**
     * Add additional paths to model source files or directories.
     * Paths should be relative to the project outdir. Any absolute paths will be
     * resolved as relative paths.
     */
    addSources(...sources) {
        const relativeSources = this.asRelativePathsToProject(sources);
        this.smithyBuild.addSources(...relativeSources);
        this.smithyBuildGradleFile.addSources(...relativeSources);
    }
    /**
     * Convert any given absolute paths to relative paths to the project outdir
     * @private
     */
    asRelativePathsToProject(paths) {
        return paths.map((p) => path.isAbsolute(p) ? path.relative(this.project.outdir, p) : p);
    }
}
exports.SmithyProjectDefinition = SmithyProjectDefinition;
_a = JSII_RTTI_SYMBOL_1;
SmithyProjectDefinition[_a] = { fqn: "@aws/pdk.type_safe_api.SmithyProjectDefinition", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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