"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeApiProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const type_safe_api_model_project_1 = require("./model/type-safe-api-model-project");
/**
 * Project for a type-safe API, defined using Smithy or OpenAPI.
 *
 * Generates a CDK construct to deploy your API, as well as client and server code to help build your API quickly.
 *
 * @pjid type-safe-api
 */
class TypeSafeApiProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.getNxWorkspace = (options) => {
            return options.parent ? monorepo_1.NxWorkspace.of(options.parent) : undefined;
        };
        const nxWorkspace = this.getNxWorkspace(options);
        const isNxWorkspace = this.parent &&
            (monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoTsProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoJavaProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoPythonProject));
        const handlerLanguages = [...new Set(options.handlers?.languages ?? [])];
        // API Definition project containing the model
        const modelDir = "model";
        this.model = new type_safe_api_model_project_1.TypeSafeApiModelProject({
            parent: nxWorkspace ? this.parent : this,
            outdir: nxWorkspace ? path.join(options.outdir, modelDir) : modelDir,
            name: `${options.name}-model`,
            modelLanguage: options.model.language,
            modelOptions: options.model.options,
            handlerLanguages,
        });
        // Ensure we always generate a runtime project for the infrastructure language, regardless of what was specified by
        // the user. Likewise we generate a runtime project for any handler languages specified
        const runtimeLanguages = [
            ...new Set([
                ...(options.runtime?.languages ?? []),
                options.infrastructure.language,
                ...(options.handlers?.languages ?? []),
            ]),
        ];
        const generatedDir = "generated";
        const runtimeDir = path.join(generatedDir, "runtime");
        const runtimeDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, runtimeDir)
            : runtimeDir;
        // Path from a generated package directory (eg api/generated/runtime/typescript) to the model dir (ie api/model)
        const relativePathToModelDirFromGeneratedPackageDir = path.relative(path.join(this.outdir, runtimeDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToGeneratedPackageDir = path.join(relativePathToModelDirFromGeneratedPackageDir, this.model.parsedSpecFile);
        // Declare the generated runtime projects
        const generatedRuntimeProjects = (0, generate_1.generateRuntimeProjects)(runtimeLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: runtimeDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated runtime dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected?.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.runtime?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.runtime?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.runtime?.options?.java,
            },
        });
        const documentationFormats = [
            ...new Set(options.documentation?.formats ?? []),
        ];
        const docsDir = path.join(generatedDir, "documentation");
        const docsDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, docsDir)
            : docsDir;
        const generatedDocs = (0, generate_1.generateDocsProjects)(documentationFormats, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedDocsDir: docsDirRelativeToParent,
            // Spec path relative to each generated doc format dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            documentationOptions: options.documentation?.options,
        });
        this.documentation = {
            htmlRedoc: generatedDocs[languages_1.DocumentationFormat.HTML_REDOC],
            html2: generatedDocs[languages_1.DocumentationFormat.HTML2],
            markdown: generatedDocs[languages_1.DocumentationFormat.MARKDOWN],
            plantuml: generatedDocs[languages_1.DocumentationFormat.PLANTUML],
        };
        const libraries = [...new Set(options.library?.libraries ?? [])];
        const libraryDir = path.join(generatedDir, "libraries");
        const libraryDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, libraryDir)
            : libraryDir;
        // Declare the generated runtime projects
        const generatedLibraryProjects = (0, generate_1.generateLibraryProjects)(libraries, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: libraryDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated library dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptReactQueryHooksOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.library?.options?.typescriptReactQueryHooks,
            },
        });
        // Ensure the generated runtime, libraries and docs projects have a dependency on the model project
        if (this.parent) {
            [
                ...Object.values(generatedRuntimeProjects),
                ...Object.values(generatedDocs),
                ...Object.values(generatedLibraryProjects),
            ].forEach((project) => {
                monorepo_1.NxProject.ensure(project).addImplicitDependency(this.model);
            });
        }
        this.runtime = {
            typescript: generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                ? generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedRuntimeProjects[languages_1.Language.JAVA]
                ? generatedRuntimeProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedRuntimeProjects[languages_1.Language.PYTHON]
                ? generatedRuntimeProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        this.library = {
            typescriptReactQueryHooks: generatedLibraryProjects[languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS]
                ? generatedLibraryProjects[languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS]
                : undefined,
        };
        const handlersDir = "handlers";
        const handlersDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, handlersDir)
            : handlersDir;
        const relativePathToModelDirFromHandlersDir = path.relative(path.join(this.outdir, handlersDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToHandlersDir = path.join(relativePathToModelDirFromHandlersDir, this.model.parsedSpecFile);
        // Declare the generated handlers projects
        const generatedHandlersProjects = (0, generate_1.generateHandlersProjects)(handlerLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: handlersDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated handlers package dir
            parsedSpecPath: parsedSpecRelativeToHandlersDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.handlers?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.handlers?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.handlers?.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
        });
        this.handlers = {
            typescript: generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                ? generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedHandlersProjects[languages_1.Language.JAVA]
                ? generatedHandlersProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedHandlersProjects[languages_1.Language.PYTHON]
                ? generatedHandlersProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        // Ensure the handlers project depends on the appropriate runtime projects
        if (this.handlers.typescript) {
            monorepo_1.NxProject.ensure(this.handlers.typescript).addImplicitDependency(this.runtime.typescript);
        }
        if (this.handlers.java) {
            monorepo_1.NxProject.ensure(this.handlers.java).addImplicitDependency(this.runtime.java);
        }
        if (this.handlers.python) {
            monorepo_1.NxProject.ensure(this.handlers.python).addImplicitDependency(this.runtime.python);
        }
        const infraDir = path.join(generatedDir, "infrastructure");
        const infraDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, infraDir)
            : infraDir;
        // Infrastructure project
        const infraProject = (0, generate_1.generateInfraProject)(options.infrastructure.language, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: infraDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated infra package dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.infrastructure.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.infrastructure.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.infrastructure.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
            generatedHandlers: {
                typescript: this.handlers.typescript,
                python: this.handlers.python,
                java: this.handlers.java,
            },
        });
        const infraProjects = {};
        // Add implicit dependencies and assign the appropriate infrastructure project member
        switch (options.infrastructure.language) {
            case languages_1.Language.JAVA:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.java);
                infraProjects.java = infraProject;
                break;
            case languages_1.Language.PYTHON:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.python);
                infraProjects.python = infraProject;
                break;
            case languages_1.Language.TYPESCRIPT:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.typescript);
                infraProjects.typescript = infraProject;
                break;
            default:
                throw new Error(`Unknown infrastructure language ${options.infrastructure.language}`);
        }
        this.infrastructure = infraProjects;
        monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.model);
        // Expose collections of projects
        const allRuntimes = Object.values(generatedRuntimeProjects);
        const allInfrastructure = [infraProject];
        const allLibraries = Object.values(generatedLibraryProjects);
        const allDocumentation = Object.values(generatedDocs);
        const allHandlers = Object.values(generatedHandlersProjects);
        this.all = {
            model: [this.model],
            runtimes: allRuntimes,
            infrastructure: allInfrastructure,
            libraries: allLibraries,
            documentation: allDocumentation,
            handlers: allHandlers,
            projects: [
                this.model,
                ...allRuntimes,
                ...allInfrastructure,
                ...allLibraries,
                ...allDocumentation,
                ...allHandlers,
            ],
        };
        if (!nxWorkspace) {
            // Add a task for the non-monorepo case to build the projects in the right order
            [
                this.model,
                ...Object.values(generatedRuntimeProjects),
                infraProject,
                ...Object.values(generatedLibraryProjects),
                ...Object.values(generatedDocs),
            ].forEach((project) => {
                this.compileTask.exec("npx projen build", {
                    cwd: path.relative(this.outdir, project.outdir),
                });
            });
        }
        // Add the README as a sample file which the user may edit
        new projen_1.SampleFile(this, "README.md", {
            sourcePath: path.resolve(__dirname, "..", "..", "samples", "type-safe-api", "readme", "TYPE_SAFE_API.md"),
        });
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // Migration code, since we've moved these generated directories under a parent "generated"
        // folder, we delete the generated projects which would otherwise be orphaned and still
        // checked into VCS
        ["runtime", "libraries", "infrastructure", "documentation"].forEach((dir) => fs.rmSync(path.join(this.outdir, dir), { force: true, recursive: true }));
        super.postSynthesize();
    }
}
exports.TypeSafeApiProject = TypeSafeApiProject;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeApiProject[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeApiProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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