"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeWebSocketApiProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const type_safe_websocket_api_model_project_1 = require("./model/type-safe-websocket-api-model-project");
/**
 * Project for a Type Safe WebSocket API, defined using Smithy or OpenAPI.
 *
 * Generates a CDK construct to deploy your API, as well as client and server code to help build your API quickly.
 *
 * @experimental
 * @pjid type-safe-ws-api
 */
class TypeSafeWebSocketApiProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.getNxWorkspace = (options) => {
            return options.parent ? monorepo_1.NxWorkspace.of(options.parent) : undefined;
        };
        const nxWorkspace = this.getNxWorkspace(options);
        const isNxWorkspace = this.parent &&
            (monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoTsProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoJavaProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoPythonProject));
        const handlerLanguages = [...new Set(options.handlers?.languages ?? [])];
        // API Definition project containing the model
        const modelDir = "model";
        this.model = new type_safe_websocket_api_model_project_1.TypeSafeWebSocketApiModelProject({
            parent: nxWorkspace ? this.parent : this,
            outdir: nxWorkspace ? path.join(options.outdir, modelDir) : modelDir,
            name: `${options.name}-model`,
            modelLanguage: options.model.language,
            modelOptions: options.model.options,
            handlerLanguages,
        });
        // Ensure we always generate a runtime project for the infrastructure language, regardless of what was specified by
        // the user. Likewise we generate a runtime project for any handler languages specified
        const runtimeLanguages = [
            ...new Set([
                ...(options.runtime?.languages ?? []),
                options.infrastructure.language,
                ...(options.handlers?.languages ?? []),
            ]),
        ];
        // TODO: Delete when python/java support is implemented
        if (runtimeLanguages.includes(languages_1.Language.JAVA) ||
            runtimeLanguages.includes(languages_1.Language.PYTHON)) {
            throw new Error(`Python and Java are not yet supported by Type Safe WebSocket API! Support is coming soon...`);
        }
        const generatedDir = "generated";
        const runtimeDir = path.join(generatedDir, "runtime");
        const runtimeDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, runtimeDir)
            : runtimeDir;
        // Path from a generated package directory (eg api/generated/runtime/typescript) to the model dir (ie api/model)
        const relativePathToModelDirFromGeneratedPackageDir = path.relative(path.join(this.outdir, runtimeDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToGeneratedPackageDir = path.join(relativePathToModelDirFromGeneratedPackageDir, this.model.parsedSpecFile);
        // Declare the generated runtime projects
        const generatedRuntimeProjects = (0, generate_1.generateAsyncRuntimeProjects)(runtimeLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: runtimeDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated runtime dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected?.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.runtime?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.runtime?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.runtime?.options?.java,
            },
        });
        const documentationFormats = [
            ...new Set(options.documentation?.formats ?? []),
        ];
        const docsDir = path.join(generatedDir, "documentation");
        const docsDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, docsDir)
            : docsDir;
        // AsyncAPI specification is used for WebSocket documentation generation
        const asyncapiJsonFilePathRelativeToGeneratedPackageDir = path.join(relativePathToModelDirFromGeneratedPackageDir, this.model.asyncApiSpecFile);
        const generatedDocs = (0, generate_1.generateDocsProjects)(documentationFormats, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedDocsDir: docsDirRelativeToParent,
            // Spec path relative to each generated doc format dir
            parsedSpecPath: asyncapiJsonFilePathRelativeToGeneratedPackageDir,
            asyncDocumentationOptions: options.documentation?.options,
        });
        // Documentation projects use AsyncAPI generator which can intermittently fail
        // when executed in parallel to other AsyncAPI generator commands. We protect against this
        // by ensuring documentation projects are built sequentially.
        const docProjects = Object.values(generatedDocs);
        docProjects.forEach((docProject, i) => {
            if (docProjects[i - 1]) {
                monorepo_1.NxProject.ensure(docProjects[i - 1]).addImplicitDependency(docProject);
            }
        });
        this.documentation = {
            html: generatedDocs[languages_1.WebSocketDocumentationFormat.HTML],
            markdown: generatedDocs[languages_1.WebSocketDocumentationFormat.MARKDOWN],
        };
        const librarySet = new Set(options.library?.libraries ?? []);
        // Hooks depend on client, so always add the client if we specified hooks
        if (librarySet.has(languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS)) {
            librarySet.add(languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT);
        }
        const libraries = [...librarySet];
        const libraryDir = path.join(generatedDir, "libraries");
        const libraryDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, libraryDir)
            : libraryDir;
        // Declare the generated runtime projects
        const generatedLibraryProjects = (0, generate_1.generateAsyncLibraryProjects)(libraries, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: libraryDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated library dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptWebSocketClientOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.library?.options?.typescriptWebSocketClient,
            },
            typescriptWebSocketHooksOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                clientPackageName: options.library?.options?.typescriptWebSocketClient?.name,
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.library?.options?.typescriptWebSocketHooks,
            },
        });
        // Ensure the generated runtime, libraries and docs projects have a dependency on the model project
        if (this.parent) {
            [
                ...Object.values(generatedRuntimeProjects),
                ...Object.values(generatedDocs),
                ...Object.values(generatedLibraryProjects),
            ].forEach((project) => {
                monorepo_1.NxProject.ensure(project).addImplicitDependency(this.model);
            });
        }
        this.runtime = {
            typescript: generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                ? generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedRuntimeProjects[languages_1.Language.JAVA]
                ? generatedRuntimeProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedRuntimeProjects[languages_1.Language.PYTHON]
                ? generatedRuntimeProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        this.library = {
            typescriptWebSocketClient: generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT]
                ? generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT]
                : undefined,
            typescriptWebSocketHooks: generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS]
                ? generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS]
                : undefined,
        };
        // For the hooks library, add a dependency on the client
        if (this.library.typescriptWebSocketHooks &&
            this.library.typescriptWebSocketClient) {
            this.library.typescriptWebSocketHooks.addDeps(this.library.typescriptWebSocketClient.package.packageName);
        }
        const handlersDir = "handlers";
        const handlersDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, handlersDir)
            : handlersDir;
        const relativePathToModelDirFromHandlersDir = path.relative(path.join(this.outdir, handlersDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToHandlersDir = path.join(relativePathToModelDirFromHandlersDir, this.model.parsedSpecFile);
        // Declare the generated handlers projects
        const generatedHandlersProjects = (0, generate_1.generateAsyncHandlersProjects)(handlerLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: handlersDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated handlers package dir
            parsedSpecPath: parsedSpecRelativeToHandlersDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.handlers?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.handlers?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.handlers?.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
        });
        this.handlers = {
            typescript: generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                ? generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedHandlersProjects[languages_1.Language.JAVA]
                ? generatedHandlersProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedHandlersProjects[languages_1.Language.PYTHON]
                ? generatedHandlersProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        // Ensure the handlers project depends on the appropriate runtime projects
        if (this.handlers.typescript) {
            monorepo_1.NxProject.ensure(this.handlers.typescript).addImplicitDependency(this.runtime.typescript);
        }
        if (this.handlers.java) {
            monorepo_1.NxProject.ensure(this.handlers.java).addImplicitDependency(this.runtime.java);
        }
        if (this.handlers.python) {
            monorepo_1.NxProject.ensure(this.handlers.python).addImplicitDependency(this.runtime.python);
        }
        const infraDir = path.join(generatedDir, "infrastructure");
        const infraDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, infraDir)
            : infraDir;
        // Infrastructure project
        const infraProject = (0, generate_1.generateAsyncInfraProject)(options.infrastructure.language, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: infraDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated infra package dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.infrastructure.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.infrastructure.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.infrastructure.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
            generatedHandlers: {
                typescript: this.handlers.typescript,
                python: this.handlers.python,
                java: this.handlers.java,
            },
        });
        const infraProjects = {};
        // Add implicit dependencies and assign the appropriate infrastructure project member
        switch (options.infrastructure.language) {
            case languages_1.Language.JAVA:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.java);
                infraProjects.java = infraProject;
                break;
            case languages_1.Language.PYTHON:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.python);
                infraProjects.python = infraProject;
                break;
            case languages_1.Language.TYPESCRIPT:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.typescript);
                infraProjects.typescript = infraProject;
                break;
            default:
                throw new Error(`Unknown infrastructure language ${options.infrastructure.language}`);
        }
        this.infrastructure = infraProjects;
        monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.model);
        // Expose collections of projects
        const allRuntimes = Object.values(generatedRuntimeProjects);
        const allInfrastructure = [infraProject];
        const allLibraries = Object.values(generatedLibraryProjects);
        const allDocumentation = Object.values(generatedDocs);
        const allHandlers = Object.values(generatedHandlersProjects);
        this.all = {
            model: [this.model],
            runtimes: allRuntimes,
            infrastructure: allInfrastructure,
            libraries: allLibraries,
            documentation: allDocumentation,
            handlers: allHandlers,
            projects: [
                this.model,
                ...allRuntimes,
                ...allInfrastructure,
                ...allLibraries,
                ...allDocumentation,
                ...allHandlers,
            ],
        };
        if (!nxWorkspace) {
            // Add a task for the non-monorepo case to build the projects in the right order
            [
                this.model,
                ...Object.values(generatedRuntimeProjects),
                infraProject,
                ...Object.values(generatedLibraryProjects),
                ...Object.values(generatedDocs),
            ].forEach((project) => {
                this.compileTask.exec("npx projen build", {
                    cwd: path.relative(this.outdir, project.outdir),
                });
            });
        }
        // Add the README as a sample file which the user may edit
        new projen_1.SampleFile(this, "README.md", {
            sourcePath: path.resolve(__dirname, "..", "..", "samples", "type-safe-api", "readme", "TYPE_SAFE_API.md"),
        });
    }
}
exports.TypeSafeWebSocketApiProject = TypeSafeWebSocketApiProject;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeWebSocketApiProject[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeWebSocketApiProject", version: "0.23.39" };
//# sourceMappingURL=data:application/json;base64,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