"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.resetSecret = void 0;
/* eslint-disable no-process-env,no-console */
const AWS = require("aws-sdk");
const AWSXRay = require("aws-xray-sdk-core");
const http = require("http");
const https = require("https");
AWSXRay.captureHTTPsGlobal(http);
AWSXRay.captureHTTPsGlobal(https);
AWSXRay.capturePromise();
const axios_1 = require("axios");
const secretsManagerClient = AWSXRay.captureAWSClient(new AWS.SecretsManager());
const ssmClient = AWSXRay.captureAWSClient(new AWS.SSM());
/**
 * @returns {Promise<string>} the secret key value stored in SSM Parameter Store
 * @param {string} secretKeyParameter The name of the secret key parameter to fetch
 */
const getSsmParameterSecret = async (secretKeyParameter) => {
    const parameterResponse = await ssmClient.getParameter({
        Name: secretKeyParameter,
        WithDecryption: true
    }).promise();
    if (!parameterResponse.Parameter?.Value) {
        throw new Error('SSM parameter response missing parameter!');
    }
    return parameterResponse.Parameter.Value;
};
/**
 * @returns {Promise<string>} the secret key value stored in Secrets Manager
 * @param {string} secretKeySecretArn The ARN of the secret key to fetch
 * @param {string | undefined} field An optional field to parse from a JSON secret
 */
const getSecretsManagerSecret = async (secretKeySecretArn, field) => {
    const secretResponse = await secretsManagerClient.getSecretValue({
        SecretId: secretKeySecretArn
    }).promise();
    if (!secretResponse.SecretString) {
        throw new Error('Secrets Manager secret response missing secret string!');
    }
    return field ? JSON.parse(secretResponse.SecretString)[field] : secretResponse.SecretString;
};
// eslint-disable-next-line init-declarations
let cachedSecret;
const getSecret = async () => {
    const { SECRET_KEY_TYPE, SECRET_KEY, SECRET_KEY_PARAMETER, SECRET_KEY_SECRET_ARN, SECRET_KEY_FIELD } = process.env;
    switch (SECRET_KEY_TYPE) {
        case 'PLAIN_TEXT':
            if (!SECRET_KEY) {
                throw new Error('SECRET_KEY was not defined!');
            }
            return SECRET_KEY;
        case 'SSM_PARAMETER':
            if (!SECRET_KEY_PARAMETER) {
                throw new Error('SECRET_KEY_PARAMETER was not defined!');
            }
            if (!cachedSecret) {
                // eslint-disable-next-line require-atomic-updates
                cachedSecret = await getSsmParameterSecret(SECRET_KEY_PARAMETER);
            }
            return cachedSecret;
        case 'SECRETS_MANAGER':
            if (!SECRET_KEY_SECRET_ARN) {
                throw new Error('SECRET_KEY_SECRET_ARN was not defined!');
            }
            if (!cachedSecret) {
                // eslint-disable-next-line require-atomic-updates
                cachedSecret = await getSecretsManagerSecret(SECRET_KEY_SECRET_ARN, SECRET_KEY_FIELD);
            }
            return cachedSecret;
        default:
            throw new Error(`Unsupported secret key type ${SECRET_KEY_TYPE}`);
    }
};
exports.resetSecret = () => {
    // eslint-disable-next-line no-undefined
    cachedSecret = undefined;
};
/**
 * @returns {APIGatewayAuthorizerResult} that allows or denies this request to the method
 * @param {'Allow' | 'Deny'} effect whether to allow or deny this request
 * @param {string} methodArn the ARN of the method for this request
 */
const generateAuthResponse = (effect, methodArn) => ({
    policyDocument: {
        Statement: [{
                Action: 'execute-api:Invoke',
                Effect: effect,
                Resource: methodArn
            }],
        Version: '2012-10-17'
    },
    principalId: 'user'
});
const DEFAULT_SCORE_THRESHOLD = 0.5;
const scoreThreshold = process.env.SCORE_THRESHOLD ? Number(process.env.SCORE_THRESHOLD) : DEFAULT_SCORE_THRESHOLD;
const allowedActions = process.env.ALLOWED_ACTIONS ? JSON.parse(process.env.ALLOWED_ACTIONS) : [];
/**
 * Lambda authorizer handler
 * @param {APIGatewayRequestAuthorizerEvent} event - API Gateway Lambda Proxy Input Format
 * @returns {APIGatewayAuthorizerResult} API Gateway Lambda Proxy Output Format
 */
// eslint-disable-next-line max-statements
exports.handler = async (event) => {
    const token = event.headers?.['x-recaptcha-token'];
    const { methodArn, requestContext: { identity: { sourceIp } } } = event;
    if (!token) {
        console.log(`DENY: IP: ${sourceIp}; ARN: ${methodArn}:`, 'X-reCAPTCHA-Token header missing');
        return generateAuthResponse('Deny', methodArn);
    }
    const response = await axios_1.default.post('https://www.google.com/recaptcha/api/siteverify', null, {
        params: {
            remoteip: sourceIp,
            response: token,
            secret: await getSecret()
        }
    });
    const { data: { success, score, action, hostname } } = response;
    if (!success) {
        console.log(`DENY: IP: ${sourceIp}; ARN: ${methodArn}; hostname: ${hostname}; score: ${score}; action: ${action}:`, `X-reCAPTCHA-Token: "${token}" not valid`);
        return generateAuthResponse('Deny', methodArn);
    }
    if (score < scoreThreshold) {
        console.log(`DENY: IP: ${sourceIp}; ARN: ${methodArn}; hostname: ${hostname}; score: ${score}; action: ${action}:`, `Score less than threshold: ${scoreThreshold}`);
        return generateAuthResponse('Deny', methodArn);
    }
    if (!allowedActions.includes(action)) {
        console.log(`DENY: IP: ${sourceIp}; ARN: ${methodArn}; hostname: ${hostname}; score: ${score}; action: ${action}:`, `Action not in allowed actions: ${allowedActions}`);
        return generateAuthResponse('Deny', methodArn);
    }
    console.log(`ALLOW: IP: ${sourceIp}; ARN: ${methodArn}; hostname: ${hostname}; score: ${score}; action: ${action}`);
    return generateAuthResponse('Allow', methodArn);
};
//# sourceMappingURL=data:application/json;base64,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