"""
Mozilla Public License Version 2.0

Copyright (c) 2022 MPI-Dortmund

1. Definitions
--------------

1.1. "Contributor"
    means each individual or legal entity that creates, contributes to
    the creation of, or owns Covered Software.

1.2. "Contributor Version"
    means the combination of the Contributions of others (if any) used
    by a Contributor and that particular Contributor's Contribution.

1.3. "Contribution"
    means Covered Software of a particular Contributor.

1.4. "Covered Software"
    means Source Code Form to which the initial Contributor has attached
    the notice in Exhibit A, the Executable Form of such Source Code
    Form, and Modifications of such Source Code Form, in each case
    including portions thereof.

1.5. "Incompatible With Secondary Licenses"
    means

    (a) that the initial Contributor has attached the notice described
        in Exhibit B to the Covered Software; or

    (b) that the Covered Software was made available under the terms of
        version 1.1 or earlier of the License, but not also under the
        terms of a Secondary License.

1.6. "Executable Form"
    means any form of the work other than Source Code Form.

1.7. "Larger Work"
    means a work that combines Covered Software with other material, in
    a separate file or files, that is not Covered Software.

1.8. "License"
    means this document.

1.9. "Licensable"
    means having the right to grant, to the maximum extent possible,
    whether at the time of the initial grant or subsequently, any and
    all of the rights conveyed by this License.

1.10. "Modifications"
    means any of the following:

    (a) any file in Source Code Form that results from an addition to,
        deletion from, or modification of the contents of Covered
        Software; or

    (b) any new file in Source Code Form that contains any Covered
        Software.

1.11. "Patent Claims" of a Contributor
    means any patent claim(s), including without limitation, method,
    process, and apparatus claims, in any patent Licensable by such
    Contributor that would be infringed, but for the grant of the
    License, by the making, using, selling, offering for sale, having
    made, import, or transfer of either its Contributions or its
    Contributor Version.

1.12. "Secondary License"
    means either the GNU General Public License, Version 2.0, the GNU
    Lesser General Public License, Version 2.1, the GNU Affero General
    Public License, Version 3.0, or any later versions of those
    licenses.

1.13. "Source Code Form"
    means the form of the work preferred for making modifications.

1.14. "You" (or "Your")
    means an individual or a legal entity exercising rights under this
    License. For legal entities, "You" includes any entity that
    controls, is controlled by, or is under common control with You. For
    purposes of this definition, "control" means (a) the power, direct
    or indirect, to cause the direction or management of such entity,
    whether by contract or otherwise, or (b) ownership of more than
    fifty percent (50%) of the outstanding shares or beneficial
    ownership of such entity.

2. License Grants and Conditions
--------------------------------

2.1. Grants

Each Contributor hereby grants You a world-wide, royalty-free,
non-exclusive license:

(a) under intellectual property rights (other than patent or trademark)
    Licensable by such Contributor to use, reproduce, make available,
    modify, display, perform, distribute, and otherwise exploit its
    Contributions, either on an unmodified basis, with Modifications, or
    as part of a Larger Work; and

(b) under Patent Claims of such Contributor to make, use, sell, offer
    for sale, have made, import, and otherwise transfer either its
    Contributions or its Contributor Version.

2.2. Effective Date

The licenses granted in Section 2.1 with respect to any Contribution
become effective for each Contribution on the date the Contributor first
distributes such Contribution.

2.3. Limitations on Grant Scope

The licenses granted in this Section 2 are the only rights granted under
this License. No additional rights or licenses will be implied from the
distribution or licensing of Covered Software under this License.
Notwithstanding Section 2.1(b) above, no patent license is granted by a
Contributor:

(a) for any code that a Contributor has removed from Covered Software;
    or

(b) for infringements caused by: (i) Your and any other third party's
    modifications of Covered Software, or (ii) the combination of its
    Contributions with other software (except as part of its Contributor
    Version); or

(c) under Patent Claims infringed by Covered Software in the absence of
    its Contributions.

This License does not grant any rights in the trademarks, service marks,
or logos of any Contributor (except as may be necessary to comply with
the notice requirements in Section 3.4).

2.4. Subsequent Licenses

No Contributor makes additional grants as a result of Your choice to
distribute the Covered Software under a subsequent version of this
License (see Section 10.2) or under the terms of a Secondary License (if
permitted under the terms of Section 3.3).

2.5. Representation

Each Contributor represents that the Contributor believes its
Contributions are its original creation(s) or it has sufficient rights
to grant the rights to its Contributions conveyed by this License.

2.6. Fair Use

This License is not intended to limit any rights You have under
applicable copyright doctrines of fair use, fair dealing, or other
equivalents.

2.7. Conditions

Sections 3.1, 3.2, 3.3, and 3.4 are conditions of the licenses granted
in Section 2.1.

3. Responsibilities
-------------------

3.1. Distribution of Source Form

All distribution of Covered Software in Source Code Form, including any
Modifications that You create or to which You contribute, must be under
the terms of this License. You must inform recipients that the Source
Code Form of the Covered Software is governed by the terms of this
License, and how they can obtain a copy of this License. You may not
attempt to alter or restrict the recipients' rights in the Source Code
Form.

3.2. Distribution of Executable Form

If You distribute Covered Software in Executable Form then:

(a) such Covered Software must also be made available in Source Code
    Form, as described in Section 3.1, and You must inform recipients of
    the Executable Form how they can obtain a copy of such Source Code
    Form by reasonable means in a timely manner, at a charge no more
    than the cost of distribution to the recipient; and

(b) You may distribute such Executable Form under the terms of this
    License, or sublicense it under different terms, provided that the
    license for the Executable Form does not attempt to limit or alter
    the recipients' rights in the Source Code Form under this License.

3.3. Distribution of a Larger Work

You may create and distribute a Larger Work under terms of Your choice,
provided that You also comply with the requirements of this License for
the Covered Software. If the Larger Work is a combination of Covered
Software with a work governed by one or more Secondary Licenses, and the
Covered Software is not Incompatible With Secondary Licenses, this
License permits You to additionally distribute such Covered Software
under the terms of such Secondary License(s), so that the recipient of
the Larger Work may, at their option, further distribute the Covered
Software under the terms of either this License or such Secondary
License(s).

3.4. Notices

You may not remove or alter the substance of any license notices
(including copyright notices, patent notices, disclaimers of warranty,
or limitations of liability) contained within the Source Code Form of
the Covered Software, except that You may alter any license notices to
the extent required to remedy known factual inaccuracies.

3.5. Application of Additional Terms

You may choose to offer, and to charge a fee for, warranty, support,
indemnity or liability obligations to one or more recipients of Covered
Software. However, You may do so only on Your own behalf, and not on
behalf of any Contributor. You must make it absolutely clear that any
such warranty, support, indemnity, or liability obligation is offered by
You alone, and You hereby agree to indemnify every Contributor for any
liability incurred by such Contributor as a result of warranty, support,
indemnity or liability terms You offer. You may include additional
disclaimers of warranty and limitations of liability specific to any
jurisdiction.

4. Inability to Comply Due to Statute or Regulation
---------------------------------------------------

If it is impossible for You to comply with any of the terms of this
License with respect to some or all of the Covered Software due to
statute, judicial order, or regulation then You must: (a) comply with
the terms of this License to the maximum extent possible; and (b)
describe the limitations and the code they affect. Such description must
be placed in a text file included with all distributions of the Covered
Software under this License. Except to the extent prohibited by statute
or regulation, such description must be sufficiently detailed for a
recipient of ordinary skill to be able to understand it.

5. Termination
--------------

5.1. The rights granted under this License will terminate automatically
if You fail to comply with any of its terms. However, if You become
compliant, then the rights granted under this License from a particular
Contributor are reinstated (a) provisionally, unless and until such
Contributor explicitly and finally terminates Your grants, and (b) on an
ongoing basis, if such Contributor fails to notify You of the
non-compliance by some reasonable means prior to 60 days after You have
come back into compliance. Moreover, Your grants from a particular
Contributor are reinstated on an ongoing basis if such Contributor
notifies You of the non-compliance by some reasonable means, this is the
first time You have received notice of non-compliance with this License
from such Contributor, and You become compliant prior to 30 days after
Your receipt of the notice.

5.2. If You initiate litigation against any entity by asserting a patent
infringement claim (excluding declaratory judgment actions,
counter-claims, and cross-claims) alleging that a Contributor Version
directly or indirectly infringes any patent, then the rights granted to
You by any and all Contributors for the Covered Software under Section
2.1 of this License shall terminate.

5.3. In the event of termination under Sections 5.1 or 5.2 above, all
end user license agreements (excluding distributors and resellers) which
have been validly granted by You or Your distributors under this License
prior to termination shall survive termination.

************************************************************************
*                                                                      *
*  6. Disclaimer of Warranty                                           *
*  -------------------------                                           *
*                                                                      *
*  Covered Software is provided under this License on an "as is"       *
*  basis, without warranty of any kind, either expressed, implied, or  *
*  statutory, including, without limitation, warranties that the       *
*  Covered Software is free of defects, merchantable, fit for a        *
*  particular purpose or non-infringing. The entire risk as to the     *
*  quality and performance of the Covered Software is with You.        *
*  Should any Covered Software prove defective in any respect, You     *
*  (not any Contributor) assume the cost of any necessary servicing,   *
*  repair, or correction. This disclaimer of warranty constitutes an   *
*  essential part of this License. No use of any Covered Software is   *
*  authorized under this License except under this disclaimer.         *
*                                                                      *
************************************************************************

************************************************************************
*                                                                      *
*  7. Limitation of Liability                                          *
*  --------------------------                                          *
*                                                                      *
*  Under no circumstances and under no legal theory, whether tort      *
*  (including negligence), contract, or otherwise, shall any           *
*  Contributor, or anyone who distributes Covered Software as          *
*  permitted above, be liable to You for any direct, indirect,         *
*  special, incidental, or consequential damages of any character      *
*  including, without limitation, damages for lost profits, loss of    *
*  goodwill, work stoppage, computer failure or malfunction, or any    *
*  and all other commercial damages or losses, even if such party      *
*  shall have been informed of the possibility of such damages. This   *
*  limitation of liability shall not apply to liability for death or   *
*  personal injury resulting from such party's negligence to the       *
*  extent applicable law prohibits such limitation. Some               *
*  jurisdictions do not allow the exclusion or limitation of           *
*  incidental or consequential damages, so this exclusion and          *
*  limitation may not apply to You.                                    *
*                                                                      *
************************************************************************

8. Litigation
-------------

Any litigation relating to this License may be brought only in the
courts of a jurisdiction where the defendant maintains its principal
place of business and such litigation shall be governed by laws of that
jurisdiction, without reference to its conflict-of-law provisions.
Nothing in this Section shall prevent a party's ability to bring
cross-claims or counter-claims.

9. Miscellaneous
----------------

This License represents the complete agreement concerning the subject
matter hereof. If any provision of this License is held to be
unenforceable, such provision shall be reformed only to the extent
necessary to make it enforceable. Any law or regulation which provides
that the language of a contract shall be construed against the drafter
shall not be used to construe this License against a Contributor.

10. Versions of the License
---------------------------

10.1. New Versions

Mozilla Foundation is the license steward. Except as provided in Section
10.3, no one other than the license steward has the right to modify or
publish new versions of this License. Each version will be given a
distinguishing version number.

10.2. Effect of New Versions

You may distribute the Covered Software under the terms of the version
of the License under which You originally received the Covered Software,
or under the terms of any subsequent version published by the license
steward.

10.3. Modified Versions

If you create software not governed by this License, and you want to
create a new license for such software, you may create and use a
modified version of this License if you rename the license and remove
any references to the name of the license steward (except to note that
such modified license differs from this License).

10.4. Distributing Source Code Form that is Incompatible With Secondary
Licenses

If You choose to distribute Source Code Form that is Incompatible With
Secondary Licenses under the terms of this version of the License, the
notice described in Exhibit B of this License must be attached.

Exhibit A - Source Code Form License Notice
-------------------------------------------

  This Source Code Form is subject to the terms of the Mozilla Public
  License, v. 2.0. If a copy of the MPL was not distributed with this
  file, You can obtain one at http://mozilla.org/MPL/2.0/.

If it is not possible or desirable to put the notice in a particular
file, then You may include the notice in a location (such as a LICENSE
file in a relevant directory) where a recipient would be likely to look
for such a notice.

You may add additional accurate notices of copyright ownership.

Exhibit B - "Incompatible With Secondary Licenses" Notice
---------------------------------------------------------

  This Source Code Form is "Incompatible With Secondary Licenses", as
  defined by the Mozilla Public License, v. 2.0.
"""

import os
import sys
import argparse
from typing import Dict
import optuna
from optuna.trial import TrialState
import optuna.samplers as samplers
from optuna.storages import RetryFailedTrialCallback
from optuna.pruners import MedianPruner
from tqdm import tqdm
import numpy as np
from pytorch_metric_learning import miners
from importlib_metadata import version
import json

from tomotwin.modules.networks.networkmanager import NetworkManager
from tomotwin.modules.common.distances import DistanceManager
from tomotwin.modules.training.torchtrainer import TorchTrainer, TripletDataset
from tomotwin.modules.training.LossPyML import LossPyML
from tomotwin.train_main import get_augmentations, generate_triplets
from tomotwin.modules.training.argparse_ui import TrainingConfiguration
from tomotwin.modules.training.mrctriplethandler import MRCTripletHandler
from tomotwin.modules.common.preprocess import label_filename




EPOCHS = 3
MAX_RETRY = 3
DISTANCE = "COSINE"
BASIC_OUTPUT = "out_train/optuna/"
PDB_PATH = "data/molmaps"
VOL_PATH = "data/validation/"
VALID_PATH = "data/validation/"
STUDY_NAME = "MYSTUDY"
PARAMS = None


class UnknownTargetError(Exception):
    pass


def generate_settings2(trial: optuna.Trial, optuna_conf: Dict) -> Dict:
    val = {}

    for suggest_func in optuna_conf:
        if suggest_func == "general":
            continue
        if suggest_func.startswith("c_"):
            for c_param in optuna_conf[suggest_func]:
                conditional_param = optuna_conf[suggest_func][c_param]
                c_var = conditional_param["condition_variable"]
                c_val = conditional_param["condition_value"]
                if c_var in val:
                    if val[c_var] == c_val:
                        val[c_param] = getattr(trial, suggest_func[2:])(
                            c_param, **conditional_param["params"]
                        )
        elif suggest_func == "constant":
            for param in optuna_conf[suggest_func]:
                sparam = optuna_conf[suggest_func][param]
                val[param] = sparam["value"]
        else:
            for param in optuna_conf[suggest_func]:
                sparam = optuna_conf[suggest_func][param]
                val[param] = getattr(trial, suggest_func)(param, **sparam)
    return val


def generate_settings(trial: optuna.Trial, optuna_conf: Dict) -> Dict:
    val = {}

    if "suggest_categorical" in optuna_conf:
        for param in optuna_conf["suggest_categorical"]:
            sparam = optuna_conf["suggest_categorical"][param]
            val[param] = trial.suggest_categorical(param, **sparam)

    if "suggest_float" in optuna_conf:
        for param in optuna_conf["suggest_float"]:
            sparam = optuna_conf["suggest_float"][param]
            val[param] = trial.suggest_float(param, **sparam)

    if "suggest_int" in optuna_conf:
        for param in optuna_conf["suggest_int"]:
            sparam = optuna_conf["suggest_int"][param]
            val[param] = trial.suggest_int(param, **sparam)

    if "constant" in optuna_conf:
        for param in optuna_conf["constant"]:
            sparam = optuna_conf["constant"][param]
            val[param] = sparam["value"]

    if "c_suggest_int" in optuna_conf:
        for c_param in optuna_conf["c_suggest_int"]:
            conditional_param = optuna_conf["c_suggest_int"][c_param]
            c_var = conditional_param["condition_variable"]
            c_val = conditional_param["condition_value"]
            if c_var in val:
                if val[c_var] == c_val:
                    trial.suggest_int(c_param, **conditional_param["param"])

    return val

def setup_network_config(settings: Dict) -> Dict:

    def add_norm_to_config():
        config["network_config"]["norm_name"] = settings["norm"]

        if settings["norm"] == "GroupNorm":
            config["network_config"]["norm_kwargs"] = {"num_groups": settings["group_size"]}
        else:
            config["network_config"]["norm_kwargs"] = {}

    if settings["network"] == "SiameseNet":
        config = {
            "identifier": settings["network"],
            "network_config": {
                "output_channels": settings["output_channels"],
                "dropout": settings["dropout"],
                "repeat_layers": settings["repeats"],
            },
        }
        add_norm_to_config()

    if settings["network"] == "ResNet":
        config = {
            "identifier": settings["network"],
            "network_config": {
                "out_head": settings["out_head"],
                "model_depth": settings["model_depth"],
            },
        }

        add_norm_to_config()

    dm = DistanceManager()
    distance = dm.get_distance(settings["distance"])
    config["distance"] = distance.name()

    return config


def objective(trial: optuna.Trial) -> float:
    output_path = os.path.join(BASIC_OUTPUT, str(trial.number))
    print("TRIAL NUMBER:", trial.number)
    ########################
    # Generate Optuna parameters
    ########################
    settings = generate_settings2(trial, PARAMS)
    settings["distance"] = DISTANCE
    print(settings)
    learning_rate = settings["learning_rate"]
    optimizer_name = settings["optimizer"]
    amsgrad = settings["amsgrad"]
    weight_decay = settings["weight_decay"]
    EPOCHS = settings["epochs"]
    BATCH_SIZE = settings["batchsize"]
    use_miner = settings["miner"]
    patience = settings["patience"]

    print("NUMBER OF EPOCHS", EPOCHS)

    ########################
    # Init distance function
    ########################
    dm = DistanceManager()
    distance = dm.get_distance(settings["distance"])
    print("Use distance function", distance.name())

    ########################
    # Setup miners and loss
    ########################
    from tomotwin import train_main as tmain

    loss_func = tmain.get_loss_func(
        net_conf=settings, train_conf=settings, distance=distance
    )

    miner = None
    if use_miner:
        miner = miners.TripletMarginMiner(
            margin=settings["miner_margin"], type_of_triplets="semihard"
        )

    ########################
    # Setup network
    ########################
    nw = NetworkManager()
    config = setup_network_config(settings=settings)
    network = nw.create_network(config)

    ############################
    # Check if restart is necessary
    ############################

    retry_trial_number = RetryFailedTrialCallback.retried_trial_number(trial)
    retry_output_path = os.path.join(BASIC_OUTPUT, str(retry_trial_number))
    retry_checkpoint_pth = os.path.join(retry_output_path, "latest.pth")
    retry_checkpoint_exists = os.path.isfile(retry_checkpoint_pth)
    checkpoint = None
    optuna_values = {}
    optuna_values["best"] = None
    optuna_values["best_loss"] = np.inf
    optuna_values["ret_value"] = None

    if retry_trial_number is not None and retry_checkpoint_exists:
        print("Restart trial", retry_trial_number)
        checkpoint = retry_checkpoint_pth
        output_path = retry_output_path

        if os.path.exists(os.path.join(output_path,"optuna_values.json")):
            with open(os.path.join(output_path,"optuna_values.json")) as json_file:
                optuna_values = json.load(json_file)
            print("Read the following optuna values:")
            print(optuna_values)

    os.makedirs(output_path, exist_ok=True)
    pth_log_out = os.path.join(output_path, "out.txt")
    pth_log_err = os.path.join(output_path, "err.txt")
    print("Redirecting stdout to", pth_log_out)
    print("Redirecting stderr to", pth_log_err)
    f = open(pth_log_out, "a")
    sys.stdout = f
    f = open(pth_log_err, "a")
    sys.stderr = f

    ########################
    # Setup datasets
    ########################
    print("RETRY NUMBER", retry_trial_number)
    print("RETRY PATH", retry_checkpoint_pth, " Exists?", retry_checkpoint_exists)

    tconf = TrainingConfiguration(
        pdb_path=PDB_PATH,
        volume_path=VOL_PATH,
        output_path=output_path,
        num_epochs=EPOCHS,
        max_neg=1,
        netconfig=None,
        checkpoint=checkpoint,
        validvolumes=VALID_PATH,
        distance=DISTANCE,
        save_after_improvement=False
    )
    train_triplets, test_triplets = generate_triplets(tconf)

    if "aug_distance" in settings:
        aug_dist = settings["aug_distance"]
    else:
        aug_dist = 2
    print("Use augmentation distance of", aug_dist)
    use_pdb_as_anchor = tconf.pdb_path is not None
    aug_anchor, aug_volumes = get_augmentations(aug_dist, use_pdb_as_anchor=use_pdb_as_anchor)

    train_ds = TripletDataset(
        training_data=train_triplets,
        handler=MRCTripletHandler(),
        augmentation_anchors=aug_anchor,
        augmentation_volumes=aug_volumes,
        label_ext_func=label_filename
    )

    test_ds = TripletDataset(
        training_data=test_triplets,
        handler=MRCTripletHandler(),
        label_ext_func=label_filename
    )

    ########################
    # Create trainer and start training
    ########################
    trainer = TorchTrainer(
        epochs=tconf.num_epochs,
        batchsize=int(BATCH_SIZE),
        learning_rate=learning_rate,
        network=network,
        criterion=LossPyML(
            loss_func=loss_func,
            miner=miner,
        ),
        workers=12,
        log_dir=os.path.join(tconf.output_path, "tensorboard"),
        training_data=train_ds,
        test_data=test_ds,
        output_path=tconf.output_path,
        checkpoint=tconf.checkpoint,
        optimizer=optimizer_name,
        amsgrad=amsgrad,
        weight_decay=weight_decay,
        patience=patience
    )
    trainer.set_network_config(config)

    train_loader, test_loader = trainer.get_train_test_dataloader()

    # Training Loop


    for epoch in tqdm(
        range(trainer.start_epoch, trainer.epochs),
        initial=trainer.start_epoch,
        total=trainer.epochs,
        desc=f"Epochs",
    ):
        print("EPOCH START:", trainer.start_epoch, epoch)
        trainer.current_epoch = epoch
        train_loss = trainer.epoch(train_loader=train_loader)
        print(f"Train Loss: {train_loss:.4f}.")

        # Validation
        if test_loader is not None:
            current_val_loss = trainer.validation_loss(test_loader)
            trainer.scheduler.step(current_val_loss)
            current_val_f1 = trainer.classification_f1_score(test_loader=test_loader)
            print(f"Validation Loss: {current_val_loss:.4f}.")
            print(f"Validation F1 Score: {current_val_f1:.4f}.")
            trainer.save_best_loss(current_val_loss, epoch)
            trainer.save_best_f1(current_val_f1, epoch)

        trainer.writer.flush()

        if trainer.output_path is not None:
            trainer.write_results_to_disk(trainer.output_path)

        if PARAMS["general"]["target"].upper() == "F1":
            trial.report(current_val_f1, epoch)
            optuna_values["ret_value"] = float(current_val_f1)
        elif PARAMS["general"]["target"].upper() == "VAL_LOSS":
            trial.report(current_val_loss, epoch)
            optuna_values["ret_value"] = float(current_val_loss)
        elif PARAMS["general"]["target"].upper() == "F1_AT_VAL_LOSS":
            if current_val_loss < optuna_values["best_loss"]:
                optuna_values["best_loss"] = float(current_val_loss)
                optuna_values["ret_value"] = float(current_val_f1)
            trial.report(optuna_values["ret_value"], epoch)
        else:
            raise UnknownTargetError(
                f"Target {PARAMS['general']['target']} is not known."
            )
        if optuna_values["best"] == None:
            optuna_values["best"] = optuna_values["ret_value"]
        else:
            if PARAMS["general"]["minmax"] == "maximize":
                if optuna_values["ret_value"] > optuna_values["best"]:
                    optuna_values["best"] = optuna_values["ret_value"]
            if PARAMS["general"]["minmax"] == "minimize":
                if optuna_values["ret_value"] < optuna_values["best"]:
                    optuna_values["best"] = optuna_values["ret_value"]

        with open(os.path.join(output_path,"optuna_values.json"), 'w') as fp:
            json.dump(optuna_values, fp)

        # Handle pruning based on the intermediate value.
        if trial.should_prune():
            raise optuna.exceptions.TrialPruned()

    return optuna_values["best"]


def create_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        description="Optuna training interface for TomoTwin",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
    )

    parser.add_argument(
        "-p",
        "--pdbpath",
        type=str,
        default=None,
        help="Path to PDB files that should be use for training",
    )
    parser.add_argument(
        "-v",
        "--volpath",
        type=str,
        required=True,
        help="Path to subtomogram volumes that should be used for training",
    )

    parser.add_argument(
        "--validvolumes",
        type=str,
        default=None,
        required=True,
        help="Path for validation volumes.",
    )

    parser.add_argument(
        "-o",
        "--outpath",
        type=str,
        required=True,
        help="All output files are written in that path.",
    )

    parser.add_argument(
        "-sn",
        "--name",
        type=str,
        required=True,
        help="name of the study.",
    )

    parser.add_argument(
        "-c",
        "--optuna_config",
        type=str,
        required=True,
        help="Path to optuna configuration",
    )

    parser.add_argument(
        "-n",
        "--ntrials",
        type=int,
        default=100,
        help="Number of trials",
    )

    return parser


def _main_():
    """
    Get arguments
    """
    global PDB_PATH
    global VOL_PATH
    global VALID_PATH
    global BASIC_OUTPUT
    global EPOCHS
    global PARAMS
    print("TomoTwin Version:", version("tomotwin"))
    parser = create_parser()
    args = parser.parse_args()
    PDB_PATH = args.pdbpath
    VOL_PATH = args.volpath
    VALID_PATH = args.validvolumes
    STUDY_NAME = args.name
    ntrials = args.ntrials
    optuna_config = args.optuna_config
    if ntrials == -1:
        ntrials = None
    BASIC_OUTPUT = os.path.join(args.outpath, STUDY_NAME)

    """
    READ CONFIGURATION
    """

    f = open(optuna_config)
    PARAMS = json.load(f)
    f.close()

    """
    Setup study
    """
    optuna.logging.set_verbosity(optuna.logging.DEBUG)

    print(PARAMS["general"])

    heartbeat_intervall = None

    storage_url = PARAMS["general"]["STORAGE"]
    # f"sqlite:///{STUDY_NAME}.db"
    storage = None
    if PARAMS["general"]["RDB"] is True:

        retry_callback = None
        if PARAMS["general"]["retry"] is True:
            retry_callback = RetryFailedTrialCallback(
                max_retry=PARAMS["general"]["max_retry"]
            )

        pruner = None
        if PARAMS["general"]["prune"] is True:
            pruner = MedianPruner(n_warmup_steps=PARAMS["general"]["n_warmup"])

        if "heartbeat" in PARAMS["general"]:
            heartbeat_intervall = PARAMS["general"]["heartbeat"]
            print("Use heartbeat interval ", heartbeat_intervall)

        storage = optuna.storages.RDBStorage(
            url=storage_url,
            engine_kwargs={"pool_pre_ping": True},
            heartbeat_interval=heartbeat_intervall,
            failed_trial_callback=retry_callback,
        )
    else:
        storage = storage_url

    study = optuna.create_study(
        direction=PARAMS["general"]["minmax"],
        sampler=samplers.RandomSampler(),
        pruner=pruner,
        study_name=STUDY_NAME,
        storage=storage,
        load_if_exists=True,
    )

    study.optimize(objective, n_trials=ntrials, gc_after_trial=True, n_jobs=1)

    pruned_trials = study.get_trials(deepcopy=False, states=[TrialState.PRUNED])
    complete_trials = study.get_trials(deepcopy=False, states=[TrialState.COMPLETE])

    print("Study statistics: ")
    print("  Number of finished trials: ", len(study.trials))
    print("  Number of pruned trials: ", len(pruned_trials))
    print("  Number of complete trials: ", len(complete_trials))

    print("Best trial:")
    trial = study.best_trial

    print("  Value: ", trial.value)

    print("  Params: ")
    for key, value in trial.params.items():
        print("    {}: {}".format(key, value))
    open(os.path.join(BASIC_OUTPUT, "DONE.txt"), "a").close()


if __name__ == "__main__":
    _main_()
