# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/07_training.metrics.ipynb (unless otherwise specified).

__all__ = ['rpd', 'rpiq', 'stb', 'mape', 'lccc', 'eval_reg']

# Cell
#nbdev_comment from __future__ import annotations
import numpy as np
from sklearn.metrics import (r2_score, mean_squared_error, mean_absolute_error)
from fastcore.test import *

# Cell
def rpd(y:np.ndarray, # Target true value
        y_hat:np.ndarray, # Target predicted value
       ):
    """Ratio of Performance to Deviation"""
    sd = np.std(y, ddof=1)
    sep = np.sqrt(mean_squared_error(y, y_hat))
    return sd / sep

# Cell
def rpiq(y:np.ndarray, # Target true value
         y_hat:np.ndarray, # Target predicted value
        ):
    """Ratio of Performance to Inter-Quartile"""
    q1, q3 = np.percentile(y, [25, 75])
    sep = np.sqrt(mean_squared_error(y, y_hat))
    return (q3 - q1) / sep

# Cell
def stb(y:np.ndarray, # Target true value
        y_hat:np.ndarray, # Target predicted value
       ):
    """Standardized Bias"""
    q1, q3 = np.percentile(y, [25, 75])
    _bias = np.mean(y - y_hat)
    return _bias / (q3 - q1)

# Cell
def mape(y:np.ndarray, # Target true value
         y_hat:np.ndarray, # Target predicted value
        ):
    """Mean Absolute Percentage Error"""
    return 100 * np.mean(np.abs(y - y_hat) / np.abs(y))

# Cell
def lccc(y:np.ndarray, # Target true value
         y_hat:np.ndarray, # Target predicted value
        ):
    """Lin’s concordance correlation coefficient"""
    y = y.flatten()
    y_hat = y_hat.flatten()
    numerator = 2 * np.corrcoef(y, y_hat)[0, 1] * np.std(y) * np.std(y_hat)
    denominator = np.var(y) + np.var(y_hat) + (np.mean(y) - np.mean(y_hat))**2
    return numerator / denominator

# Cell
def eval_reg(y:np.ndarray, # Target true value
             y_hat:np.ndarray, # Target predicted value
             is_log:bool=True, # True if evaluated values are log-10 transformed
            ):
    """Return metrics bundle (rpd, rpiq, r2, lccc, rmse, mse, mae, mape, bias, stb)"""
    _rpd = rpd(y, y_hat)
    _rpiq = rpiq(y, y_hat)
    _r2_score = r2_score(y, y_hat)
    _bias = np.mean(y - y_hat)
    _stb = stb(y, y_hat)
    _lccc = lccc(y, y_hat)
    if is_log:
        y = np.power(10, y)
        y_hat = np.power(10, y_hat)
    _rmse = np.sqrt(mean_squared_error(y, y_hat))
    _mae = mean_absolute_error(y, y_hat)
    _mape = mape(y, y_hat)
    return {
        'rpd': _rpd,
        'rpiq': _rpiq,
        'r2': _r2_score,
        'lccc': _lccc,
        'rmse': _rmse,
        'mse': mean_squared_error(y, y_hat),
        'mae': _mae,
        'mape': _mape,
        'bias': _bias,
        'stb': _stb
    }