"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.QueueEncryption = void 0;
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue.
 */
var QueueEncryption;
(function (QueueEncryption) {
    QueueEncryption["UNENCRYPTED"] = "NONE";
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue.
 */
class Queue extends queue_base_1.QueueBase {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        /**
         * Controls automatic creation of policy objects.
         *
         * Set by subclasses.
         */
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * Import an existing SQS queue provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1).
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue.
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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