import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import { Duration } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { BaseLoadBalancer, BaseLoadBalancerLookupOptions, BaseLoadBalancerProps, ILoadBalancerV2 } from '../shared/base-load-balancer';
import { IpAddressType, ApplicationProtocol } from '../shared/enums';
import { ApplicationListener, BaseApplicationListenerProps } from './application-listener';
/**
 * Properties for defining an Application Load Balancer.
 *
 * @stability stable
 */
export interface ApplicationLoadBalancerProps extends BaseLoadBalancerProps {
    /**
     * Security group to associate with this load balancer.
     *
     * @default A security group is created
     * @stability stable
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The type of IP addresses to use.
     *
     * Only applies to application load balancers.
     *
     * @default IpAddressType.Ipv4
     * @stability stable
     */
    readonly ipAddressType?: IpAddressType;
    /**
     * Indicates whether HTTP/2 is enabled.
     *
     * @default true
     * @stability stable
     */
    readonly http2Enabled?: boolean;
    /**
     * The load balancer idle timeout, in seconds.
     *
     * @default 60
     * @stability stable
     */
    readonly idleTimeout?: Duration;
}
/**
 * Options for looking up an ApplicationLoadBalancer.
 *
 * @stability stable
 */
export interface ApplicationLoadBalancerLookupOptions extends BaseLoadBalancerLookupOptions {
}
/**
 * Define an Application Load Balancer.
 *
 * @stability stable
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
export declare class ApplicationLoadBalancer extends BaseLoadBalancer implements IApplicationLoadBalancer {
    /**
     * Look up an application load balancer.
     *
     * @stability stable
     */
    static fromLookup(scope: Construct, id: string, options: ApplicationLoadBalancerLookupOptions): IApplicationLoadBalancer;
    /**
     * Import an existing Application Load Balancer.
     *
     * @stability stable
     */
    static fromApplicationLoadBalancerAttributes(scope: Construct, id: string, attrs: ApplicationLoadBalancerAttributes): IApplicationLoadBalancer;
    /**
     * @stability stable
     */
    readonly connections: ec2.Connections;
    /**
     * The IP Address Type for this load balancer.
     *
     * @stability stable
     */
    readonly ipAddressType?: IpAddressType;
    /**
     * A list of listeners that have been added to the load balancer.
     *
     * This list is only valid for owned constructs.
     *
     * @stability stable
     */
    readonly listeners: ApplicationListener[];
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ApplicationLoadBalancerProps);
    /**
     * Add a new listener to this load balancer.
     *
     * @stability stable
     */
    addListener(id: string, props: BaseApplicationListenerProps): ApplicationListener;
    /**
     * Add a redirection listener to this load balancer.
     *
     * @stability stable
     */
    addRedirect(props?: ApplicationLoadBalancerRedirectConfig): ApplicationListener;
    /**
     * Add a security group to this load balancer.
     *
     * @stability stable
     */
    addSecurityGroup(securityGroup: ec2.ISecurityGroup): void;
    /**
     * Return the given named metric for this Application Load Balancer.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of concurrent TCP connections active from clients to the load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricActiveConnectionCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of TLS connections initiated by the client that did not establish a session with the load balancer.
     *
     * Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricClientTlsNegotiationErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricConsumedLCUs(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpFixedResponseCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpRedirectCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of redirect actions that couldn't be completed because the URL in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpRedirectUrlLimitExceededCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpCodeElb(code: HttpCodeElb, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricHttpCodeTarget(code: HttpCodeTarget, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricIpv6ProcessedBytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricIpv6RequestCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of new TCP connections established from clients to the load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricNewConnectionCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricProcessedBytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of connections that were rejected because the load balancer had reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRejectedConnectionCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRequestCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricRuleEvaluations(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTargetConnectionErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricTargetResponseTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricTargetTLSNegotiationErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of user authentications that could not be completed.
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricElbAuthError(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of user authentications that could not be completed because the IdP denied access to the user or an authorization code was used more than once.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricElbAuthFailure(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricElbAuthLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     * @stability stable
     */
    metricElbAuthSuccess(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    private cannedMetric;
}
/**
 * Count of HTTP status originating from the load balancer.
 *
 * This count does not include any response codes generated by the targets.
 *
 * @stability stable
 */
export declare enum HttpCodeElb {
    /**
     * The number of HTTP 3XX redirection codes that originate from the load balancer.
     *
     * @stability stable
     */
    ELB_3XX_COUNT = "HTTPCode_ELB_3XX_Count",
    /**
     * The number of HTTP 4XX client error codes that originate from the load balancer.
     *
     * Client errors are generated when requests are malformed or incomplete.
     * These requests have not been received by the target. This count does not
     * include any response codes generated by the targets.
     *
     * @stability stable
     */
    ELB_4XX_COUNT = "HTTPCode_ELB_4XX_Count",
    /**
     * The number of HTTP 5XX server error codes that originate from the load balancer.
     *
     * @stability stable
     */
    ELB_5XX_COUNT = "HTTPCode_ELB_5XX_Count"
}
/**
 * Count of HTTP status originating from the targets.
 *
 * @stability stable
 */
export declare enum HttpCodeTarget {
    /**
     * The number of 2xx response codes from targets.
     *
     * @stability stable
     */
    TARGET_2XX_COUNT = "HTTPCode_Target_2XX_Count",
    /**
     * The number of 3xx response codes from targets.
     *
     * @stability stable
     */
    TARGET_3XX_COUNT = "HTTPCode_Target_3XX_Count",
    /**
     * The number of 4xx response codes from targets.
     *
     * @stability stable
     */
    TARGET_4XX_COUNT = "HTTPCode_Target_4XX_Count",
    /**
     * The number of 5xx response codes from targets.
     *
     * @stability stable
     */
    TARGET_5XX_COUNT = "HTTPCode_Target_5XX_Count"
}
/**
 * An application load balancer.
 *
 * @stability stable
 */
export interface IApplicationLoadBalancer extends ILoadBalancerV2, ec2.IConnectable {
    /**
     * The ARN of this load balancer.
     *
     * @stability stable
     */
    readonly loadBalancerArn: string;
    /**
     * The VPC this load balancer has been created in (if available).
     *
     * If this interface is the result of an import call to fromApplicationLoadBalancerAttributes,
     * the vpc attribute will be undefined unless specified in the optional properties of that method.
     *
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The IP Address Type for this load balancer.
     *
     * @default IpAddressType.IPV4
     * @stability stable
     */
    readonly ipAddressType?: IpAddressType;
    /**
     * A list of listeners that have been added to the load balancer.
     *
     * This list is only valid for owned constructs.
     *
     * @stability stable
     */
    readonly listeners: ApplicationListener[];
    /**
     * Add a new listener to this load balancer.
     *
     * @stability stable
     */
    addListener(id: string, props: BaseApplicationListenerProps): ApplicationListener;
}
/**
 * Properties to reference an existing load balancer.
 *
 * @stability stable
 */
export interface ApplicationLoadBalancerAttributes {
    /**
     * ARN of the load balancer.
     *
     * @stability stable
     */
    readonly loadBalancerArn: string;
    /**
     * ID of the load balancer's security group.
     *
     * @stability stable
     */
    readonly securityGroupId: string;
    /**
     * The canonical hosted zone ID of this load balancer.
     *
     * @default - When not provided, LB cannot be used as Route53 Alias target.
     * @stability stable
     */
    readonly loadBalancerCanonicalHostedZoneId?: string;
    /**
     * The DNS name of this load balancer.
     *
     * @default - When not provided, LB cannot be used as Route53 Alias target.
     * @stability stable
     */
    readonly loadBalancerDnsName?: string;
    /**
     * Whether the security group allows all outbound traffic or not.
     *
     * Unless set to `false`, no egress rules will be added to the security group.
     *
     * @default true
     * @stability stable
     */
    readonly securityGroupAllowsAllOutbound?: boolean;
    /**
     * The VPC this load balancer has been created in, if available.
     *
     * @default - If the Load Balancer was imported and a VPC was not specified,
     * the VPC is not available.
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
}
/**
 * Properties for a redirection config.
 *
 * @stability stable
 */
export interface ApplicationLoadBalancerRedirectConfig {
    /**
     * The protocol of the listener being created.
     *
     * @default HTTP
     * @stability stable
     */
    readonly sourceProtocol?: ApplicationProtocol;
    /**
     * The port number to listen to.
     *
     * @default 80
     * @stability stable
     */
    readonly sourcePort?: number;
    /**
     * The protocol of the redirection target.
     *
     * @default HTTPS
     * @stability stable
     */
    readonly targetProtocol?: ApplicationProtocol;
    /**
     * The port number to redirect to.
     *
     * @default 443
     * @stability stable
     */
    readonly targetPort?: number;
    /**
     * Allow anyone to connect to this listener.
     *
     * If this is specified, the listener will be opened up to anyone who can reach it.
     * For internal load balancers this is anyone in the same VPC. For public load
     * balancers, this is anyone on the internet.
     *
     * If you want to be more selective about who can access this load
     * balancer, set this to `false` and use the listener's `connections`
     * object to selectively grant access to the listener.
     *
     * @default true
     * @stability stable
     */
    readonly open?: boolean;
}
