# Copyright (c) 2014 Pavel 'Blane' Tuchin
# This file is part of pynetdicom2, released under a modified MIT license.
#    See the file license.txt included with this distribution.
"""
This module provides some useful constants for SOP Class UIDs and
Transfer Syntax UIDs.
Storage-related SOP Classes are grouped under STORAGE_SOP_CLASSES
constant.
"""
__author__ = 'Blane'

from . import _dicom

# Transfer syntax

#: Deflated Explicit VR Little Endian
DEFLATED_EXPLICIT_VR_LITTLE_ENDIAN = _dicom.UID('1.2.840.10008.1.2.1.99')
#: JPEG Baseline (Process 1)
JPEG_BASELINE_PROCESS_1 = _dicom.UID('1.2.840.10008.1.2.4.50')
#: JPEG Extended (Process 2 and 4)
JPEG_EXTENDED_PROCESS_2_AND_4 = _dicom.UID('1.2.840.10008.1.2.4.51')
#: JPEG Lossless, Non-Hierarchical (Process 14)
JPEG_LOSSLESS_NON_HIERARCHICAL_PROCESS_14 = _dicom.UID('1.2.840.10008.1.2.4.57')
#: JPEG Lossless, Non-Hierarchical, First-Order Prediction (Process 14 [Selection Value 1])
JPEG_LOSSLESS_NON_HIERARCHICAL_FIRST_ORDER_PREDICTION_PROCESS_14_SELECTION_VALUE_1 = _dicom.UID('1.2.840.10008.1.2.4.70')
#: JPEG-LS Lossless Image Compression
JPEG_LS_LOSSLESS_IMAGE_COMPRESSION = _dicom.UID('1.2.840.10008.1.2.4.80')
#: JPEG-LS Lossy (Near-Lossless) Image Compression
JPEG_LS_LOSSY_NEAR_LOSSLESS_IMAGE_COMPRESSION = _dicom.UID('1.2.840.10008.1.2.4.81')
#: JPEG 2000 Image Compression (Lossless Only)
JPEG_2000_IMAGE_COMPRESSION_LOSSLESS_ONLY = _dicom.UID('1.2.840.10008.1.2.4.90')
#: JPEG 2000 Image Compression
JPEG_2000_IMAGE_COMPRESSION = _dicom.UID('1.2.840.10008.1.2.4.91')
#: JPEG 2000 Part 2 Multi-component Image Compression (Lossless Only)
JPEG_2000_PART_2_MULTI_COMPONENT_IMAGE_COMPRESSION_LOSSLESS_ONLY = _dicom.UID('1.2.840.10008.1.2.4.92')
#: JPEG 2000 Part 2 Multi-component Image Compression
JPEG_2000_PART_2_MULTI_COMPONENT_IMAGE_COMPRESSION = _dicom.UID('1.2.840.10008.1.2.4.93')
#: JPIP Referenced
JPIP_REFERENCED = _dicom.UID('1.2.840.10008.1.2.4.94')
#: JPIP Referenced Deflate
JPIP_REFERENCED_DEFLATE = _dicom.UID('1.2.840.10008.1.2.4.95')
#: MPEG2 Main Profile / Main Level
MPEG2_MAIN_PROFILE_MAIN_LEVEL = _dicom.UID('1.2.840.10008.1.2.4.100')
#: MPEG2 Main Profile / High Level
MPEG2_MAIN_PROFILE_HIGH_LEVEL = _dicom.UID('1.2.840.10008.1.2.4.101')
#: MPEG-4 AVC/H.264 High Profile / Level 4.1
MPEG_4_AVC_H_264_HIGH_PROFILE_LEVEL_4_1 = _dicom.UID('1.2.840.10008.1.2.4.102')
#: MPEG-4 AVC/H.264 BD-compatible High Profile / Level 4.1
MPEG_4_AVC_H_264_BD_COMPATIBLE_HIGH_PROFILE_LEVEL_4_1 = _dicom.UID('1.2.840.10008.1.2.4.103')
#: MPEG-4 AVC/H.264 High Profile / Level 4.2 For 2D Video
MPEG_4_AVC_H_264_HIGH_PROFILE_LEVEL_4_2_FOR_2D_VIDEO = _dicom.UID('1.2.840.10008.1.2.4.104')
#: MPEG-4 AVC/H.264 High Profile / Level 4.2 For 3D Video
MPEG_4_AVC_H_264_HIGH_PROFILE_LEVEL_4_2_FOR_3D_VIDEO = _dicom.UID('1.2.840.10008.1.2.4.105')
#: MPEG-4 AVC/H.264 Stereo High Profile / Level 4.2
MPEG_4_AVC_H_264_STEREO_HIGH_PROFILE_LEVEL_4_2 = _dicom.UID('1.2.840.10008.1.2.4.106')
#: HEVC/H.265 Main Profile / Level 5.1
HEVC_H_265_MAIN_PROFILE_LEVEL_5_1 = _dicom.UID('1.2.840.10008.1.2.4.107')
#: HEVC/H.265 Main 10 Profile / Level 5.1
HEVC_H_265_MAIN_10_PROFILE_LEVEL_5_1 = _dicom.UID('1.2.840.10008.1.2.4.108')
#: RLE Lossless
RLE_LOSSLESS = _dicom.UID('1.2.840.10008.1.2.5')
#: RFC 2557 MIME encapsulation
RFC_2557_MIME_ENCAPSULATION = _dicom.UID('1.2.840.10008.1.2.6.1')
#: XML Encoding
XML_ENCODING = _dicom.UID('1.2.840.10008.1.2.6.2')

# VERIFICATION SOP CLASSES
#: Verification SOP Class
VERIFICATION_SOP_CLASS = _dicom.UID('1.2.840.10008.1.1')

# STORAGE SOP CLASSES

#: Computed Radiography Image Storage
CR_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.1')
#: Digital X-Ray Image Storage - For Presentation
DX_IMAGE_STORAGE_FOR_PRESENTATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.1.1')
#: Digital X-Ray Image Storage - For Processing
DX_IMAGE_STORAGE_FOR_PROCESSING = _dicom.UID('1.2.840.10008.5.1.4.1.1.1.1.1')
#: Digital Mammography X-Ray Image Storage - For Presentation
MG_IMAGE_STORAGE_FOR_PRESENTATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.1.2')
#: Digital Mammography X-Ray Image Storage - For Processing
MG_IMAGE_STORAGE_FOR_PROCESSING = _dicom.UID('1.2.840.10008.5.1.4.1.1.1.2.1')
#: Digital Intra-Oral X-Ray Image Storage - For Presentation
IO_IMAGE_STORAGE_FOR_PRESENTATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.1.3')
#: Digital Intra-Oral X-Ray Image Storage - For Processing
IO_IMAGE_STORAGE_FOR_PROCESSING = _dicom.UID('1.2.840.10008.5.1.4.1.1.1.3.1')
#: CT Image Storage
CT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.2')
#: Enhanced CT Image Storage
ENHANCED_CT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.2.1')
#: Legacy Converted Enhanced CT Image Storage
LEGACY_ENHANCED_CT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.2.2')
#: Ultrasound Multi-frame Image Storage
ULTRASOUND_MULTI_FRAME_IMAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.3.1')
#: MR Image Storage
MR_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.4')
#: Enhanced MR Image Storage
ENHANCED_MR_IMAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.4.1')
#: MR Spectroscopy Storage
MR_SPECTROSCOPY_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.4.2')
#: Enhanced MR Color Image Storage
ENHANCED_MR_COLOR_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.4.3')
#: Legacy Converted Enhanced MR Image Storage
LEGACY_CONVERTED_ENHANCED_MR_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.4.4')
#: Ultrasound Image Storage
ULTRASOUND_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.6.1')
#: Enhanced US Volume Storage
ENHANCED_US_VOLUME_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.6.2')
#: Secondary Capture Image Storage
SC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.7')
#: Multi-frame Single Bit Secondary Capture Image Storage
MULTI_FRAME_SINGLE_BIT_SC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.7.1')
#: Multi-frame Grayscale Byte Secondary Capture Image Storage
MULTI_FRAME_GRAYSCALE_BYTE_SC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.7.2')
#: Multi-frame Grayscale Word Secondary Capture Image Storage
MULTI_FRAME_GRAYSCALE_WORD_SC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.7.3')
#: Multi-frame True Color Secondary Capture Image Storage
MULTI_FRAME_TRUE_COLOR_SC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.7.4')
#: 12-lead ECG Waveform Storage
TWELVE_LEAD_ECG_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.1.1')
#: General ECG Waveform Storage
GENERAL_ECG_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.1.2')
#: Ambulatory ECG Waveform Storage
AMBULATORY_ECG_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.1.3')
#: Hemodynamic Waveform Storage
HEMODYNAMIC_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.2.1')
#: Cardiac Electrophysiology Waveform Storage
CARDIAC_ELECTROPHYSIOLOGY_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.3.1')
#: Basic Voice Audio Waveform Storage
BASIC_VOICE_AUDIO_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.4.1')
#: General Audio Waveform Storage
GENERAL_AUDIO_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.4.2')
#: Arterial Pulse Waveform Storage
ARTERIAL_PULSE_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.5.1')
#: Respiratory Waveform Storage
RESPIRATORY_WAVEFORM_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.9.6.1')
#: Grayscale Softcopy Presentation State Storage
GRAYSCALE_SOFTCOPY_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.1')
#: Color Softcopy Presentation State Storage
COLOR_SOFTCOPY_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.2')
#: Pseudo-Color Softcopy Presentation State Storage
PSEUDO_COLOR_SOFTCOPY_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.3')
#: Blending Softcopy Presentation State Storage
BLENDING_SOFTCOPY_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.4')
#: XA/XRF Grayscale Softcopy Presentation State Storage
XA_XRF_GRAYSCALE_SOFTCOPY_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.5')
#: Grayscale Planar MPR Volumetric Presentation State Storage
GRAYSCALE_PLANAR_MPR_VOLUMETRIC_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.6')
#: Compositing Planar MPR Volumetric Presentation State Storage
COMPOSITING_PLANAR_MPR_VOLUMETRIC_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.7')
#: Advanced Blending Presentation State Storage
ADVANCED_BLENDING_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.8')
#: Volume Rendering Volumetric Presentation State Storage
VOLUME_RENDERING_VOLUMETRIC_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.9')
#: Segmented Volume Rendering Volumetric Presentation State Storage
SEGMENTED_VOLUME_RENDERING_VOLUMETRIC_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.10')
#: Multiple Volume Rendering Volumetric Presentation State Storage
MULTIPLE_VOLUME_RENDERING_VOLUMETRIC_PRESENTATION_STATE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.11.11')
#: X-Ray Angiographic Image Storage
XA_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.12.1')
#: Enhanced XA Image Storage
ENHANCED_XA_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.12.1.1')
#: X-Ray Radiofluoroscopic Image Storage
RF_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.12.2')
#: Enhanced XRF Image Storage
ENHANCED_RF_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.12.2.1')
#: X-Ray 3D Angiographic Image Storage
X_RAY_3D_ANGIOGRAPHIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.13.1.1')
#: X-Ray 3D Craniofacial Image Storage
X_RAY_3D_CRANIOFACIAL_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.13.1.2')
#: Breast Tomosynthesis Image Storage
BREAST_TOMOSYNTHESIS_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.13.1.3')
#: Breast Projection X-Ray Image Storage - For Presentation
BREAST_PROJECTION_X_RAY_IMAGE_STORAGE_FOR_PRESENTATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.13.1.4')
#: Breast Projection X-Ray Image Storage - For Processing
BREAST_PROJECTION_X_RAY_IMAGE_STORAGE_FOR_PROCESSING = _dicom.UID('1.2.840.10008.5.1.4.1.1.13.1.5')
#: Intravascular Optical Coherence Tomography Image Storage - For Presentation
INTRAVASCULAR_OPTICAL_COHERENCE_TOMOGRAPHY_IMAGE_STORAGE_FOR_PRESENTATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.14.1')
#: Intravascular Optical Coherence Tomography Image Storage - For Processing
INTRAVASCULAR_OPTICAL_COHERENCE_TOMOGRAPHY_IMAGE_STORAGE_FOR_PROCESSING = _dicom.UID('1.2.840.10008.5.1.4.1.1.14.2')
#: Nuclear Medicine Image Storage
NM_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.20')
#: Parametric Map Storage
PARAMETRIC_MAP_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.30')
#: Raw Data Storage
RAW_DATA_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.66')
#: Spatial Registration Storage
SPATIAL_REGISTRATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.66.1')
#: Spatial Fiducials Storage
SPATIAL_FIDUCIALS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.66.2')
#: Deformable Spatial Registration Storage
DEFORMABLE_SPATIAL_REGISTRATION_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.66.3')
#: Segmentation Storage
SEGMENTATION_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.66.4')
#: Surface Segmentation Storage
SURFACE_SEGMENTATION_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.66.5')
#: Tractography Results Storage
TRACTOGRAPHY_RESULTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.66.6')
#: Real World Value Mapping Storage
REAL_WORLD_VALUE_MAPPING_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.67')
#: Surface Scan Mesh Storage
SURFACE_SCAN_MESH_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.68.1')
#: Surface Scan Point Cloud Storage
SURFACE_SCAN_POINT_CLOUD_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.68.2')
#: VL Endoscopic Image Storage
VL_ENDOSCOPIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.1')
#: Video Endoscopic Image Storage
VIDEO_ENDOSCOPIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.1.1')
#: VL Microscopic Image Storage
VL_MICROSCOPIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.2')
#: Video Microscopic Image Storage
VIDEO_MICROSCOPIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.2.1')
#: VL Slide-Coordinates Microscopic Image Storage
VL_SLIDE_COORDINATES_MICROSCOPIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.3')
#: VL Photographic Image Storage
VL_PHOTOGRAPHIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.4')
#: Video Photographic Image Storage
VIDEO_PHOTOGRAPHIC_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.4.1')
#: Ophthalmic Photography 8 Bit Image Storage
OPHTHALMIC_PHOTOGRAPHY_8_BIT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.1')
#: Ophthalmic Photography 16 Bit Image Storage
OPHTHALMIC_PHOTOGRAPHY_16_BIT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.2')
#: Stereometric Relationship Storage
STEREOMETRIC_RELATIONSHIP_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.3')
#: Ophthalmic Tomography Image Storage
OPHTHALMIC_TOMOGRAPHY_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.4')
#: Wide Field Ophthalmic Photography Stereographic Projection Image Storage
WIDE_FIELD_OPHTHALMIC_PHOTOGRAPHY_STEREOGRAPHIC_PROJECTION_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.5')
#: Wide Field Ophthalmic Photography 3D Coordinates Image Storage
WIDE_FIELD_OPHTHALMIC_PHOTOGRAPHY_3D_COORDINATES_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.6')
#: Ophthalmic Optical Coherence Tomography En Face Image Storage
OPHTHALMIC_OPTICAL_COHERENCE_TOMOGRAPHY_EN_FACE_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.7')
#: Ophthalmic Optical Coherence Tomography B-scan Volume Analysis Storage
OPHTHALMIC_OPTICAL_COHERENCE_TOMOGRAPHY_B_SCAN_VOLUME_ANALYSIS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.5.8')
#: VL Whole Slide Microscopy Image Storage
VL_WHOLE_SLIDE_MICROSCOPY_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.77.1.6')
#: Lensometry Measurements Storage
LENSOMETRY_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.1')
#: Autorefraction Measurements Storage
AUTOREFRACTION_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.2')
#: Keratometry Measurements Storage
KERATOMETRY_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.3')
#: Subjective Refraction Measurements Storage
SUBJECTIVE_REFRACTION_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.4')
#: Visual Acuity Measurements Storage
VISUAL_ACUITY_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.5')
#: Spectacle Prescription Report Storage
SPECTACLE_PRESCRIPTION_REPORT_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.6')
#: Ophthalmic Axial Measurements Storage
OPHTHALMIC_AXIAL_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.7')
#: Intraocular Lens Calculations Storage
INTRAOCULAR_LENS_CALCULATIONS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.78.8')
#: Macular Grid Thickness and Volume Report Storage
MACULAR_GRID_THICKNESS_AND_VOLUME_REPORT_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.79.1')
#: Ophthalmic Visual Field Static Perimetry Measurements Storage
OPHTHALMIC_VISUAL_FIELD_STATIC_PERIMETRY_MEASUREMENTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.80.1')
#: Ophthalmic Thickness Map Storage
OPHTHALMIC_THICKNESS_MAP_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.81.1')
#: Corneal Topography Map Storage
CORNEAL_TOPOGRAPHY_MAP_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.82.1')
#: Basic Text SR Storage
BASIC_TEXT_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.11')
#: Enhanced SR Storage
ENHANCED_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.22')
#: Comprehensive SR Storage
COMPREHENSIVE_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.33')
#: Comprehensive 3D SR Storage
COMPREHENSIVE_3D_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.34')
#: Extensible SR Storage
EXTENSIBLE_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.35')
#: Procedure Log Storage
PROCEDURE_LOG_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.40')
#: Mammography CAD SR Storage
MAMMOGRAPHY_CAD_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.50')
#: Key Object Selection Document Storage
KEY_OBJECT_SELECTION_DOCUMENT_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.59')
#: Chest CAD SR Storage
CHEST_CAD_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.65')
#: X-Ray Radiation Dose SR Storage
XRAY_RADIATION_DOSE_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.67')
#: Radiopharmaceutical Radiation Dose SR Storage
RADIOPHARMACEUTICAL_RADIATION_DOSE_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.68')
#: Colon CAD SR Storage
COLON_CAD_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.69')
#: Implantation Plan SR Storage
IMPLANTATION_PLAN_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.70')
#: Acquisition Context SR Storage
ACQUISITION_CONTEXT_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.71')
#: Simplified Adult Echo SR Storage
SIMPLIFIED_ADULT_ECHO_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.72')
#: Patient Radiation Dose SR Storage
PATIENT_RADIATION_DOSE_SR_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.88.73')
#: Content Assessment Results Storage
CONTENT_ASSESSMENT_RESULTS_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.90.1')
#: Encapsulated PDF Storage
ENCAPSULATED_PDF_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.104.1')
#: Encapsulated CDA Storage
ENCAPSULATED_CDA_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.104.2')
#: Positron Emission Tomography Image Storage
PET_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.128')
#: Legacy Converted Enhanced PET Image Storage
LEGACY_CONVERTED_ENHANCED_PET_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.128.1')
#: Standalone PET Curve Storage
STANDALONE_PET_CURVE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.129')
#: Enhanced PET Image Storage
ENHANCED_PET_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.130')
#: Basic Structured Display Storage
BASIC_STRUCTURED_DISPLAY_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.131')
#: CT Defined Procedure Protocol Storage
CT_DEFINED_PROCEDURE_PROTOCOL_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.200.1')
#: CT Performed Procedure Protocol Storage
CT_PERFORMED_PROCEDURE_PROTOCOL_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.200.2')
#: Protocol Approval Storage
PROTOCOL_APPROVAL_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.200.3')
#: RT Image Storage
RT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.1')
#: RT Dose Storage
RT_DOSE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.2')
#: RT Structure Set Storage
RT_STRUCTURE_SET_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.3')
#: RT Beams Treatment Record Storage
RT_BEAMS_TREATMENT_RECORD_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.4')
#: RT Plan Storage
RT_PLAN_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.5')
#: RT Brachy Treatment Record Storage
RT_BRACHY_TREATMENT_RECORD_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.6')
#: RT Treatment Summary Record Storage
RT_TREATMENT_SUMMARY_RECORD_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.7')
#: RT Ion Plan Storage
RT_ION_PLAN_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.8')
#: RT Ion Beams Treatment Record Storage
RT_ION_BEAMS_TREATMENT_RECORD_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.481.9')
#: DICOS CT Image Storage
DICOS_CT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.1')
#: DICOS Digital X-Ray Image Storage - For Presentation
DICOS_DIGITAL_X_RAY_IMAGE_STORAGE_FOR_PRESENTATION = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.2.1')
#: DICOS Digital X-Ray Image Storage - For Processing
DICOS_DIGITAL_X_RAY_IMAGE_STORAGE_FOR_PROCESSING = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.2.2')
#: DICOS Threat Detection Report Storage
DICOS_THREAT_DETECTION_REPORT_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.3')
#: DICOS 2D AIT Storage
DICOS_2D_AIT_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.4')
#: DICOS 3D AIT Storage
DICOS_3D_AIT_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.5')
#: DICOS Quadrupole Resonance (QR) Storage
DICOS_QR_Storage = _dicom.UID('1.2.840.10008.5.1.4.1.1.501.6')
#: Eddy Current Image Storage
EDDY_CURRENT_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.601.1')
#: Eddy Current Multi-frame Image Storage
EDDY_CURRENT_MULTI_FRAME_IMAGE_STORAGE = _dicom.UID('1.2.840.10008.5.1.4.1.1.601.2')


# QUERY RETRIEVE SOP Classes
PATIENT_ROOT_FIND_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.1.1')
PATIENT_ROOT_MOVE_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.1.2')
PATIENT_ROOT_GET_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.1.3')
STUDY_ROOT_FIND_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.2.1')
STUDY_ROOT_MOVE_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.2.2')
STUDY_ROOT_GET_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.2.3')
PATIENT_STUDY_ONLY_FIND_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.3.1')
PATIENT_STUDY_ONLY_MOVE_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.3.2')
PATIENT_STUDY_ONLY_GET_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.1.2.3.3')

MODALITY_WORK_LIST_INFORMATION_FIND_SOP_CLASS = _dicom.UID('1.2.840.10008.5.1.4.31')

STORAGE_COMMITMENT_SOP_CLASS = _dicom.UID('1.2.840.10008.1.20.1')

STORAGE_SOP_CLASSES = [
    CR_IMAGE_STORAGE,
    DX_IMAGE_STORAGE_FOR_PRESENTATION,
    DX_IMAGE_STORAGE_FOR_PROCESSING,
    MG_IMAGE_STORAGE_FOR_PRESENTATION,
    MG_IMAGE_STORAGE_FOR_PROCESSING,
    IO_IMAGE_STORAGE_FOR_PRESENTATION,
    IO_IMAGE_STORAGE_FOR_PROCESSING,
    CT_IMAGE_STORAGE,
    ENHANCED_CT_IMAGE_STORAGE,
    LEGACY_ENHANCED_CT_IMAGE_STORAGE,
    ULTRASOUND_MULTI_FRAME_IMAGE,
    MR_IMAGE_STORAGE,
    ENHANCED_MR_IMAGE,
    MR_SPECTROSCOPY_STORAGE,
    ENHANCED_MR_COLOR_IMAGE_STORAGE,
    LEGACY_CONVERTED_ENHANCED_MR_IMAGE_STORAGE,
    ULTRASOUND_IMAGE_STORAGE,
    ENHANCED_US_VOLUME_STORAGE,
    SC_IMAGE_STORAGE,
    MULTI_FRAME_SINGLE_BIT_SC_IMAGE_STORAGE,
    MULTI_FRAME_GRAYSCALE_BYTE_SC_IMAGE_STORAGE,
    MULTI_FRAME_GRAYSCALE_WORD_SC_IMAGE_STORAGE,
    MULTI_FRAME_TRUE_COLOR_SC_IMAGE_STORAGE,
    TWELVE_LEAD_ECG_WAVEFORM_STORAGE,
    GENERAL_ECG_WAVEFORM_STORAGE,
    AMBULATORY_ECG_WAVEFORM_STORAGE,
    HEMODYNAMIC_WAVEFORM_STORAGE,
    CARDIAC_ELECTROPHYSIOLOGY_WAVEFORM_STORAGE,
    BASIC_VOICE_AUDIO_WAVEFORM_STORAGE,
    GENERAL_AUDIO_WAVEFORM_STORAGE,
    ARTERIAL_PULSE_WAVEFORM_STORAGE,
    RESPIRATORY_WAVEFORM_STORAGE,
    GRAYSCALE_SOFTCOPY_PRESENTATION_STATE_STORAGE,
    COLOR_SOFTCOPY_PRESENTATION_STATE_STORAGE,
    PSEUDO_COLOR_SOFTCOPY_PRESENTATION_STATE_STORAGE,
    BLENDING_SOFTCOPY_PRESENTATION_STATE_STORAGE,
    XA_XRF_GRAYSCALE_SOFTCOPY_PRESENTATION_STATE_STORAGE,
    GRAYSCALE_PLANAR_MPR_VOLUMETRIC_PRESENTATION_STATE_STORAGE,
    COMPOSITING_PLANAR_MPR_VOLUMETRIC_PRESENTATION_STATE_STORAGE,
    ADVANCED_BLENDING_PRESENTATION_STATE_STORAGE,
    VOLUME_RENDERING_VOLUMETRIC_PRESENTATION_STATE_STORAGE,
    SEGMENTED_VOLUME_RENDERING_VOLUMETRIC_PRESENTATION_STATE_STORAGE,
    MULTIPLE_VOLUME_RENDERING_VOLUMETRIC_PRESENTATION_STATE_STORAGE,
    XA_IMAGE_STORAGE,
    ENHANCED_XA_IMAGE_STORAGE,
    RF_IMAGE_STORAGE,
    ENHANCED_RF_IMAGE_STORAGE,
    X_RAY_3D_ANGIOGRAPHIC_IMAGE_STORAGE,
    X_RAY_3D_CRANIOFACIAL_IMAGE_STORAGE,
    BREAST_TOMOSYNTHESIS_IMAGE_STORAGE,
    BREAST_PROJECTION_X_RAY_IMAGE_STORAGE_FOR_PRESENTATION,
    BREAST_PROJECTION_X_RAY_IMAGE_STORAGE_FOR_PROCESSING,
    INTRAVASCULAR_OPTICAL_COHERENCE_TOMOGRAPHY_IMAGE_STORAGE_FOR_PRESENTATION,
    INTRAVASCULAR_OPTICAL_COHERENCE_TOMOGRAPHY_IMAGE_STORAGE_FOR_PROCESSING,
    NM_IMAGE_STORAGE,
    PARAMETRIC_MAP_STORAGE,
    RAW_DATA_STORAGE,
    SPATIAL_REGISTRATION,
    SPATIAL_FIDUCIALS_STORAGE,
    DEFORMABLE_SPATIAL_REGISTRATION_STORAGE,
    SEGMENTATION_STORAGE,
    SURFACE_SEGMENTATION_STORAGE,
    TRACTOGRAPHY_RESULTS_STORAGE,
    REAL_WORLD_VALUE_MAPPING_STORAGE,
    SURFACE_SCAN_MESH_STORAGE,
    SURFACE_SCAN_POINT_CLOUD_STORAGE,
    VL_ENDOSCOPIC_IMAGE_STORAGE,
    VIDEO_ENDOSCOPIC_IMAGE_STORAGE,
    VL_MICROSCOPIC_IMAGE_STORAGE,
    VIDEO_MICROSCOPIC_IMAGE_STORAGE,
    VL_SLIDE_COORDINATES_MICROSCOPIC_IMAGE_STORAGE,
    VL_PHOTOGRAPHIC_IMAGE_STORAGE,
    VIDEO_PHOTOGRAPHIC_IMAGE_STORAGE,
    OPHTHALMIC_PHOTOGRAPHY_8_BIT_IMAGE_STORAGE,
    OPHTHALMIC_PHOTOGRAPHY_16_BIT_IMAGE_STORAGE,
    STEREOMETRIC_RELATIONSHIP_STORAGE,
    OPHTHALMIC_TOMOGRAPHY_IMAGE_STORAGE,
    WIDE_FIELD_OPHTHALMIC_PHOTOGRAPHY_STEREOGRAPHIC_PROJECTION_IMAGE_STORAGE,
    WIDE_FIELD_OPHTHALMIC_PHOTOGRAPHY_3D_COORDINATES_IMAGE_STORAGE,
    OPHTHALMIC_OPTICAL_COHERENCE_TOMOGRAPHY_EN_FACE_IMAGE_STORAGE,
    OPHTHALMIC_OPTICAL_COHERENCE_TOMOGRAPHY_B_SCAN_VOLUME_ANALYSIS_STORAGE,
    VL_WHOLE_SLIDE_MICROSCOPY_IMAGE_STORAGE,
    LENSOMETRY_MEASUREMENTS_STORAGE,
    AUTOREFRACTION_MEASUREMENTS_STORAGE,
    KERATOMETRY_MEASUREMENTS_STORAGE,
    SUBJECTIVE_REFRACTION_MEASUREMENTS_STORAGE,
    VISUAL_ACUITY_MEASUREMENTS_STORAGE,
    SPECTACLE_PRESCRIPTION_REPORT_STORAGE,
    OPHTHALMIC_AXIAL_MEASUREMENTS_STORAGE,
    INTRAOCULAR_LENS_CALCULATIONS_STORAGE,
    MACULAR_GRID_THICKNESS_AND_VOLUME_REPORT_STORAGE,
    OPHTHALMIC_VISUAL_FIELD_STATIC_PERIMETRY_MEASUREMENTS_STORAGE,
    OPHTHALMIC_THICKNESS_MAP_STORAGE,
    CORNEAL_TOPOGRAPHY_MAP_STORAGE,
    BASIC_TEXT_SR_STORAGE,
    ENHANCED_SR_STORAGE,
    COMPREHENSIVE_SR_STORAGE,
    COMPREHENSIVE_3D_SR_STORAGE,
    EXTENSIBLE_SR_STORAGE,
    PROCEDURE_LOG_STORAGE,
    MAMMOGRAPHY_CAD_SR_STORAGE,
    KEY_OBJECT_SELECTION_DOCUMENT_STORAGE,
    CHEST_CAD_SR_STORAGE,
    XRAY_RADIATION_DOSE_SR_STORAGE,
    RADIOPHARMACEUTICAL_RADIATION_DOSE_SR_STORAGE,
    COLON_CAD_SR_STORAGE,
    IMPLANTATION_PLAN_SR_STORAGE,
    ACQUISITION_CONTEXT_SR_STORAGE,
    SIMPLIFIED_ADULT_ECHO_SR_STORAGE,
    PATIENT_RADIATION_DOSE_SR_STORAGE,
    CONTENT_ASSESSMENT_RESULTS_STORAGE,
    ENCAPSULATED_PDF_STORAGE,
    ENCAPSULATED_CDA_STORAGE,
    PET_IMAGE_STORAGE,
    LEGACY_CONVERTED_ENHANCED_PET_IMAGE_STORAGE,
    STANDALONE_PET_CURVE_STORAGE,
    ENHANCED_PET_IMAGE_STORAGE,
    BASIC_STRUCTURED_DISPLAY_STORAGE,
    CT_DEFINED_PROCEDURE_PROTOCOL_STORAGE,
    CT_PERFORMED_PROCEDURE_PROTOCOL_STORAGE,
    PROTOCOL_APPROVAL_STORAGE,
    RT_IMAGE_STORAGE,
    RT_DOSE_STORAGE,
    RT_STRUCTURE_SET_STORAGE,
    RT_BEAMS_TREATMENT_RECORD_STORAGE,
    RT_PLAN_STORAGE,
    RT_BRACHY_TREATMENT_RECORD_STORAGE,
    RT_TREATMENT_SUMMARY_RECORD_STORAGE,
    RT_ION_PLAN_STORAGE,
    RT_ION_BEAMS_TREATMENT_RECORD_STORAGE,
    DICOS_CT_IMAGE_STORAGE,
    DICOS_DIGITAL_X_RAY_IMAGE_STORAGE_FOR_PRESENTATION,
    DICOS_DIGITAL_X_RAY_IMAGE_STORAGE_FOR_PROCESSING,
    DICOS_THREAT_DETECTION_REPORT_STORAGE,
    DICOS_2D_AIT_STORAGE,
    DICOS_3D_AIT_STORAGE,
    DICOS_QR_Storage,
    EDDY_CURRENT_IMAGE_STORAGE,
    EDDY_CURRENT_MULTI_FRAME_IMAGE_STORAGE
]