import functools

from django.urls import include, path, re_path
from django.views.decorators.cache import never_cache
from django.views.generic import TemplateView
from django.http import Http404
from django.views.defaults import page_not_found

from wagtail.admin.auth import require_admin_access
from wagtail.admin.urls import pages as wagtailadmin_pages_urls
from wagtail.admin.urls import collections as wagtailadmin_collections_urls
from wagtail.admin.urls import reports as wagtailadmin_reports_urls
from wagtail.admin.urls import password_reset as wagtailadmin_password_reset_urls
from wagtail.admin.urls import workflows as wagtailadmin_workflows_urls
from wagtail.admin.views import account, chooser, home, pages, tags, userbar
from wagtail.admin.api import urls as api_urls
from wagtail.core import hooks
from wagtail.utils.urlpatterns import decorate_urlpatterns


urlpatterns = [
    path('', home.home, name='wagtailadmin_home'),

    path('test404/', TemplateView.as_view(template_name='wagtailadmin/404.html')),

    path('api/', include(api_urls)),

    path('failwhale/', home.error_test, name='wagtailadmin_error_test'),

    # TODO: Move into wagtailadmin_pages namespace
    path('pages/', pages.index, name='wagtailadmin_explore_root'),
    path('pages/<int:parent_page_id>/', pages.index, name='wagtailadmin_explore'),

    path('pages/', include(wagtailadmin_pages_urls, namespace='wagtailadmin_pages')),

    # TODO: Move into wagtailadmin_pages namespace
    path('choose-page/', chooser.browse, name='wagtailadmin_choose_page'),
    path('choose-page/<int:parent_page_id>/', chooser.browse, name='wagtailadmin_choose_page_child'),
    path('choose-page/search/', chooser.search, name='wagtailadmin_choose_page_search'),
    path('choose-external-link/', chooser.external_link, name='wagtailadmin_choose_page_external_link'),
    path('choose-email-link/', chooser.email_link, name='wagtailadmin_choose_page_email_link'),
    path('choose-phone-link/', chooser.phone_link, name='wagtailadmin_choose_page_phone_link'),
    path('choose-anchor-link/', chooser.anchor_link, name='wagtailadmin_choose_page_anchor_link'),

    path('tag-autocomplete/', tags.autocomplete, name='wagtailadmin_tag_autocomplete'),
    path('tag-autocomplete/<slug:app_name>/<slug:model_name>/', tags.autocomplete, name='wagtailadmin_tag_model_autocomplete'),

    path('collections/', include(wagtailadmin_collections_urls, namespace='wagtailadmin_collections')),
    path('workflows/', include(wagtailadmin_workflows_urls, namespace='wagtailadmin_workflows')),
    path('reports/', include(wagtailadmin_reports_urls, namespace='wagtailadmin_reports')),

    path('account/', account.account, name='wagtailadmin_account'),
    path('account/change_password/', account.change_password, name='wagtailadmin_account_change_password'),
    path('account/change_email/', account.change_email, name='wagtailadmin_account_change_email'),
    path('account/change_name/', account.change_name, name='wagtailadmin_account_change_name'),
    path(
        'account/notification_preferences/',
        account.notification_preferences,
        name='wagtailadmin_account_notification_preferences'
    ),
    path('account/change_avatar/', account.change_avatar, name='wagtailadmin_account_change_avatar'),
    path(
        'account/language_preferences/',
        account.language_preferences,
        name='wagtailadmin_account_language_preferences'
    ),
    path(
        'account/current_time_zone/',
        account.current_time_zone,
        name='wagtailadmin_account_current_time_zone'
    ),
    path('logout/', account.LogoutView.as_view(), name='wagtailadmin_logout'),
]


# Import additional urlpatterns from any apps that define a register_admin_urls hook
for fn in hooks.get_hooks('register_admin_urls'):
    urls = fn()
    if urls:
        urlpatterns += urls


# Add "wagtailadmin.access_admin" permission check
urlpatterns = decorate_urlpatterns(urlpatterns, require_admin_access)


# These url patterns do not require an authenticated admin user
urlpatterns += [
    path('login/', account.LoginView.as_view(), name='wagtailadmin_login'),

    # These two URLs have the "permission_required" decorator applied directly
    # as they need to fail with a 403 error rather than redirect to the login page
    path('userbar/<int:page_id>/', userbar.for_frontend, name='wagtailadmin_userbar_frontend'),
    path('userbar/moderation/<int:revision_id>/', userbar.for_moderation, name='wagtailadmin_userbar_moderation'),

    # Password reset
    path('password_reset/', include(wagtailadmin_password_reset_urls)),

    # Default view (will show 404 page)
    # This must be the last URL in this file!
    re_path(r'^', home.default),
]


# Hook in our own 404 handler
def display_custom_404(view_func):
    @functools.wraps(view_func)
    def wrapper(request, *args, **kwargs):
        try:
            return view_func(request, *args, **kwargs)
        except Http404:
            return page_not_found(request, '', template_name='wagtailadmin/404.html')

    return wrapper


urlpatterns = decorate_urlpatterns(urlpatterns, display_custom_404)


# Decorate all views with cache settings to prevent caching
urlpatterns = decorate_urlpatterns(
    urlpatterns,
    never_cache
)
