#!/usr/bin/env python3
# MIT License
#
# Copyright (c) 2020 FABRIC Testbed
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#
#
# Author: Komal Thareja (kthare10@renci.org)

import enum

from .base_sliver import BaseSliver
from .switch_fabric import SwitchFabricInfo


@enum.unique
class ComponentType(enum.Enum):
    GPU = enum.auto()
    SmartNIC = enum.auto()
    SharedNIC = enum.auto()
    FPGA = enum.auto()
    NVME = enum.auto()

    def __repr__(self):
        return self.name

    def __str__(self):
        return self.name


class ComponentSliver(BaseSliver):

    def __init__(self):
        super().__init__()
        self.switch_fabric_info = None

    def set_switch_fabric_info(self, sf_info: SwitchFabricInfo):
        self.switch_fabric_info = sf_info

    @staticmethod
    def type_from_str(ctype: str) -> ComponentType or None:
        if ctype is None:
            return None
        for t in ComponentType:
            if ctype == str(t):
                return t


class AttachedComponentsInfo:
    """
    Stores attached components as a dictionary by PCI ID
    """
    def __init__(self):
        self.devices = {}

    def add_device(self, device_info: ComponentSliver):
        self.devices[device_info.resource_name] = device_info

    def remove_device(self, name: str):
        if name in self.devices:
            self.devices.pop(name)

    def get_device(self, name: str):
        return self.devices.get(name, None)
