#!/usr/bin/env python
# coding: utf-8
# /*##########################################################################
#
# Copyright (c) 2017 European Synchrotron Radiation Facility
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# ###########################################################################*/
"""This module describe tomwer applications which are available  through
the silx launcher.

Your environment should provide a command `tomwer`. You can reach help with
`tomwer --help`, and check the version with `tomwer --version`.
"""

__authors__ = ["V. Valls", "P. Knobel", "H. Payno"]
__license__ = "MIT"
__date__ = "10/11/2020"


import sys
from tomwer.utils import Launcher
import tomwer.version


def main():
    """Main function of the launcher

    This function is referenced in the setup.py file, to create a
    launcher script generated by setuptools.

    :rtype: int
    :returns: The execution status
    """
    _version = tomwer.version.version
    launcher = Launcher(prog="tomwer", version=_version)
    launcher.add_command(
        "axis",
        module_name="tomwer.app.axis",
        description="Compute center of rotation of a scan or "
        "between two projections",
    )
    launcher.add_command(
        "canvas", module_name="tomwer.app.canvas", description="open the orange-canvas"
    )
    launcher.add_command(
        "dark-ref",
        module_name="tomwer.app.darkref",
        description="Compute mean / median darks and flat fields",
    )
    launcher.add_command(
        "dark-ref-patch",
        module_name="tomwer.app.darkrefpatch",
        description="patch dark and / or flat to an NXTomo entry",
    )
    launcher.add_command(
        "diff-frame",
        module_name="tomwer.app.diffframe",
        description="Tool to compare any frame from the acquisition",
    )
    launcher.add_command(
        "image-key-editor",
        module_name="tomwer.app.imagekeyeditor",
        description="Tool to modify in place frames image key parameter.",
    )
    launcher.add_command(
        "norm-intensity",
        module_name="tomwer.app.intensitynormalization",
        description="Tool to compute intensity normalization.",
    )
    launcher.add_command(
        "lamino",
        module_name="tomwer.app.lamino",
        description="Run laminography reconstruction using tofu",
    )
    launcher.add_command(
        "liveslice",
        module_name="tomwer.app.liveslice",
        description="Run liveslice application",
    )
    launcher.add_command(
        "nabu",
        module_name="tomwer.app.nabuapp",
        description="Use nabu to reconstruct a given scan",
    )
    launcher.add_command(
        "ows-to-script",
        module_name="tomwer.app.ows_to_script",
        description="Convert a .ows file to python script",
    )
    launcher.add_command(
        "process",
        module_name="tomwer.app.process",
        description="Process a given scan with a defined" "workflow",
    )
    launcher.add_command(
        "rsync",
        module_name="tomwer.app.rsync",
        description="Tool for synchronization between "
        "acquisition contained in a folder and "
        "a target folder",
    )
    launcher.add_command(
        "sample-moved",
        module_name="tomwer.app.samplemoved",
        description="Interface to check if a sample have " "moved during acquisition ",
    )
    launcher.add_command(
        "slice-stack",
        module_name="tomwer.app.slicestack",
        description="Display each reconstruction contained "
        "in the ROOT_DIR or sub-folders",
    )
    launcher.add_command(
        "sinogram-viewer",
        module_name="tomwer.app.sinogramviewer",
        description="Allows to compute on the fly sinogram " "and display them",
    )
    launcher.add_command(
        "sa-axis",
        module_name="tomwer.app.saaxis",
        description="Semi-automatic center of rotation research",
    )
    launcher.add_command(
        "sa-delta-beta",
        module_name="tomwer.app.sadeltabeta",
        description="Semi-automatic delta/beta Paganin parameter research",
    )
    launcher.add_command(
        "scan-viewer",
        module_name="tomwer.app.scanviewer",
        description="Scan viewer / browser",
    )
    launcher.add_command(
        "stop-data-listener",
        module_name="tomwer.app.stopdatalistener",
        description="Send SIGTERM or SIGKILL signal to the process occupying"
        "rpc-server port",
    )
    launcher.add_command(
        "test", module_name="tomwer.app.test_", description="Launch tomwer unittest"
    )
    status = launcher.execute(sys.argv)
    return status


if __name__ == "__main__":
    # executed when using python -m PROJECT_NAME
    status = main()
    sys.exit(status)
