"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * Adds a header to the received email
 */
class ReceiptRuleAddHeaderAction {
    constructor(props) {
        if (!/^[a-zA-Z0-9-]{1,50}$/.test(props.name)) {
            // tslint:disable:max-line-length
            throw new Error('Header `name` must be between 1 and 50 characters, inclusive, and consist of alphanumeric (a-z, A-Z, 0-9) characters and dashes only.');
            // tslint:enable:max-line-length
        }
        if (!/^[^\n\r]{0,2047}$/.test(props.value)) {
            throw new Error('Header `value` must be less than 2048 characters, and must not contain newline characters ("\r" or "\n").');
        }
        this.name = props.name;
        this.value = props.value;
    }
    render() {
        return {
            addHeaderAction: {
                headerName: this.name,
                headerValue: this.value
            }
        };
    }
}
exports.ReceiptRuleAddHeaderAction = ReceiptRuleAddHeaderAction;
/**
 * A bounce action template.
 */
class ReceiptRuleBounceActionTemplate {
    constructor(props) {
        this.message = props.message;
        this.smtpReplyCode = props.smtpReplyCode;
        this.statusCode = props.statusCode;
    }
}
exports.ReceiptRuleBounceActionTemplate = ReceiptRuleBounceActionTemplate;
ReceiptRuleBounceActionTemplate.MAILBOX_DOES_NOT_EXIST = new ReceiptRuleBounceActionTemplate({
    message: 'Mailbox does not exist',
    smtpReplyCode: '550',
    statusCode: '5.1.1'
});
ReceiptRuleBounceActionTemplate.MESSAGE_TOO_LARGE = new ReceiptRuleBounceActionTemplate({
    message: 'Message too large',
    smtpReplyCode: '552',
    statusCode: '5.3.4'
});
ReceiptRuleBounceActionTemplate.MAILBOX_FULL = new ReceiptRuleBounceActionTemplate({
    message: 'Mailbox full',
    smtpReplyCode: '552',
    statusCode: '5.2.2'
});
ReceiptRuleBounceActionTemplate.MESSAGE_CONTENT_REJECTED = new ReceiptRuleBounceActionTemplate({
    message: 'Message content rejected',
    smtpReplyCode: '500',
    statusCode: '5.6.1'
});
ReceiptRuleBounceActionTemplate.TEMPORARY_FAILURE = new ReceiptRuleBounceActionTemplate({
    message: 'Temporary failure',
    smtpReplyCode: '450',
    statusCode: '4.0.0'
});
/**
 * Rejects the received email by returning a bounce response to the sender and,
 * optionally, publishes a notification to Amazon SNS.
 */
class ReceiptRuleBounceAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        return {
            bounceAction: {
                sender: this.props.sender,
                smtpReplyCode: this.props.template.smtpReplyCode,
                message: this.props.template.message,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
                statusCode: this.props.template.statusCode
            }
        };
    }
}
exports.ReceiptRuleBounceAction = ReceiptRuleBounceAction;
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class ReceiptRuleLambdaAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.permissionsNode.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.ACCOUNT_ID
            });
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined
            }
        };
    }
}
exports.ReceiptRuleLambdaAction = ReceiptRuleLambdaAction;
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
class ReceiptRuleS3Action {
    constructor(props) {
        this.props = props;
    }
    render() {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID
                }
            }
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['km:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false'
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID
                    }
                }
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined
            }
        };
    }
}
exports.ReceiptRuleS3Action = ReceiptRuleS3Action;
/**
 * The type of email encoding to use for a SNS action.
 */
var EmailEncoding;
(function (EmailEncoding) {
    /**
     * Base 64
     */
    EmailEncoding["BASE64"] = "Base64";
    /**
     * UTF-8
     */
    EmailEncoding["UTF8"] = "UTF-8";
})(EmailEncoding = exports.EmailEncoding || (exports.EmailEncoding = {}));
/**
 * Publishes the email content within a notification to Amazon SNS.
 */
class ReceiptRuleSnsAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        return {
            snsAction: {
                encoding: this.props.encoding,
                topicArn: this.props.topic.topicArn
            }
        };
    }
}
exports.ReceiptRuleSnsAction = ReceiptRuleSnsAction;
/**
 * Terminates the evaluation of the receipt rule set and optionally publishes a
 * notification to Amazon SNS.
 */
class ReceiptRuleStopAction {
    constructor(props) {
        this.props = props;
    }
    render() {
        return {
            stopAction: {
                scope: 'RuleSet',
                topicArn: this.props && this.props.topic ? this.props.topic.topicArn : undefined
            }
        };
    }
}
exports.ReceiptRuleStopAction = ReceiptRuleStopAction;
//# sourceMappingURL=data:application/json;base64,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