"use strict";
const assert_1 = require("@aws-cdk/assert");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
// tslint:disable:max-line-length
const lib_1 = require("../lib");
module.exports = {
    'can add an add header action'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleAddHeaderAction({
                            name: 'X-My-Header',
                            value: 'value'
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        AddHeaderAction: {
                            HeaderName: 'X-My-Header',
                            HeaderValue: 'value'
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    },
    'fails when header name is invalid'(test) {
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleAddHeaderAction({
                            name: 'He@der',
                            value: 'value'
                        })
                    ]
                }
            ]
        }), /`name`/);
        test.done();
    },
    'fails when header value is invalid'(test) {
        const stack = new core_1.Stack();
        const ruleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        test.throws(() => ruleSet.addRule('Rule', {
            actions: [
                new lib_1.ReceiptRuleAddHeaderAction({
                    name: 'Header',
                    value: `va
          lu`
                })
            ]
        }), /`value`/);
        test.done();
    },
    'can add a bounce action'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleBounceAction({
                            sender: 'noreply@aws.com',
                            template: lib_1.ReceiptRuleBounceActionTemplate.MESSAGE_CONTENT_REJECTED,
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        BounceAction: {
                            Message: 'Message content rejected',
                            Sender: 'noreply@aws.com',
                            SmtpReplyCode: '500',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            },
                            StatusCode: '5.6.1',
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    },
    'can add a lambda action'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('boom'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_8_10
        });
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleLambdaAction({
                            function: fn,
                            invocationType: lib_1.LambdaInvocationType.REQUEST_RESPONSE,
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        LambdaAction: {
                            FunctionArn: {
                                'Fn::GetAtt': [
                                    'Function76856677',
                                    'Arn'
                                ]
                            },
                            InvocationType: 'RequestResponse',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            }
                        }
                    },
                ],
                Enabled: true
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: {
                'Fn::GetAtt': [
                    'Function76856677',
                    'Arn'
                ]
            },
            Principal: 'ses.amazonaws.com',
            SourceAccount: {
                Ref: 'AWS::AccountId'
            }
        }));
        test.done();
    },
    'can add a s3 action'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const kmsKey = new kms.Key(stack, 'Key');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleS3Action({
                            bucket,
                            kmsKey,
                            objectKeyPrefix: 'emails/',
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        S3Action: {
                            BucketName: {
                                Ref: 'Bucket83908E77'
                            },
                            KmsKeyArn: {
                                'Fn::GetAtt': [
                                    'Key961B73FD',
                                    'Arn'
                                ]
                            },
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            },
                            ObjectKeyPrefix: 'emails/'
                        }
                    }
                ],
                Enabled: true
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
            Bucket: {
                Ref: 'Bucket83908E77'
            },
            PolicyDocument: {
                Statement: [
                    {
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                'aws:Referer': {
                                    Ref: 'AWS::AccountId'
                                }
                            }
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: "ses.amazonaws.com"
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'Arn'
                                        ]
                                    },
                                    '/emails/*'
                                ]
                            ]
                        }
                    }
                ],
                Version: '2012-10-17'
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: [
                            'kms:Create*',
                            'kms:Describe*',
                            'kms:Enable*',
                            'kms:List*',
                            'kms:Put*',
                            'kms:Update*',
                            'kms:Revoke*',
                            'kms:Disable*',
                            'kms:Get*',
                            'kms:Delete*',
                            'kms:ScheduleKeyDeletion',
                            'kms:CancelKeyDeletion',
                            "kms:GenerateDataKey"
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':root'
                                    ]
                                ]
                            }
                        },
                        Resource: '*'
                    },
                    {
                        Action: [
                            'km:Encrypt',
                            'kms:GenerateDataKey'
                        ],
                        Condition: {
                            Null: {
                                'kms:EncryptionContext:aws:ses:rule-name': 'false',
                                'kms:EncryptionContext:aws:ses:message-id': 'false'
                            },
                            StringEquals: {
                                'kms:EncryptionContext:aws:ses:source-account': {
                                    Ref: 'AWS::AccountId'
                                }
                            }
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: "ses.amazonaws.com"
                        },
                        Resource: '*'
                    }
                ],
                Version: '2012-10-17'
            }
        }));
        test.done();
    },
    'can add a sns action'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleSnsAction({
                            encoding: lib_1.EmailEncoding.BASE64,
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        SNSAction: {
                            Encoding: 'Base64',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            }
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    },
    'can add a stop action'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    actions: [
                        new lib_1.ReceiptRuleStopAction({
                            topic
                        })
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            Rule: {
                Actions: [
                    {
                        StopAction: {
                            Scope: 'RuleSet',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E'
                            }
                        }
                    }
                ],
                Enabled: true
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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