"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserTablePrivileges = void 0;
const cdk = require("@aws-cdk/core");
const table_1 = require("../table");
const database_query_1 = require("./database-query");
const handler_name_1 = require("./database-query-provider/handler-name");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Privileges granted to a Redshift user on Redshift tables.
 *
 * This construct is located in the `private` directory to ensure that it is not exported for direct public use. This
 * means that user privileges must be managed through the `Table.grant` method or the `User.addTablePrivileges`
 * method. Thus, each `User` will have at most one `UserTablePrivileges` construct to manage its privileges. For details
 * on why this is a Good Thing, see the README, under "Granting Privileges".
 */
class UserTablePrivileges extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.privileges = props.privileges ?? [];
        new database_query_1.DatabaseQuery(this, 'Resource', {
            ...props,
            handler: handler_name_1.HandlerName.UserTablePrivileges,
            properties: {
                username: props.user.username,
                tablePrivileges: cdk.Lazy.any({
                    produce: () => {
                        const reducedPrivileges = this.privileges.reduce((privileges, { table, actions }) => {
                            const tableName = table.tableName;
                            if (!(tableName in privileges)) {
                                privileges[tableName] = [];
                            }
                            actions = actions.concat(privileges[tableName]);
                            if (actions.includes(table_1.TableAction.ALL)) {
                                actions = [table_1.TableAction.ALL];
                            }
                            if (actions.includes(table_1.TableAction.UPDATE) || actions.includes(table_1.TableAction.DELETE)) {
                                actions.push(table_1.TableAction.SELECT);
                            }
                            privileges[tableName] = Array.from(new Set(actions));
                            return privileges;
                        }, {});
                        const serializedPrivileges = Object.entries(reducedPrivileges).map(([tableName, actions]) => ({
                            tableName: tableName,
                            actions: actions.map(action => table_1.TableAction[action]),
                        }));
                        return serializedPrivileges;
                    },
                }),
            },
        });
    }
    /**
     * Grant this user additional privileges.
     */
    addPrivileges(table, ...actions) {
        this.privileges.push({ table, actions });
    }
}
exports.UserTablePrivileges = UserTablePrivileges;
//# sourceMappingURL=data:application/json;base64,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