"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = exports.MetricType = exports.JobState = exports.WorkerType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const _1 = require(".");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * (experimental) The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 *
 * @experimental
 */
class WorkerType {
    constructor(name) {
        this.name = name;
    }
    /**
     * (experimental) Custom worker type.
     *
     * @param workerType custom worker type.
     * @experimental
     */
    static of(workerType) {
        return new WorkerType(workerType);
    }
}
exports.WorkerType = WorkerType;
_a = JSII_RTTI_SYMBOL_1;
WorkerType[_a] = { fqn: "@aws-cdk/aws-glue-alpha.WorkerType", version: "2.0.0-alpha.9" };
/**
 * (experimental) Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 *
 * @experimental
 */
WorkerType.STANDARD = new WorkerType('Standard');
/**
 * (experimental) Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker.
 *
 * Suitable for memory-intensive jobs.
 *
 * @experimental
 */
WorkerType.G_1X = new WorkerType('G.1X');
/**
 * (experimental) Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker.
 *
 * Suitable for memory-intensive jobs.
 *
 * @experimental
 */
WorkerType.G_2X = new WorkerType('G.2X');
/**
 * (experimental) Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 * @experimental
 */
var JobState;
(function (JobState) {
    JobState["SUCCEEDED"] = "SUCCEEDED";
    JobState["FAILED"] = "FAILED";
    JobState["TIMEOUT"] = "TIMEOUT";
    JobState["STARTING"] = "STARTING";
    JobState["RUNNING"] = "RUNNING";
    JobState["STOPPING"] = "STOPPING";
    JobState["STOPPED"] = "STOPPED";
})(JobState = exports.JobState || (exports.JobState = {}));
/**
 * (experimental) The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 * @experimental
 */
var MetricType;
(function (MetricType) {
    MetricType["GAUGE"] = "gauge";
    MetricType["COUNT"] = "count";
})(MetricType = exports.MetricType || (exports.MetricType = {}));
class JobBase extends cdk.Resource {
    /**
     * (experimental) Create a CloudWatch Event Rule for this Glue Job when it's in a given state.
     *
     * @param id construct id.
     * @param options event options.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.glue'],
            detailType: ['Glue Job State Change', 'Glue Job Run Status'],
            detail: {
                jobName: [this.jobName],
            },
        });
        return rule;
    }
    /**
     * (experimental) Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     * @experimental
     */
    onStateChange(id, jobState, options = {}) {
        const rule = this.onEvent(id, {
            description: `Rule triggered when Glue job ${this.jobName} is in ${jobState} state`,
            ...options,
        });
        rule.addEventPattern({
            detail: {
                state: [jobState],
            },
        });
        return rule;
    }
    /**
     * (experimental) Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onSuccess(id, options = {}) {
        return this.onStateChange(id, JobState.SUCCEEDED, options);
    }
    /**
     * (experimental) Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onFailure(id, options = {}) {
        return this.onStateChange(id, JobState.FAILED, options);
    }
    /**
     * (experimental) Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onTimeout(id, options = {}) {
        return this.onStateChange(id, JobState.TIMEOUT, options);
    }
    /**
     * (experimental) Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     * @experimental
     */
    metric(metricName, type, props) {
        return new cloudwatch.Metric({
            metricName,
            namespace: 'Glue',
            dimensionsMap: {
                JobName: this.jobName,
                JobRunId: 'ALL',
                Type: type,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     *
     * @experimental
     */
    metricSuccess(props) {
        return metricRule(this.metricJobStateRule('SuccessMetricRule', JobState.SUCCEEDED), props);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     *
     * @experimental
     */
    metricFailure(props) {
        return metricRule(this.metricJobStateRule('FailureMetricRule', JobState.FAILED), props);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     *
     * @experimental
     */
    metricTimeout(props) {
        return metricRule(this.metricJobStateRule('TimeoutMetricRule', JobState.TIMEOUT), props);
    }
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    metricJobStateRule(id, jobState) {
        var _c;
        return (_c = this.node.tryFindChild(id)) !== null && _c !== void 0 ? _c : this.onStateChange(id, jobState);
    }
}
/**
 * (experimental) A Glue Job.
 *
 * @experimental
 */
class Job extends JobBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.jobName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobProps(props);
        const executable = props.executable.bind();
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')],
        });
        this.grantPrincipal = this.role;
        const sparkUI = ((_d = props.sparkUI) === null || _d === void 0 ? void 0 : _d.enabled) ? this.setupSparkUI(executable, this.role, props.sparkUI) : undefined;
        this.sparkUILoggingLocation = sparkUI === null || sparkUI === void 0 ? void 0 : sparkUI.location;
        const continuousLoggingArgs = ((_e = props.continuousLogging) === null || _e === void 0 ? void 0 : _e.enabled) ? this.setupContinuousLogging(this.role, props.continuousLogging) : {};
        const profilingMetricsArgs = props.enableProfilingMetrics ? { '--enable-metrics': '' } : {};
        const defaultArguments = {
            ...this.executableArguments(executable),
            ...continuousLoggingArgs,
            ...profilingMetricsArgs,
            ...sparkUI === null || sparkUI === void 0 ? void 0 : sparkUI.args,
            ...this.checkNoReservedArgs(props.defaultArguments),
        };
        const jobResource = new aws_glue_1.CfnJob(this, 'Resource', {
            name: props.jobName,
            description: props.description,
            role: this.role.roleArn,
            command: {
                name: executable.type.name,
                scriptLocation: this.codeS3ObjectUrl(executable.script),
                pythonVersion: executable.pythonVersion,
            },
            glueVersion: executable.glueVersion.name,
            workerType: (_f = props.workerType) === null || _f === void 0 ? void 0 : _f.name,
            numberOfWorkers: props.workerCount,
            maxCapacity: props.maxCapacity,
            maxRetries: props.maxRetries,
            executionProperty: props.maxConcurrentRuns ? { maxConcurrentRuns: props.maxConcurrentRuns } : undefined,
            notificationProperty: props.notifyDelayAfter ? { notifyDelayAfter: props.notifyDelayAfter.toMinutes() } : undefined,
            timeout: (_g = props.timeout) === null || _g === void 0 ? void 0 : _g.toMinutes(),
            connections: props.connections ? { connections: props.connections.map((connection) => connection.connectionName) } : undefined,
            securityConfiguration: (_h = props.securityConfiguration) === null || _h === void 0 ? void 0 : _h.securityConfigurationName,
            tags: props.tags,
            defaultArguments,
        });
        const resourceName = this.getResourceNameAttribute(jobResource.ref);
        this.jobArn = jobArn(this, resourceName);
        this.jobName = resourceName;
    }
    /**
     * (experimental) Creates a Glue Job.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromJobAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobAttributes(attrs);
        class Import extends JobBase {
            constructor() {
                var _c;
                super(...arguments);
                this.jobName = attrs.jobName;
                this.jobArn = jobArn(scope, attrs.jobName);
                this.grantPrincipal = (_c = attrs.role) !== null && _c !== void 0 ? _c : new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    checkNoReservedArgs(defaultArguments) {
        if (defaultArguments) {
            const reservedArgs = new Set(['--conf', '--debug', '--mode', '--JOB_NAME']);
            Object.keys(defaultArguments).forEach((arg) => {
                if (reservedArgs.has(arg)) {
                    throw new Error(`The ${arg} argument is reserved by Glue. Don't set it`);
                }
            });
        }
        return defaultArguments;
    }
    executableArguments(config) {
        var _c;
        const args = {};
        args['--job-language'] = config.language;
        if (config.className) {
            args['--class'] = config.className;
        }
        if (config.extraJars && ((_c = config.extraJars) === null || _c === void 0 ? void 0 : _c.length) > 0) {
            args['--extra-jars'] = config.extraJars.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraPythonFiles && config.extraPythonFiles.length > 0) {
            args['--extra-py-files'] = config.extraPythonFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraFiles && config.extraFiles.length > 0) {
            args['--extra-files'] = config.extraFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraJarsFirst) {
            args['--user-jars-first'] = 'true';
        }
        return args;
    }
    setupSparkUI(executable, role, props) {
        var _c;
        if (_1.JobType.PYTHON_SHELL === executable.type) {
            throw new Error('Spark UI is not available for JobType.PYTHON_SHELL jobs');
        }
        const bucket = (_c = props.bucket) !== null && _c !== void 0 ? _c : new s3.Bucket(this, 'SparkUIBucket');
        bucket.grantReadWrite(role);
        const args = {
            '--enable-spark-ui': 'true',
            '--spark-event-logs-path': bucket.s3UrlForObject(props.prefix),
        };
        return {
            location: {
                prefix: props.prefix,
                bucket,
            },
            args,
        };
    }
    setupContinuousLogging(role, props) {
        var _c;
        const args = {
            '--enable-continuous-cloudwatch-log': 'true',
            '--enable-continuous-log-filter': ((_c = props.quiet) !== null && _c !== void 0 ? _c : true).toString(),
        };
        if (props.logGroup) {
            args['--continuous-log-logGroup'] = props.logGroup.logGroupName;
            props.logGroup.grantWrite(role);
        }
        if (props.logStreamPrefix) {
            args['--continuous-log-logStreamPrefix'] = props.logStreamPrefix;
        }
        if (props.conversionPattern) {
            args['--continuous-log-conversionPattern'] = props.conversionPattern;
        }
        return args;
    }
    codeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
}
exports.Job = Job;
_b = JSII_RTTI_SYMBOL_1;
Job[_b] = { fqn: "@aws-cdk/aws-glue-alpha.Job", version: "2.0.0-alpha.9" };
/**
 * Create a CloudWatch Metric that's based on Glue Job events
 * {@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types}
 * The metric has namespace = 'AWS/Events', metricName = 'TriggeredRules' and RuleName = rule.ruleName dimension.
 *
 * @param rule for use in setting RuleName dimension value
 * @param props metric properties
 */
function metricRule(rule, props) {
    return new cloudwatch.Metric({
        namespace: 'AWS/Events',
        metricName: 'TriggeredRules',
        dimensionsMap: { RuleName: rule.ruleName },
        statistic: cloudwatch.Statistic.SUM,
        ...props,
    }).attachTo(rule);
}
/**
 * Returns the job arn
 * @param scope
 * @param jobName
 */
function jobArn(scope, jobName) {
    return cdk.Stack.of(scope).formatArn({
        service: 'glue',
        resource: 'job',
        resourceName: jobName,
    });
}
//# sourceMappingURL=data:application/json;base64,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