"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const receipt_rule_1 = require("./receipt-rule");
const ses_generated_1 = require("./ses.generated");
/**
 * A new or imported receipt rule set.
 */
class ReceiptRuleSetBase extends core_1.Resource {
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id, options) {
        this.lastAddedRule = new receipt_rule_1.ReceiptRule(this, id, Object.assign({ after: this.lastAddedRule ? this.lastAddedRule : undefined, ruleSet: this }, options));
        return this.lastAddedRule;
    }
    /**
     * Adds a drop spam rule
     */
    addDropSpamRule() {
        const dropSpam = new receipt_rule_1.DropSpamReceiptRule(this, 'DropSpam', {
            ruleSet: this
        });
        this.lastAddedRule = dropSpam.rule;
    }
}
/**
 * A new receipt rule set.
 */
class ReceiptRuleSet extends ReceiptRuleSetBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.receiptRuleSetName,
        });
        const resource = new ses_generated_1.CfnReceiptRuleSet(this, 'Resource', {
            ruleSetName: this.physicalName,
        });
        this.receiptRuleSetName = resource.ref;
        if (props) {
            const rules = props.rules || [];
            rules.forEach((ruleOption, idx) => this.addRule(`Rule${idx}`, ruleOption));
            if (props.dropSpam) {
                this.addDropSpamRule();
            }
        }
    }
    /**
     * Import an exported receipt rule set.
     */
    static fromReceiptRuleSetName(scope, id, receiptRuleSetName) {
        class Import extends ReceiptRuleSetBase {
            constructor() {
                super(...arguments);
                this.receiptRuleSetName = receiptRuleSetName;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReceiptRuleSet = ReceiptRuleSet;
//# sourceMappingURL=data:application/json;base64,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