import autoscaling_api = require('@aws-cdk/aws-autoscaling-api');
import iam = require('@aws-cdk/aws-iam');
import kms = require('@aws-cdk/aws-kms');
import s3n = require('@aws-cdk/aws-s3-notifications');
import { IResource, Resource } from '@aws-cdk/cdk';
export interface IQueue extends IResource, s3n.IBucketNotificationDestination, autoscaling_api.ILifecycleHookTarget {
    /**
     * The ARN of this queue
     * @attribute
     */
    readonly queueArn: string;
    /**
     * The URL of this queue
     * @attribute
     */
    readonly queueUrl: string;
    /**
     * The name of this queue
     * @attribute
     */
    readonly queueName: string;
    /**
     * If this queue is server-side encrypted, this is the KMS encryption key.
     */
    readonly encryptionMasterKey?: kms.IEncryptionKey;
    /**
     * Export a queue
     */
    export(): QueueAttributes;
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:ChangeMessageVisibilityBatch
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:DeleteMessageBatch
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:SendMessageBatch
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param queueActions The actions to grant
     */
    grant(grantee: iam.IGrantable, ...queueActions: string[]): iam.Grant;
}
/**
 * Reference to a new or existing Amazon SQS queue
 */
export declare abstract class QueueBase extends Resource implements IQueue {
    /**
     * The ARN of this queue
     */
    abstract readonly queueArn: string;
    /**
     * The URL of this queue
     */
    abstract readonly queueUrl: string;
    /**
     * The name of this queue
     */
    abstract readonly queueName: string;
    /**
     * If this queue is server-side encrypted, this is the KMS encryption key.
     */
    abstract readonly encryptionMasterKey?: kms.IEncryptionKey;
    /**
     * Controls automatic creation of policy objects.
     *
     * Set by subclasses.
     */
    protected abstract readonly autoCreatePolicy: boolean;
    private policy?;
    /**
     * The set of S3 bucket IDs that are allowed to send notifications to this queue.
     */
    private readonly notifyingBuckets;
    /**
     * Export a queue
     */
    abstract export(): QueueAttributes;
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * Allows using SQS queues as destinations for bucket notifications.
     * Use `bucket.onEvent(event, queue)` to subscribe.
     * @param bucketArn The ARN of the notifying bucket.
     * @param bucketId A unique ID for the notifying bucket.
     */
    asBucketNotificationDestination(bucketArn: string, bucketId: string): s3n.BucketNotificationDestinationProps;
    /**
     * Allow using SQS queues as lifecycle hook targets
     */
    asLifecycleHookTarget(lifecycleHook: autoscaling_api.ILifecycleHook): autoscaling_api.LifecycleHookTargetProps;
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:ChangeMessageVisibilityBatch
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:DeleteMessageBatch
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:SendMessageBatch
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
}
/**
 * Reference to a queue
 */
export interface QueueAttributes {
    /**
     * The ARN of the queue.
     */
    readonly queueArn: string;
    /**
     * The URL of the queue.
     */
    readonly queueUrl?: string;
    /**
     * The name of the queue.
     * @default if queue name is not specified, the name will be derived from the queue ARN
     */
    readonly queueName?: string;
    /**
     * KMS encryption key, if this queue is server-side encrypted by a KMS key.
     */
    readonly keyArn?: string;
}
