"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk_1 = require("@aws-cdk/cdk");
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends cdk_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * The set of S3 bucket IDs that are allowed to send notifications to this queue.
         */
        this.notifyingBuckets = new Set();
    }
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatement(statement);
        }
    }
    /**
     * Allows using SQS queues as destinations for bucket notifications.
     * Use `bucket.onEvent(event, queue)` to subscribe.
     * @param bucketArn The ARN of the notifying bucket.
     * @param bucketId A unique ID for the notifying bucket.
     */
    asBucketNotificationDestination(bucketArn, bucketId) {
        if (!this.notifyingBuckets.has(bucketId)) {
            this.addToResourcePolicy(new iam.PolicyStatement()
                .addServicePrincipal('s3.amazonaws.com')
                .addAction('sqs:SendMessage')
                .addResource(this.queueArn)
                .addCondition('ArnLike', { 'aws:SourceArn': bucketArn }));
            // if this queue is encrypted, we need to allow S3 to read messages since that's how
            // it verifies that the notification destination configuration is valid.
            // by setting allowNoOp to false, we ensure that only custom keys that we can actually
            // control access to can be used here as described in:
            // https://docs.aws.amazon.com/AmazonS3/latest/dev/ways-to-add-notification-config-to-bucket.html
            if (this.encryptionMasterKey) {
                this.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement()
                    .addServicePrincipal('s3.amazonaws.com')
                    .addAction('kms:GenerateDataKey')
                    .addAction('kms:Decrypt')
                    .addAllResources(), /* allowNoOp */ false);
            }
            this.notifyingBuckets.add(bucketId);
        }
        return {
            arn: this.queueArn,
            type: s3n.BucketNotificationDestinationType.Queue,
            dependencies: [this.policy]
        };
    }
    /**
     * Allow using SQS queues as lifecycle hook targets
     */
    asLifecycleHookTarget(lifecycleHook) {
        this.grantSendMessages(lifecycleHook.role);
        return { notificationTargetArn: this.queueArn };
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:ChangeMessageVisibilityBatch
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:DeleteMessageBatch
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        return this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:ChangeMessageVisibilityBatch', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:DeleteMessageBatch', 'sqs:GetQueueAttributes');
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:SendMessageBatch
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        return this.grant(grantee, 'sqs:SendMessage', 'sqs:SendMessageBatch', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
//# sourceMappingURL=data:application/json;base64,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