"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk_1 = require("@aws-cdk/cdk");
const sqs = require("../lib");
const lib_1 = require("../lib");
function testGrant(action, ...expectedActions) {
    const stack = new cdk_1.Stack();
    const queue = new lib_1.Queue(stack, 'MyQueue');
    const principal = new iam.User(stack, 'User');
    action(queue, principal);
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        "PolicyDocument": {
            "Statement": [
                {
                    "Action": expectedActions,
                    "Effect": "Allow",
                    "Resource": {
                        "Fn::GetAtt": [
                            "MyQueueE6CA6235",
                            "Arn"
                        ]
                    }
                }
            ],
            "Version": "2012-10-17"
        }
    }));
}
module.exports = {
    'default properties'(test) {
        const stack = new cdk_1.Stack();
        new sqs.Queue(stack, 'Queue');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue"
                }
            }
        });
        test.done();
    },
    'with a dead letter queue'(test) {
        const stack = new cdk_1.Stack();
        const dlq = new sqs.Queue(stack, 'DLQ');
        new sqs.Queue(stack, 'Queue', { deadLetterQueue: { queue: dlq, maxReceiveCount: 3 } });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "DLQ581697C4": {
                    "Type": "AWS::SQS::Queue"
                },
                "Queue4A7E3555": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "RedrivePolicy": {
                            "deadLetterTargetArn": {
                                "Fn::GetAtt": [
                                    "DLQ581697C4",
                                    "Arn"
                                ]
                            },
                            "maxReceiveCount": 3
                        }
                    }
                }
            }
        });
        test.done();
    },
    'addToPolicy will automatically create a policy for this queue'(test) {
        const stack = new cdk_1.Stack();
        const queue = new sqs.Queue(stack, 'MyQueue');
        queue.addToResourcePolicy(new iam.PolicyStatement().addAllResources().addActions('sqs:*').addPrincipal(new iam.ArnPrincipal('arn')));
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyQueueE6CA6235": {
                    "Type": "AWS::SQS::Queue"
                },
                "MyQueuePolicy6BBEDDAC": {
                    "Type": "AWS::SQS::QueuePolicy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:*",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": "arn"
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "Queues": [
                            {
                                "Ref": "MyQueueE6CA6235"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'exporting and importing works'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const queue = new sqs.Queue(stack, 'Queue');
        // WHEN
        const ref = queue.export();
        const imports = sqs.Queue.fromQueueAttributes(stack, 'Imported', ref);
        // THEN
        // "import" returns an IQueue bound to `Fn::ImportValue`s.
        test.deepEqual(stack.node.resolve(imports.queueArn), { 'Fn::ImportValue': 'Stack:QueueQueueArn8CF496D5' });
        test.deepEqual(stack.node.resolve(imports.queueUrl), { 'Fn::ImportValue': 'Stack:QueueQueueUrlC30FF916' });
        // the exporting stack has Outputs for QueueARN and QueueURL
        const outputs = assert_1.SynthUtils.toCloudFormation(stack).Outputs;
        // tslint:disable-next-line:max-line-length
        test.deepEqual(outputs.QueueQueueArn8CF496D5, { Value: { 'Fn::GetAtt': ['Queue4A7E3555', 'Arn'] }, Export: { Name: 'Stack:QueueQueueArn8CF496D5' } });
        test.deepEqual(outputs.QueueQueueUrlC30FF916, { Value: { Ref: 'Queue4A7E3555' }, Export: { Name: 'Stack:QueueQueueUrlC30FF916' } });
        test.done();
    },
    'grants': {
        'grantConsumeMessages'(test) {
            testGrant((q, p) => q.grantConsumeMessages(p), 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:ChangeMessageVisibilityBatch', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:DeleteMessageBatch', 'sqs:GetQueueAttributes');
            test.done();
        },
        'grantSendMessages'(test) {
            testGrant((q, p) => q.grantSendMessages(p), 'sqs:SendMessage', 'sqs:SendMessageBatch', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grantPurge'(test) {
            testGrant((q, p) => q.grantPurge(p), 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
            test.done();
        },
        'grant() is general purpose'(test) {
            testGrant((q, p) => q.grant(p, 'hello', 'world'), 'hello', 'world');
            test.done();
        },
        'grants also work on imported queues'(test) {
            const stack = new cdk_1.Stack();
            const queue = lib_1.Queue.fromQueueAttributes(stack, 'Import', {
                queueArn: 'arn:aws:sqs:us-east-1:123456789012:queue1',
                queueUrl: 'https://queue-url'
            });
            const user = new iam.User(stack, 'User');
            queue.grantPurge(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                "sqs:PurgeQueue",
                                "sqs:GetQueueAttributes",
                                "sqs:GetQueueUrl"
                            ],
                            "Effect": "Allow",
                            "Resource": "arn:aws:sqs:us-east-1:123456789012:queue1"
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        }
    },
    'queue encryption': {
        'encryptionMasterKey can be set to a custom KMS key'(test) {
            const stack = new cdk_1.Stack();
            const key = new kms.EncryptionKey(stack, 'CustomKey');
            const queue = new sqs.Queue(stack, 'Queue', { encryptionMasterKey: key });
            test.same(queue.encryptionMasterKey, key);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": { "Fn::GetAtt": ["CustomKey1E6D0D07", "Arn"] }
            }));
            test.done();
        },
        'a kms key will be allocated if encryption = kms but a master key is not specified'(test) {
            const stack = new cdk_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.Kms });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
                "KmsMasterKeyId": {
                    "Fn::GetAtt": [
                        "QueueKey39FCBAE6",
                        "Arn"
                    ]
                }
            }));
            test.done();
        },
        'it is possible to use a managed kms key'(test) {
            const stack = new cdk_1.Stack();
            new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KmsManaged });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "Queue4A7E3555": {
                        "Type": "AWS::SQS::Queue",
                        "Properties": {
                            "KmsMasterKeyId": "alias/aws/sqs"
                        }
                    }
                }
            });
            test.done();
        },
        'export should produce outputs the key arn and return import-values for these outputs': {
            'with custom key'(test) {
                const stack = new cdk_1.Stack();
                const customKey = new sqs.Queue(stack, 'QueueWithCustomKey', { encryption: sqs.QueueEncryption.Kms });
                const exportCustom = customKey.export();
                test.deepEqual(stack.node.resolve(exportCustom), {
                    queueArn: { 'Fn::ImportValue': 'Stack:QueueWithCustomKeyQueueArnD326BB9B' },
                    queueUrl: { 'Fn::ImportValue': 'Stack:QueueWithCustomKeyQueueUrlF07DDC70' },
                    keyArn: { 'Fn::ImportValue': 'Stack:QueueWithCustomKeyKeyArn537F6E42' }
                });
                test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
                    "QueueWithCustomKeyQueueArnD326BB9B": {
                        "Value": {
                            "Fn::GetAtt": [
                                "QueueWithCustomKeyB3E22087",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "Stack:QueueWithCustomKeyQueueArnD326BB9B"
                        }
                    },
                    "QueueWithCustomKeyQueueUrlF07DDC70": {
                        "Value": {
                            "Ref": "QueueWithCustomKeyB3E22087"
                        },
                        "Export": {
                            "Name": "Stack:QueueWithCustomKeyQueueUrlF07DDC70"
                        }
                    },
                    "QueueWithCustomKeyKeyArn537F6E42": {
                        "Value": {
                            "Fn::GetAtt": [
                                "QueueWithCustomKeyD80425F1",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "Stack:QueueWithCustomKeyKeyArn537F6E42"
                        }
                    }
                });
                test.done();
            },
            'with managed key'(test) {
                const stack = new cdk_1.Stack();
                const managedKey = new sqs.Queue(stack, 'QueueWithManagedKey', { encryption: sqs.QueueEncryption.KmsManaged });
                const exportManaged = managedKey.export();
                test.deepEqual(stack.node.resolve(exportManaged), {
                    queueArn: { 'Fn::ImportValue': 'Stack:QueueWithManagedKeyQueueArn8798A14E' },
                    queueUrl: { 'Fn::ImportValue': 'Stack:QueueWithManagedKeyQueueUrlD735C981' },
                    keyArn: { 'Fn::ImportValue': 'Stack:QueueWithManagedKeyKeyArn9C42A85D' }
                });
                test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
                    "QueueWithManagedKeyQueueArn8798A14E": {
                        "Value": {
                            "Fn::GetAtt": [
                                "QueueWithManagedKeyE1B747A1",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "Stack:QueueWithManagedKeyQueueArn8798A14E"
                        }
                    },
                    "QueueWithManagedKeyQueueUrlD735C981": {
                        "Value": {
                            "Ref": "QueueWithManagedKeyE1B747A1"
                        },
                        "Export": {
                            "Name": "Stack:QueueWithManagedKeyQueueUrlD735C981"
                        }
                    },
                    "QueueWithManagedKeyKeyArn9C42A85D": {
                        "Value": "alias/aws/sqs",
                        "Export": {
                            "Name": "Stack:QueueWithManagedKeyKeyArn9C42A85D"
                        }
                    }
                });
                test.done();
            }
        }
    },
    'bucket notifications': {
        'queues can be used as destinations'(test) {
            const stack = new cdk_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue');
            const bucket = new s3.Bucket(stack, 'Bucket');
            bucket.onObjectRemoved(queue);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "sqs:SendMessage",
                            "Condition": {
                                "ArnLike": {
                                    "aws:SourceArn": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "Arn"
                                        ]
                                    }
                                }
                            },
                            "Effect": "Allow",
                            "Principal": {
                                "Service": {
                                    "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]]
                                }
                            },
                            "Resource": {
                                "Fn::GetAtt": [
                                    "Queue4A7E3555",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Queues": [
                    {
                        "Ref": "Queue4A7E3555"
                    }
                ]
            }));
            assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
                "BucketName": {
                    "Ref": "Bucket83908E77"
                },
                "NotificationConfiguration": {
                    "QueueConfigurations": [
                        {
                            "Events": [
                                "s3:ObjectRemoved:*"
                            ],
                            "QueueArn": {
                                "Fn::GetAtt": [
                                    "Queue4A7E3555",
                                    "Arn"
                                ]
                            }
                        }
                    ]
                }
            }));
            // make sure the queue policy is added as a dependency to the bucket
            // notifications resource so it will be created first.
            test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Resources.BucketNotifications8F2E257D.DependsOn, ['QueuePolicy25439813']);
            test.done();
        },
        'if the queue is encrypted with a custom kms key, the key resource policy is updated to allow s3 to read messages'(test) {
            const stack = new cdk_1.Stack();
            const bucket = new s3.Bucket(stack, 'Bucket');
            const queue = new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.Kms });
            bucket.onObjectCreated(queue);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                "KeyPolicy": {
                    "Statement": [
                        {
                            "Action": [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion"
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                "Ref": "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            "Resource": "*"
                        },
                        {
                            "Action": [
                                "kms:GenerateDataKey",
                                "kms:Decrypt"
                            ],
                            "Effect": "Allow",
                            "Principal": {
                                "Service": {
                                    "Fn::Join": ["", ["s3.", { Ref: "AWS::URLSuffix" }]]
                                }
                            },
                            "Resource": "*"
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Description": "Created by Queue"
            }));
            test.done();
        },
        'fails if trying to subscribe to a queue with managed kms encryption'(test) {
            const stack = new cdk_1.Stack();
            const queue = new sqs.Queue(stack, 'Queue', { encryption: sqs.QueueEncryption.KmsManaged });
            const bucket = new s3.Bucket(stack, 'Bucket');
            test.throws(() => bucket.onObjectRemoved(queue), 'Unable to add statement to IAM resource policy for KMS key: "alias/aws/sqs"');
            test.done();
        }
    },
    'test metrics'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const topic = new lib_1.Queue(stack, 'Queue');
        // THEN
        test.deepEqual(stack.node.resolve(topic.metricNumberOfMessagesSent()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'NumberOfMessagesSent',
            periodSec: 300,
            statistic: 'Sum'
        });
        test.deepEqual(stack.node.resolve(topic.metricSentMessageSize()), {
            dimensions: { QueueName: { 'Fn::GetAtt': ['Queue4A7E3555', 'QueueName'] } },
            namespace: 'AWS/SQS',
            metricName: 'SentMessageSize',
            periodSec: 300,
            statistic: 'Average'
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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