"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PlaceIndex = exports.IntendedUse = exports.DataSource = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_location_1 = require("aws-cdk-lib/aws-location");
/**
 * Data source for a place index
 */
var DataSource;
(function (DataSource) {
    /**
     * Esri
     *
     * @see https://docs.aws.amazon.com/location/latest/developerguide/esri.html
     */
    DataSource["ESRI"] = "Esri";
    /**
     * HERE
     *
     * @see https://docs.aws.amazon.com/location/latest/developerguide/HERE.html
     */
    DataSource["HERE"] = "Here";
})(DataSource = exports.DataSource || (exports.DataSource = {}));
/**
 * Intend use for the results of an operation
 */
var IntendedUse;
(function (IntendedUse) {
    /**
     * The results won't be stored
     */
    IntendedUse["SINGLE_USE"] = "SingleUse";
    /**
     * The result can be cached or stored in a database
     */
    IntendedUse["STORAGE"] = "Storage";
})(IntendedUse = exports.IntendedUse || (exports.IntendedUse = {}));
class PlaceIndexBase extends aws_cdk_lib_1.Resource {
    /**
     * Grant the given principal identity permissions to perform the actions on this place index.
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.placeIndexArn],
        });
    }
    /**
     * Grant the given identity permissions to search using this index
     */
    grantSearch(grantee) {
        return this.grant(grantee, 'geo:SearchPlaceIndexForPosition', 'geo:SearchPlaceIndexForSuggestions', 'geo:SearchPlaceIndexForText');
    }
}
/**
 * A Place Index
 *
 * @see https://docs.aws.amazon.com/location/latest/developerguide/places-concepts.html
 */
class PlaceIndex extends PlaceIndexBase {
    /**
     * Use an existing place index by name
     */
    static fromPlaceIndexName(scope, id, placeIndexName) {
        const placeIndexArn = aws_cdk_lib_1.Stack.of(scope).formatArn({
            service: 'geo',
            resource: 'place-index',
            resourceName: placeIndexName,
        });
        return PlaceIndex.fromPlaceIndexArn(scope, id, placeIndexArn);
    }
    /**
     * Use an existing place index by ARN
     */
    static fromPlaceIndexArn(scope, id, placeIndexArn) {
        const parsedArn = aws_cdk_lib_1.Stack.of(scope).splitArn(placeIndexArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Place Index Arn ${placeIndexArn} does not have a resource name.`);
        }
        class Import extends PlaceIndexBase {
            constructor() {
                super(...arguments);
                this.placeIndexName = parsedArn.resourceName;
                this.placeIndexArn = placeIndexArn;
            }
        }
        return new Import(scope, id, {
            account: parsedArn.account,
            region: parsedArn.region,
        });
    }
    constructor(scope, id, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_location_alpha_PlaceIndexProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PlaceIndex);
            }
            throw error;
        }
        if (props.placeIndexName && !aws_cdk_lib_1.Token.isUnresolved(props.placeIndexName) && !/^[-.\w]{1,100}$/.test(props.placeIndexName)) {
            throw new Error(`Invalid place index name. The place index name must be between 1 and 100 characters and contain only alphanumeric characters, hyphens, periods and underscores. Received: ${props.placeIndexName}`);
        }
        super(scope, id, {
            physicalName: props.placeIndexName ?? aws_cdk_lib_1.Lazy.string({ produce: () => this.generateUniqueId() }),
        });
        const placeIndex = new aws_location_1.CfnPlaceIndex(this, 'Resource', {
            indexName: this.physicalName,
            dataSource: props.dataSource ?? DataSource.ESRI,
            dataSourceConfiguration: props.intendedUse
                ? { intendedUse: props.intendedUse }
                : undefined,
            description: props.description,
        });
        this.placeIndexName = placeIndex.ref;
        this.placeIndexArn = placeIndex.attrArn;
        this.placeIndexCreateTime = placeIndex.attrCreateTime;
        this.placeIndexUpdateTime = placeIndex.attrUpdateTime;
    }
    generateUniqueId() {
        const name = aws_cdk_lib_1.Names.uniqueId(this);
        if (name.length > 100) {
            return name.substring(0, 50) + name.substring(name.length - 50);
        }
        return name;
    }
}
_a = JSII_RTTI_SYMBOL_1;
PlaceIndex[_a] = { fqn: "@aws-cdk/aws-location-alpha.PlaceIndex", version: "2.73.0-alpha.0" };
exports.PlaceIndex = PlaceIndex;
//# sourceMappingURL=data:application/json;base64,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