import * as iam from '@aws-cdk/aws-iam';
import { Duration, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IAliasRecordTarget } from './alias-record-target';
import { IHostedZone } from './hosted-zone-ref';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * A record set.
 *
 * @stability stable
 */
export interface IRecordSet extends IResource {
    /**
     * The domain name of the record.
     *
     * @stability stable
     */
    readonly domainName: string;
}
/**
 * The record type.
 *
 * @stability stable
 */
export declare enum RecordType {
    /**
     * route traffic to a resource, such as a web server, using an IPv4 address in dotted decimal notation.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AFormat
     * @stability stable
     */
    A = "A",
    /**
     * route traffic to a resource, such as a web server, using an IPv6 address in colon-separated hexadecimal format.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AAAAFormat
     * @stability stable
     */
    AAAA = "AAAA",
    /**
     * A CAA record specifies which certificate authorities (CAs) are allowed to issue certificates for a domain or subdomain.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CAAFormat
     * @stability stable
     */
    CAA = "CAA",
    /**
     * A CNAME record maps DNS queries for the name of the current record, such as acme.example.com, to another domain (example.com or example.net) or subdomain (acme.example.com or zenith.example.org).
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CNAMEFormat
     * @stability stable
     */
    CNAME = "CNAME",
    /**
     * An MX record specifies the names of your mail servers and, if you have two or more mail servers, the priority order.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#MXFormat
     * @stability stable
     */
    MX = "MX",
    /**
     * A Name Authority Pointer (NAPTR) is a type of record that is used by Dynamic Delegation Discovery System (DDDS) applications to convert one value to another or to replace one value with another.
     *
     * For example, one common use is to convert phone numbers into SIP URIs.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NAPTRFormat
     * @stability stable
     */
    NAPTR = "NAPTR",
    /**
     * An NS record identifies the name servers for the hosted zone.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NSFormat
     * @stability stable
     */
    NS = "NS",
    /**
     * A PTR record maps an IP address to the corresponding domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#PTRFormat
     * @stability stable
     */
    PTR = "PTR",
    /**
     * A start of authority (SOA) record provides information about a domain and the corresponding Amazon Route 53 hosted zone.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SOAFormat
     * @stability stable
     */
    SOA = "SOA",
    /**
     * SPF records were formerly used to verify the identity of the sender of email messages.
     *
     * Instead of an SPF record, we recommend that you create a TXT record that contains the applicable value.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SPFFormat
     * @stability stable
     */
    SPF = "SPF",
    /**
     * An SRV record Value element consists of four space-separated values.
     *
     * The first three values are
     * decimal numbers representing priority, weight, and port. The fourth value is a domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SRVFormat
     * @stability stable
     */
    SRV = "SRV",
    /**
     * A TXT record contains one or more strings that are enclosed in double quotation marks (").
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#TXTFormat
     * @stability stable
     */
    TXT = "TXT"
}
/**
 * Options for a RecordSet.
 *
 * @stability stable
 */
export interface RecordSetOptions {
    /**
     * The hosted zone in which to define the new record.
     *
     * @stability stable
     */
    readonly zone: IHostedZone;
    /**
     * The domain name for this record.
     *
     * @default zone root
     * @stability stable
     */
    readonly recordName?: string;
    /**
     * The resource record cache time to live (TTL).
     *
     * @default Duration.minutes(30)
     * @stability stable
     */
    readonly ttl?: Duration;
    /**
     * A comment to add on the record.
     *
     * @default no comment
     * @stability stable
     */
    readonly comment?: string;
}
/**
 * Type union for a record that accepts multiple types of target.
 *
 * @stability stable
 */
export declare class RecordTarget {
    readonly values?: string[] | undefined;
    readonly aliasTarget?: IAliasRecordTarget | undefined;
    /**
     * Use string values as target.
     *
     * @stability stable
     */
    static fromValues(...values: string[]): RecordTarget;
    /**
     * Use an alias as target.
     *
     * @stability stable
     */
    static fromAlias(aliasTarget: IAliasRecordTarget): RecordTarget;
    /**
     * Use ip addresses as target.
     *
     * @stability stable
     */
    static fromIpAddresses(...ipAddresses: string[]): RecordTarget;
    /**
     * @param values correspond with the chosen record type (e.g. for 'A' Type, specify one or more IP addresses).
     * @param aliasTarget alias for targets such as CloudFront distribution to route traffic to.
     * @stability stable
     */
    protected constructor(values?: string[] | undefined, aliasTarget?: IAliasRecordTarget | undefined);
}
/**
 * Construction properties for a RecordSet.
 *
 * @stability stable
 */
export interface RecordSetProps extends RecordSetOptions {
    /**
     * The record type.
     *
     * @stability stable
     */
    readonly recordType: RecordType;
    /**
     * The target for this record, either `RecordTarget.fromValues()` or `RecordTarget.fromAlias()`.
     *
     * @stability stable
     */
    readonly target: RecordTarget;
}
/**
 * A record set.
 *
 * @stability stable
 */
export declare class RecordSet extends Resource implements IRecordSet {
    /**
     * The domain name of the record.
     *
     * @stability stable
     */
    readonly domainName: string;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: RecordSetProps);
}
/**
 * (deprecated) Target for a DNS A Record.
 *
 * @deprecated Use RecordTarget
 */
export declare class AddressRecordTarget extends RecordTarget {
}
/**
 * Construction properties for a ARecord.
 *
 * @stability stable
 */
export interface ARecordProps extends RecordSetOptions {
    /**
     * The target.
     *
     * @stability stable
     */
    readonly target: RecordTarget;
}
/**
 * A DNS A record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class ARecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ARecordProps);
}
/**
 * Construction properties for a AaaaRecord.
 *
 * @stability stable
 */
export interface AaaaRecordProps extends RecordSetOptions {
    /**
     * The target.
     *
     * @stability stable
     */
    readonly target: RecordTarget;
}
/**
 * A DNS AAAA record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class AaaaRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: AaaaRecordProps);
}
/**
 * Construction properties for a CnameRecord.
 *
 * @stability stable
 */
export interface CnameRecordProps extends RecordSetOptions {
    /**
     * The domain name.
     *
     * @stability stable
     */
    readonly domainName: string;
}
/**
 * A DNS CNAME record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class CnameRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: CnameRecordProps);
}
/**
 * Construction properties for a TxtRecord.
 *
 * @stability stable
 */
export interface TxtRecordProps extends RecordSetOptions {
    /**
     * The text values.
     *
     * @stability stable
     */
    readonly values: string[];
}
/**
 * A DNS TXT record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class TxtRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: TxtRecordProps);
}
/**
 * Properties for a SRV record value.
 *
 * @stability stable
 */
export interface SrvRecordValue {
    /**
     * The priority.
     *
     * @stability stable
     */
    readonly priority: number;
    /**
     * The weight.
     *
     * @stability stable
     */
    readonly weight: number;
    /**
     * The port.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * The server host name.
     *
     * @stability stable
     */
    readonly hostName: string;
}
/**
 * Construction properties for a SrvRecord.
 *
 * @stability stable
 */
export interface SrvRecordProps extends RecordSetOptions {
    /**
     * The values.
     *
     * @stability stable
     */
    readonly values: SrvRecordValue[];
}
/**
 * A DNS SRV record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class SrvRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SrvRecordProps);
}
/**
 * The CAA tag.
 *
 * @stability stable
 */
export declare enum CaaTag {
    /**
     * Explicity authorizes a single certificate authority to issue a certificate (any type) for the hostname.
     *
     * @stability stable
     */
    ISSUE = "issue",
    /**
     * Explicity authorizes a single certificate authority to issue a wildcard certificate (and only wildcard) for the hostname.
     *
     * @stability stable
     */
    ISSUEWILD = "issuewild",
    /**
     * Specifies a URL to which a certificate authority may report policy violations.
     *
     * @stability stable
     */
    IODEF = "iodef"
}
/**
 * Properties for a CAA record value.
 *
 * @stability stable
 */
export interface CaaRecordValue {
    /**
     * The flag.
     *
     * @stability stable
     */
    readonly flag: number;
    /**
     * The tag.
     *
     * @stability stable
     */
    readonly tag: CaaTag;
    /**
     * The value associated with the tag.
     *
     * @stability stable
     */
    readonly value: string;
}
/**
 * Construction properties for a CaaRecord.
 *
 * @stability stable
 */
export interface CaaRecordProps extends RecordSetOptions {
    /**
     * The values.
     *
     * @stability stable
     */
    readonly values: CaaRecordValue[];
}
/**
 * A DNS CAA record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class CaaRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: CaaRecordProps);
}
/**
 * Construction properties for a CaaAmazonRecord.
 *
 * @stability stable
 */
export interface CaaAmazonRecordProps extends RecordSetOptions {
}
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class CaaAmazonRecord extends CaaRecord {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: CaaAmazonRecordProps);
}
/**
 * Properties for a MX record value.
 *
 * @stability stable
 */
export interface MxRecordValue {
    /**
     * The priority.
     *
     * @stability stable
     */
    readonly priority: number;
    /**
     * The mail server host name.
     *
     * @stability stable
     */
    readonly hostName: string;
}
/**
 * Construction properties for a MxRecord.
 *
 * @stability stable
 */
export interface MxRecordProps extends RecordSetOptions {
    /**
     * The values.
     *
     * @stability stable
     */
    readonly values: MxRecordValue[];
}
/**
 * A DNS MX record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class MxRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: MxRecordProps);
}
/**
 * Construction properties for a NSRecord.
 *
 * @stability stable
 */
export interface NsRecordProps extends RecordSetOptions {
    /**
     * The NS values.
     *
     * @stability stable
     */
    readonly values: string[];
}
/**
 * A DNS NS record.
 *
 * @stability stable
 * @resource AWS::Route53::RecordSet
 */
export declare class NsRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: NsRecordProps);
}
/**
 * Construction properties for a ZoneDelegationRecord.
 *
 * @stability stable
 */
export interface ZoneDelegationRecordProps extends RecordSetOptions {
    /**
     * The name servers to report in the delegation records.
     *
     * @stability stable
     */
    readonly nameServers: string[];
}
/**
 * A record to delegate further lookups to a different set of name servers.
 *
 * @stability stable
 */
export declare class ZoneDelegationRecord extends RecordSet {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ZoneDelegationRecordProps);
}
/**
 * Construction properties for a CrossAccountZoneDelegationRecord.
 *
 * @stability stable
 */
export interface CrossAccountZoneDelegationRecordProps {
    /**
     * The zone to be delegated.
     *
     * @stability stable
     */
    readonly delegatedZone: IHostedZone;
    /**
     * The hosted zone name in the parent account.
     *
     * @default - no zone name
     * @stability stable
     */
    readonly parentHostedZoneName?: string;
    /**
     * The hosted zone id in the parent account.
     *
     * @default - no zone id
     * @stability stable
     */
    readonly parentHostedZoneId?: string;
    /**
     * The delegation role in the parent account.
     *
     * @stability stable
     */
    readonly delegationRole: iam.IRole;
    /**
     * The resource record cache time to live (TTL).
     *
     * @default Duration.days(2)
     * @stability stable
     */
    readonly ttl?: Duration;
}
/**
 * A Cross Account Zone Delegation record.
 *
 * @stability stable
 */
export declare class CrossAccountZoneDelegationRecord extends CoreConstruct {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: CrossAccountZoneDelegationRecordProps);
}
