# -*- coding: utf-8 -*-
# Copyright © 2020 Contrast Security, Inc.
# See https://www.contrastsecurity.com/enduser-terms-0317a for more details.
import hashlib
import json
import os
import time
import threading
from io import open

from contrast.extern.six import PY2
from contrast.agent import scope

if PY2:
    import imp
else:
    from importlib.util import find_spec

import pip
from pip._vendor import pkg_resources

from contrast.api.dtm_pb2 import Library, ApplicationUpdate
from contrast.utils.patch_utils import get_loaded_modules

import logging

logger = logging.getLogger("contrast")


def get_active_library_names_from_pkg():
    library_names = ["contrast"]
    libraries = list(pkg_resources.working_set)
    for library in libraries:
        library_names.append(library.key)
    return library_names


class LibraryReader(object):
    def __init__(self, library_tags="", settings=None):
        self.settings = settings
        self.analysis_thread = None
        self.installed_distribution_keys = []

        self.libraries = []
        self.library_tags = (
            library_tags if isinstance(library_tags, str) else ",".join(library_tags)
        )

        self.top_level_modules = dict()
        self.unused_libraries = []
        # All packages installed in the site-packages/ directory
        self.installed_dists = LibraryReader.get_installed_distributions()

    def start_library_analysis_thread(self, daemon=True):
        self.analysis_thread = threading.Thread(target=self.read_libraries)
        self.analysis_thread.daemon = daemon

        self.analysis_thread.start()

    def join_library_analysis_thread(self):
        if self.analysis_thread:
            self.analysis_thread.join()

    def read_libraries(self):
        """
        Looks at every library installed in self.installed_dists, then calls search_dist
        on each dist that has a file top_level.txt, which is autogenerated by installation,
        specifying which files/directories are associated with the package

        :exception: IOError if top_level.txt, METADATA, or RECORD cannot be found
        :return: None
        """
        with scope.contrast_scope():
            logger.debug("Analyzing libraries... ")

            if not self.installed_dists:
                self.libraries = []
                self.unused_libraries = []
                return

            packages = []
            unused_packages = []

            all_dist_keys = [dist._key for dist in self.installed_dists]

            self.installed_distribution_keys = [
                x for x in all_dist_keys if x != CONTRAST_AGENT_DIST
            ]

            logger.debug("Found %s libraries", len(self.installed_distribution_keys))

            results = read_dists(self.installed_dists)

            for found, dist_dict in results:
                library_name = dist_dict.get("file_path", "")

                if library_name and library_name != CONTRAST_AGENT_DIST:
                    if found:
                        packages.append(dist_dict)
                    else:
                        unused_packages.append(dist_dict)

            logger.debug(
                "Used libraries: %s", [lib.get("file_path", "") for lib in packages]
            )
            logger.debug(
                "Unused libraries: %s",
                [lib.get("file_path", "") for lib in unused_packages],
            )

            libraries = [
                convert_dict_to_library(lib, self.library_tags, self.top_level_modules)
                for lib in packages
            ]
            unused_libraries = [
                convert_dict_to_library(lib, self.library_tags, self.top_level_modules)
                for lib in unused_packages
            ]

            self.libraries = libraries
            self.unused_libraries = unused_libraries

            self.send_analysis_results_appstart()

    def send_analysis_results_appstart(self):
        from contrast.agent import service_client

        if not self.libraries or not self.settings:
            return

        app_update_msg = ApplicationUpdate()

        for library in self.libraries:
            if library.used_class_count > library.class_count:
                logger.debug(
                    "WARNING! Used count  > total count for library %s",
                    library.file_path,
                )

            if library.hash_code is not None:
                app_update_msg.libraries[library.hash_code].CopyFrom(library)

        logger.debug("Sending ApplicationUpdate message with library analysis results")

        service_client.send_messages([app_update_msg])

    @staticmethod
    def get_installed_distributions():
        with scope.contrast_scope():
            pip_version = pip.__version__
            major_version = int(pip_version.split(".")[0])

            # newest versions of pip
            if major_version >= 10:  # ['10', '18', '19']
                from pip._internal.utils.misc import get_installed_distributions

                return get_installed_distributions()

            if 6 <= major_version <= 9:  # ['6', '7', '8', '9']
                # 9 and below; these releases go back until Dec 22, 2014
                return pip.get_installed_distributions()

            return []


def read_dists(installed_dists):
    results = []

    for dist in installed_dists:
        analysis_result = search_dist(dist)

        if analysis_result:
            results.append(analysis_result)

    return results


def convert_dict_to_library(lib, library_tags, top_level_modules):
    """
    DTM objects are not pickle-able for the the multiprocess library;
    we'll return a dict from search_dist then convert to DTM Library
    :param top_level_modules: save these for after the process returns
    :param lib: library dict
    :param library_tags: tags to add to each library
    :return: Library DTM object
    """

    top_level_modules[lib["file_path"]] = lib["top_level_modules"]

    current_time = int(time.time() * 1000)

    library = Library()
    library.version = lib["version"]
    library.manifest = lib["manifest"]
    library.class_count = lib["class_count"]
    library.used_class_count = lib["used_class_count"]
    library.file_path = lib["file_path"]
    library.url = lib["url"]
    library.hash_code = lib["hash_code"]
    library.external_ms = current_time
    library.internal_ms = current_time

    if library_tags:
        library.tags = library_tags

    return library


CONTRAST_AGENT_DIST = "contrast-agent"
RECORD = "RECORD"
TOP_LEVEL_TXT = "top_level.txt"

PY_SUFFIX = ".py"
SO_SUFFIX = ".so"


def _get_top_level_directories(dist):
    # some packages have multiple top level directories, so check for all of them

    top_level_dirs = []

    # file storing directory names for top level directories of package
    if dist.has_metadata(TOP_LEVEL_TXT):
        top_level_dirs = list(dist.get_metadata_lines(TOP_LEVEL_TXT))
    elif dist.has_metadata(RECORD):
        metadata_lines = [d.split(",")[0] for d in dist.get_metadata_lines(RECORD)]

        top_level_dirs = [
            x
            for x in metadata_lines
            if "/" in x and x.split(os.sep)[1] == "__init__.py"
        ]
    else:
        if isinstance(dist._provider, pip._vendor.pkg_resources.FileMetadata):
            path = "/".join([dist._provider.path, TOP_LEVEL_TXT])
        else:
            # pip._vendor.pkg_resources.PathMetadata
            path = "/".join([dist._provider.egg_info, TOP_LEVEL_TXT])

        try:
            with open("/".join([path, TOP_LEVEL_TXT]), "rb") as tl:
                top_level_dirs = tl.readlines()
        except IOError as _:
            pass

    return top_level_dirs


def search_dist(dist):
    """
    Searches directories related to dist, gathering relevant statistics and metadata for LibraryDTM.
    Then, assuming library was loaded, appends that metadata to the self.libraries in a LibraryDTM

    Created package is added to the output for the process

    :param dist: contains a pip._vendor.pkg_resources.DistInfoDistribution object
    referencing the current library being looked for
    :return: None
    """
    # __init__.py is required in order to be a package. If there are multiple subpackages
    # we want to make sure we include all __init__.py files as part of the count
    total_files = list()
    total_used_files = set()
    top_level_dirs = _get_top_level_directories(dist)
    version = manifest = url = ""

    for directory in top_level_dirs:
        package_path = "/".join([dist.location, directory])

        all_dir_files = get_all_files(package_path)
        dir_used_files = search_modules(package_path, directory)

        total_files.extend(all_dir_files)
        total_used_files.update(dir_used_files)

    file_count = len(total_files)
    used_file_count = len(total_used_files)
    # Set the package path to be the first directory mentioned in top_level.txt
    name = str(dist).split(" ")[0]
    package_found = used_file_count > 0

    if package_found:
        version, manifest, url = get_data(dist)

    library_hash = get_hash(name, version)

    # tuple of if we found it on the path and the package object
    # we only report those on the path
    result = (
        package_found,
        create_package(
            version,
            manifest,
            file_count,
            used_file_count,
            name,
            url,
            library_hash,
            top_level_dirs,
        ),
    )

    return result


def create_package(
    version, manifest, class_count, used_class_count, name, url, sha, top_level_dirs
):
    """
    Add package with appropriate metadata to the library; Reason for this is that protobuf cannot be pickled
    :param version: version of package
    :param manifest: all metadata
    :param class_count: count of files
    :param used_class_count: count of used/loaded files
    :param name: name of the library
    :param url: homepage of package
    :param sha: sha1 hash of the name(space)version
    :param top_level_dirs: names of the first level of modules
    :return: dict of package
    """
    current_time = int(time.time() * 1000)

    return {
        "version": version,
        "manifest": manifest,
        "class_count": class_count,
        "used_class_count": used_class_count,
        "file_path": name,
        "url": url,
        "hash_code": sha,
        "external_ms": current_time,
        "internal_ms": current_time,
        "tags": "",
        "top_level_modules": top_level_dirs,
    }


def search_modules(package_path, directory_name):
    """
    Searches every module to see if it is associated with the current package being looked at.
    If it is, update used file count.

    :param directory_name: name of top level directory (i.e package or subpackage name)
    :param package_path: Path to the package currently being looked for
    :return: The list of used files for this directory, package_found, name, version, manifest, and url
    name, version, manifest, and url are empty strings if the package was already found
    """
    used_files = set()
    sys_modules = get_loaded_modules().values()
    top_level_package = directory_name + "."

    for module in sys_modules:
        if (
            module.__name__
            and (module.__name__ + ".").startswith(top_level_package)
            and hasattr(module, "__file__")
            and module.__file__
        ):
            # First we need to check and see if the top-level package (the name up till the
            # first dot) of __name__ is the same as the current directory_name/module name

            # Then, process __file__ (Path to the loaded file)
            # __file__ is optional. If set, this attribute’s value must be a string. The import
            # system may opt to leave __file__ unset if it has no semantic meaning
            # (e.g. a module loaded from a database).

            # using os.path.realpath because in some environments there is a symlink to the
            # lib/python_version_here/site-packages directory. When checking module.__file__
            # the symlink path is used instead of the resolved path. Resolved path is needed
            # because that is what the package_path will point to.
            module_file = os.path.realpath(module.__file__)

            file_path = module_file if module_file.startswith(package_path) else ""
            if len(file_path) > 0:
                used_files.add(file_path)

    return used_files


def find_module_file(module_name, module_location):
    """
    If module is found, returns the module file path under the location specified
    by module_location.
    :param module_name: Name of the module
    :param module_location: Path to the module file
    """
    module_file = None

    if PY2:
        try:
            module = imp.find_module(module_name, [module_location])
            if module:
                module_file = module[1]
        except ImportError:
            pass
    else:
        try:
            # A module spec contains information used to import a module. Such information, origin/location
            # is needed here to get the location of the module found under module_location. PEP 451 contains
            # additional info on this.
            module_spec = find_spec(module_name)
            if hasattr(module_spec, "origin") and module_spec.origin:
                if module_spec.origin.startswith(module_location):
                    module_file = module_spec.origin
            elif hasattr(module_spec, "location") and module_spec.location:
                if module_spec.location.startswith(module_location):
                    module_file = module_spec.location
        except Exception as e:
            # Raises ModuleNotFoundError (new in 3.6+) instead of AttributeError if package is in fact not a package
            # (i.e. lacks a __path__ attribute).
            # This is something that shouldn't happen so we log it here
            logger.debug(
                "WARNING - Module %s at location %s is not a package: %s",
                module_name,
                module_location,
                e,
            )

    if module_file and os.path.isfile(module_file):
        return module_file

    return None


def get_all_files(package_path):
    """
    Get all .py and .so files in all directories in package_path.
    :param package_path: Path to the current package being looked or a
    path to the module file.
    If its a module, we need to get the full path to the module file
    since package_path only includes the first portion of the module name.
    :return: list of files
    """
    total_files = []

    if not os.path.isdir(package_path):
        module_name = os.path.basename(package_path)
        module_location = os.path.dirname(package_path)

        module_file = find_module_file(module_name, module_location)
        if module_file:
            total_files.append(module_file)

        return total_files

    for _, _, files in os.walk(package_path):
        for f in files:
            if f.endswith((PY_SUFFIX, SO_SUFFIX)):
                total_files.append(f)

    # check for one file in site-packages
    if os.path.exists(package_path + PY_SUFFIX):
        total_files.append(package_path + PY_SUFFIX)

    return total_files


def get_data(dist):
    """
    Given a dist, pulls name, version, manifest, and url out of the metadata
    :param dist: the distribution package whose package info is being retrieved
    :return: the package info from the metadata
    """
    version = dist.version
    manifest = dist.get_metadata(dist.PKG_INFO)

    # If the data has a .json format
    if dist.PKG_INFO == "METADATA":
        url = get_metadata(dist)

    # If the data is in PKG-INFO form
    elif dist.PKG_INFO == "PKG-INFO":
        url = get_pkg_info(dist)

    # If the metadata is not in either form
    else:
        raise Exception(dist.PKG_INFO, "Package metadata not found")

    return version, manifest, str(url)


EXTENTIONS = "extensions"
HOME = "Home"
PROJECT_URLS = "project_urls"
PYTHON_DETAILS = "python.details"


def get_metadata(dist):
    """
    Gets the library data if PKG_INFO is packaged in a json
    :param dist: the distribution package who's data is being parsed
    :return: the url of the package
    """

    try:
        metadata = json.loads(dist.get_metadata("metadata.json"))

        if PROJECT_URLS in metadata[EXTENTIONS][PYTHON_DETAILS]:
            return metadata[EXTENTIONS][PYTHON_DETAILS][PROJECT_URLS][HOME]
    except:
        pass

    try:
        metadata = json.loads(dist.get_metadata("pydist.json"))
        if PROJECT_URLS in metadata:
            return metadata[PROJECT_URLS][HOME]
    except:
        pass

    return ""


HOME_PAGE = "Home-page: "


def get_pkg_info(dist):
    """
    Gets the library data if PKG_INFO is packaged in a text file
    :param dist: the distribution package who's data is being parsed
    :return: the url of the package
    """

    # Split metadata so it can be searched
    metadata = list(dist.get_metadata_lines(dist.PKG_INFO))

    # Search metadata for homepage
    for line in metadata:
        if line.startswith(HOME_PAGE):
            return line.split(HOME_PAGE)[1]

    return ""


def get_hash(name, version):
    """
    DO NOT ALTER OR REMOVE
    """
    to_hash = name + " " + version

    return hashlib.sha1(to_hash.encode("utf-8")).hexdigest()
