from typing import Any, Dict, Callable, List, Tuple

from ..types import Update, Message, Response, CallbackQuery
from ..helpers import ApiHelper

try:
    from starlette.responses import PlainTextResponse
    from starlette.requests import Request
except: pass

import asyncio


class BaseBot:
    def __init__(self, token: str) -> None:
        self.token = token

        self._api = ApiHelper(token)

        self._update_offset = 0

        self._key_text_msgs: Dict[str, Callable[[Message], Any]] = {}
        self._key_cmd_msgs: Dict[str, Callable[[Message], Any]] = {}
        self._file_id_sticker_msgs: Dict[str, Callable[[Message], Any]] = {}
        self._global_msg_types: Dict[str, Callable[[Message], Any]] = {}

        self.question_queues: Dict[str, asyncio.Queue] = {}

        self._func_handlers: List[Tuple[ Callable[[Message], bool], Callable[[Message], Any] ]] = []

        self._cb_datas: Dict[str, Callable[[CallbackQuery]], Any] = {}
        self._cb_funcs: List[Tuple[Callable[[CallbackQuery], bool], Callable[[Message], Any]]] = []
        

    async def _process_message(self, message: Message):
        if 'all' in self._global_msg_types:
            asyncio.create_task( self._global_msg_types['all'](message) )

        if message.text:
            cmd = message.text.split()[0][1:]
            is_cmd = cmd in self._key_cmd_msgs

            ans_queue = self.question_queues.get(message.chat.id)

            if 'text' in self._global_msg_types:
                asyncio.create_task( self._global_msg_types['text'](message) )

            if message.text in self._key_text_msgs and ans_queue is None:
                asyncio.create_task( self._key_text_msgs[message.text](message) )

            if is_cmd:
                asyncio.create_task( self._key_cmd_msgs[cmd](message) )

            if not is_cmd and ans_queue != None:
                ans_queue.put_nowait(message)

        if message.sticker:
            if 'sticker' in self._global_msg_types:
                asyncio.create_task( self._global_msg_types['sticker'](message) )
            
            if (message.sticker.file_unique_id in self._file_id_sticker_msgs):
                asyncio.create_task( self._file_id_sticker_msgs[message.sticker.file_unique_id](message) )

        if message.document:
            if 'document' in self._global_msg_types:
                asyncio.create_task( self._global_msg_types['document'](message) )

        if message.photo:
            if 'photo' in self._global_msg_types:
                asyncio.create_task( self._global_msg_types['photo'](message) )

        for item in self._func_handlers:
            if item[0](message):
                asyncio.create_task( item[1](message) )

        
    async def _proceed_callback_query(self, callback_query: CallbackQuery):
        cb = self._cb_datas.get(callback_query.data)
        if cb is not None:
            asyncio.create_task(cb(callback_query))

        for cb_func in self._cb_funcs:
            if cb_func[0](callback_query):
                cb_func[1](callback_query)


    def _process_response(self, json_data: dict, cls):
        response = Response(json_data)
        
        if response.ok:
            return response.result_instance(cls)
        else:
            print(f'Error: {response.error_code} => {response.description}')


    async def _process_updates_json(self, json_data: dict):
        updates: List[Update] = self._process_response(json_data, Update)

        for update in updates:
            self._update_offset = update.update_id + 1
            
            if update.message:
                await self._process_message(update.message)
            elif update.callback_query:
                await self._proceed_callback_query(update.callback_query)
            else:
                print(f'Update attribute {update.dict()} not support')
        

    async def polling(self):
        async with self._api.session:
            while True:
                json_data = await self._api.get_updates(timeout=60, offset=self._update_offset)
                await self._process_updates_json(json_data)


    async def __call__(self, scope, receive, send):
        if scope['type'] != 'http': return

        if scope['method'] != 'POST':
            await PlainTextResponse('Method not allowed', 405)(scope, receive, send)

        if (scope['path']) != f'/{self.token}':
            await PlainTextResponse("Unknown path", 404)(scope, receive, send)

        json_data = await Request(scope, receive).json()
        await self._process_updates_json(json_data)

        await PlainTextResponse('success', 200)        