"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleMetadata = exports.PackName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const cdk_nag_1 = require("cdk-nag");
/**
 * Pack name
 *
 */
exports.PackName = "AwsPrototyping";
/**
 * Rule metadata
 *
 */
exports.RuleMetadata = [
    {
        info: "The REST API stage is not associated with AWS WAFv2 web ACL.",
        explanation: "AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAssociatedWithWAF,
    },
    {
        info: "The API does not implement authorization.",
        explanation: "In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.\n\nExample threat: An actor with a network path to an API gateway stage end-point can interact with the API method in question without authorization, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAuthorization,
    },
    {
        info: "The REST API does not have request validation enabled.",
        explanation: "The API should have basic request validation enabled. If the API is integrated with a custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWRequestValidation,
    },
    {
        info: "The Athena workgroup does not encrypt query results.",
        explanation: "Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.athena.AthenaWorkgroupEncryptedQueryResults,
    },
    {
        info: "The Auto Scaling launch configuration does not have public IP addresses disabled.",
        explanation: "If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingLaunchConfigPublicIpDisabled,
    },
    {
        info: "The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.",
        explanation: "SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.\n\nExample threat: A global internet-based actor can discover Cloud9 EC2 instances that have public IP addresses and that are exposing SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloud9.Cloud9InstanceNoIngressSystemsManager,
    },
    {
        info: "The prototypes CloudFront distribution has not been configured with geographic restrictions (GeoRestriction)",
        explanation: "Geo restriction should be enabled for the distribution in order to limit the surface area exposed to expected geographies\n\nExample threat: A global internet-based actor can discover prototype web assets that are exposed via CloudFront distributions, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionGeoRestrictions,
    },
    {
        info: "The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use that use the default CloudFront viewer certificate or use 'vip' for the SslSupportMethod are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified MinimumProtocolVersion",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
    },
    {
        info: "The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionNoOutdatedSSL,
    },
    {
        info: "The CloudFront distribution does not use an origin access identity an S3 origin.",
        explanation: "Origin access identities help with security by restricting any direct access to objects through S3 URLs.\n\nExample threat: A global internet-based actor can bypass the CloudFront distribution and associated controls (e.g. geo blocking), which may lead to direct access to static assets hosted on the S3 origin possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 origin for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionS3OriginAccessIdentity,
    },
    {
        info: "The CloudFront distribution may require integration with AWS WAF.",
        explanation: "The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionWAFIntegration,
    },
    {
        info: "The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.",
        explanation: "Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.\n\nExample threat: An actor who can view the CodeBuild environment variables can obtain the AWS Access Key and Secret Access Key, which may lead to the actor being able to do anything the AWS keys are authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectEnvVarAwsCred,
    },
    {
        info: "The CodeBuild project does not use an AWS KMS key for encryption.",
        explanation: "Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectKMSEncryptedArtifacts,
    },
    {
        info: "The CodeBuild project does not use images provided by the CodeBuild service or have a cdk-nag suppression rule explaining the need for a custom image.",
        explanation: "Explaining differences/edits to Docker images helps operators better understand system dependencies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectManagedImages,
    },
    {
        info: "The CodeBuild project has privileged mode enabled.",
        explanation: "Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectPrivilegedModeDisabled,
    },
    {
        info: "The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.",
        explanation: "Advanced security features enable the system to detect and act upon malicious sign-in attempts.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAdvancedSecurityModeEnforced,
    },
    {
        info: "The API Gateway method does not use a Cognito user pool authorizer.",
        explanation: "API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAPIGWAuthorizer,
    },
    {
        info: "The Cognito user pool is not configured to require MFA.",
        explanation: "Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolMFA,
    },
    {
        info: "The Cognito identity pool allows for unauthenticated logins and does not have a cdk-nag rule suppression with a reason.",
        explanation: "Applications do not warrant unauthenticated guest access in many cases. Metadata explaining the use case allows for transparency to operators.\n\nExample threat: A global internet-based actor who has discovered a prototype endpoint with Cognito unauthenticated logins can does not need to provide credentials to interact with the endpoint, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolNoUnauthenticatedLogins,
    },
    {
        info: "The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.",
        explanation: "Strong password policies increase system security by encouraging users to create reliable and secure passwords.\n\nExample threat: An actor who has discovered a prototype endpoint with Cognito authenticated logins can perform a dictionary or brute force attack to authenticate as an authorized Cognito user, which may lead to the actor being able to do anything the associated Cognito user is authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolStrongPasswordPolicy,
    },
    {
        info: "The DMS replication instance is public.",
        explanation: "DMS replication instances can contain sensitive information and access control is required for such accounts.\n\nExample threat: A global internet-based actor can discover DMS instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.dms.DMSReplicationNotPublic,
    },
    {
        info: "The Document DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterEncryptionAtRest,
    },
    {
        info: "The Document DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterNonDefaultPort,
    },
    {
        info: "The Document DB cluster does not have the username and password stored in Secrets Manager.",
        explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.\n\nExample threat: An actor who can view the DocumentDB configuration can obtain the username and password for the DocumentDB cluster, which may lead to the actor being able to access anything the associated DocumentDB user is authorised to do possibly impacting the confidentiality, integrity and availability of the data assets hosted on the DocumentDB cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.documentdb.DocumentDBCredentialsInSecretsManager,
    },
    {
        info: "The DAX cluster does not have server-side encryption enabled.",
        explanation: "Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.dynamodb.DAXEncrypted,
    },
    {
        info: "The EBS volume has encryption disabled.",
        explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2EBSVolumeEncrypted,
    },
    {
        info: "The EC2 instance is associated with a public IP address.",
        explanation: "Amazon EC2 instances can contain sensitive information and access control is required for such resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceNoPublicIp,
    },
    {
        info: "The EC2 instance does not have an instance profile attached.",
        explanation: "EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceProfileAttached,
    },
    {
        info: "The EC2 instance is not within a VPC.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2InstancesInVPC,
    },
    {
        info: "The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).",
        explanation: "Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.\n\nExample threat: A global internet-based actor can discover exposed services (e.g. Telnet, SSH, RDS, MySQL) using their common port numbers, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedCommonPorts,
    },
    {
        info: "The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.",
        explanation: "Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public EC2 instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedInbound,
    },
    {
        info: "The Security Group allows unrestricted SSH access.",
        explanation: "Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address to SSH or move laterally to non-public EC2 instances via SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedSSH,
    },
    {
        info: "The Security Group does not have a description.",
        explanation: "Descriptions help simplify operations and remove any opportunities for operator errors.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2SecurityGroupDescription,
    },
    {
        info: "The ECR Repository allows open access.",
        explanation: "Removing * principals in an ECR Repository helps protect against unauthorized access.\n\nExample threat: A global internet-based actor who has discovered a ECR repository can access the container images hosted within the repository, which may lead to information disclosure that aids in the intrusion activities being successful against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ecr.ECROpenAccess,
    },
    {
        info: "One or more containers in the ECS Task Definition do not have container logging enabled.",
        explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionContainerLogging,
    },
    {
        info: "The ECS Task Definition includes a container definition that directly specifies environment variables.",
        explanation: "Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionNoEnvironmentVariables,
    },
    {
        info: "The EFS does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.efs.EFSEncrypted,
    },
    {
        info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
        explanation: "EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format LogExport::<log> for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.eks.EKSClusterControlPlaneLogs,
    },
    {
        info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
        explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).\n\nExample threat: A global internet-based actor who has discovered a EKS cluster Kubernetes API server endpoint can perform reconnaissance and intrusion activities against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eks.EKSClusterNoEndpointPublicAccess,
    },
    {
        info: "The ElastiCache cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover the ElastiCache cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterInVPC,
    },
    {
        info: "The ElastiCache cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterNonDefaultPort,
    },
    {
        info: "The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.",
        explanation: "Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterEncryption,
    },
    {
        info: "The ElastiCache Redis cluster does not use Redis AUTH for user authentication.",
        explanation: "Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterRedisAuth,
    },
    {
        info: "The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.",
        explanation: "Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkEC2InstanceLogsToS3,
    },
    {
        info: "The Elastic Beanstalk environment does not have managed updates enabled.",
        explanation: "Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.\n\nExample threat: An actor with a network path to the Elastic Beanstalk environment can attempt to take advantage of a known vulnerability in a platform component used by Elastic Beanstalk, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkManagedUpdatesEnabled,
    },
    {
        info: "The Elastic Beanstalk environment is not configured to use a specific VPC.",
        explanation: "Use a non-default VPC in order to separate your environment from default resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkVPCSpecified,
    },
    {
        info: "The ALB's HTTP listeners are not configured to redirect to HTTPS.",
        explanation: "To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpToHttpsRedirection,
    },
    {
        info: "The CLB does not restrict its listeners to only the SSL and HTTPS protocols.",
        explanation: "Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBTlsHttpsListenersOnly,
    },
    {
        info: "The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.",
        explanation: "SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRAuthEC2KeyPairOrKerberos,
    },
    {
        info: "The EMR cluster does not use a security configuration with encryption in transit enabled and configured.",
        explanation: "EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMREncryptionInTransit,
    },
    {
        info: "The EMR cluster does not use a security configuration with local disk encryption enabled.",
        explanation: "Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRLocalDiskEncryption,
    },
    {
        info: "The event bus policy allows for open access.",
        explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.\n\nExample threat: A global internet-based actor who has discovered the Event Bridge event bus (e.g. Endpoint ID) can put arbitrary events onto the bus, which may lead to which could be processed by the prototype possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype',
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eventbridge.EventBusOpenAccess,
    },
    {
        info: "The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.",
        explanation: "Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueEncryptedCloudWatchLogs,
    },
    {
        info: "The Glue job does not have use a security configuration with job bookmark encryption enabled.",
        explanation: "Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueJobBookmarkEncrypted,
    },
    {
        info: "The IAM user, role, or group uses AWS managed policies.",
        explanation: "An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Policy::<policy> for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoManagedPolicies,
    },
    {
        info: "The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.",
        explanation: "Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Action::<action> for policy actions and Resource::<resource> for resources. Example: appliesTo: ['Action::s3:*'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoWildcardPermissions,
    },
    {
        info: "The IAM policy grants admin access - meaning the policy allows a principal to perform unlimited actions on any service",
        explanation: "AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.\n\nExample threat: A global internet-based actor who has successfully obtained valid keys or a session associated of the IAM Principal associated with the IAM policy can perform unlimited AWS actions on any AWS service which are exposed via the AWS API/Management Console/CLI, which may lead to broad and unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithAdminAccess,
    },
    {
        info: "The IAM policy grants full access - meaning the policy allows unlimited actions for a given AWS service",
        explanation: "Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithFullAccess,
    },
    {
        info: "The Kinesis Data Firehose delivery stream does not have server-side encryption enabled.",
        explanation: "Enabling encryption allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataFirehoseSSE,
    },
    {
        info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
        explanation: "Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamDefaultKeyWhenSSE,
    },
    {
        info: "The Kinesis Data Stream does not have server-side encryption enabled.",
        explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamSSE,
    },
    {
        info: "The KMS Symmetric key does not have automatic key rotation enabled.",
        explanation: "KMS key rotation allow a system to set a rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kms.KMSBackingKeyRotationEnabled,
    },
    {
        info: "The Lambda function permission grants public access",
        explanation: "Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.\n\nExample threat: A global internet-based actor who has discovered the Lambda function name or ARN can invoke/delete/modify the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionPublicAccessProhibited,
    },
    {
        info: "The Lambda Function URL allows for public, unauthenticated access.",
        explanation: "AWS Lambda Function URLs allow you to invoke your function via a HTTPS end-point, setting the authentication to NONE allows anyone on the internet to invoke your function.\n\nExample threat: A global internet-based actor who has discovered the Lambda Function URL can invoke the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionUrlAuth,
    },
    {
        info: "The non-container Lambda function is not configured to use the latest runtime version.",
        explanation: "Use the latest available runtime for the targeted language to avoid technical debt. Runtimes specific to a language or framework version are deprecated when the version reaches end of life. This rule only applies to non-container Lambda functions.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaLatestVersion,
    },
    {
        info: "The MediaStore container does not define a CORS policy.",
        explanation: "Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerCORSPolicy,
    },
    {
        info: "The MediaStore container does not define a container policy.",
        explanation: "Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerHasContainerPolicy,
    },
    {
        info: "The MediaStore container does not require requests to use SSL.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerSSLRequestsOnly,
    },
    {
        info: "The MSK cluster uses plaintext communication between brokers.",
        explanation: "TLS communication secures data-in-transit by encrypting the connection between brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKBrokerToBrokerTLS,
    },
    {
        info: "The MSK cluster uses plaintext communication between clients and brokers.",
        explanation: "TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKClientToBrokerTLS,
    },
    {
        info: "The Neptune DB instance does have Auto Minor Version Upgrade enabled.",
        explanation: "The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.\n\nExample threat: An actor with a network path to the Neptune cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by Neptune, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterAutomaticMinorVersionUpgrade,
    },
    {
        info: "The Neptune DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterEncryptionAtRest,
    },
    {
        info: "The Neptune DB cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterIAMAuth,
    },
    {
        info: "The OpenSearch Service domain does not only grant access via allowlisted IP addresses.",
        explanation: "Using allowlisted IP addresses helps protect the domain against unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchAllowlistedIPs,
    },
    {
        info: "The OpenSearch Service domain does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchEncryptedAtRest,
    },
    {
        info: "The OpenSearch Service domain is not provisioned inside a VPC.",
        explanation: "Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.\n\nExample threat: A global internet-based actor can discover the OpenSearch Service domain that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchInVPCOnly,
    },
    {
        info: "The OpenSearch Service domain does not have node-to-node encryption enabled.",
        explanation: "Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNodeToNodeEncryption,
    },
    {
        info: "The OpenSearch Service domain does not allow for unsigned requests or anonymous access.",
        explanation: "Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.\n\nExample threat: An actor with a network path to the OpenSearch Service domain can directly access the domain without authentication, which may lead to allowing access to data hosted within the domain possibly impacting the confidentiality, integrity and availability of the data assets hosted on the OpenSearch Service domain for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNoUnsignedOrAnonymousAccess,
    },
    {
        info: "The Quicksight data sources connection is not configured to use SSL.",
        explanation: "SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.quicksight.QuicksightSSLConnections,
    },
    {
        info: "The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.AuroraMySQLPostgresIAMAuth,
    },
    {
        info: "The RDS DB instance does not have automatic minor version upgrades enabled.",
        explanation: "Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.\n\nExample threat: An actor with a network path to the RDS cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by RDS, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSAutomaticMinorVersionUpgradeEnabled,
    },
    {
        info: "The RDS DB instance allows public access.",
        explanation: "Amazon RDS database instances can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the RDS DB instance endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSInstancePublicAccess,
    },
    {
        info: "The RDS instance or Aurora DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSNonDefaultPort,
    },
    {
        info: "The RDS DB Security Group allows for 0.0.0.0/0 inbound access.",
        explanation: "RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.\n\nExample threat: A global internet-based actor can discover RDS DB instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public RDS DB instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSRestrictedInbound,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon RDS DB instances, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSStorageEncrypted,
    },
    {
        info: "The Redshift cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterEncryptionAtRest,
    },
    {
        info: "The Redshift cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover a RedShift cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterInVPC,
    },
    {
        info: "The Redshift cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultPort,
    },
    {
        info: 'The Redshift cluster uses the default "awsuser" username.',
        explanation: 'Using a custom user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultUsername,
    },
    {
        info: "The Redshift cluster allows public access.",
        explanation: "Amazon Redshift clusters can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the Redshift cluster endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterPublicAccess,
    },
    {
        info: "The Redshift cluster does not have version upgrade enabled.",
        explanation: "Version Upgrade must enabled on the cluster in order to automatically receive upgrades during the maintenance window.\n\nExample threat: An actor with a network path to the Redshift cluster can attempt to take advantage of a known vulnerability in a component exposed by Redshift, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterVersionUpgrade,
    },
    {
        info: "The Redshift cluster does not require TLS/SSL encryption.",
        explanation: "Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftRequireTlsSSL,
    },
    {
        info: "The S3 bucket does not prohibit public access through bucket level settings.",
        explanation: "Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read or write can read or write data to or from the S3 bucket, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketLevelPublicAccessProhibited,
    },
    {
        info: "The S3 Bucket does not have server access logs enabled.",
        explanation: "Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketLoggingEnabled,
    },
    {
        info: "The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read can read data from the S3 bucket, which may lead to possibly impacting the confidentiality of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicReadProhibited,
    },
    {
        info: "The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public write can write data to, or overwrite data within the S3 bucket, which may lead to possibly impacting the integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicWriteProhibited,
    },
    {
        info: "The S3 Bucket does not have default server-side encryption enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketServerSideEncryptionEnabled,
    },
    {
        info: "The S3 Bucket or bucket policy does not require requests to use SSL/TLS.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketSSLRequestsOnly,
    },
    {
        info: "The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.",
        explanation: "An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.\n\nExample threat: A global internet-based actor who has discovered a S3 hosted website can discover prototype web assets that are hosted on the website, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3WebBucketOAIAccess,
    },
    {
        info: "The SageMaker notebook instance does not have an encrypted storage volume.",
        explanation: "Encrypting storage volumes helps protect SageMaker data-at-rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInstanceKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance is not provisioned inside a VPC.",
        explanation: "Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInVPC,
    },
    {
        info: "The SageMaker notebook does not disable direct internet access.",
        explanation: "By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookNoDirectInternetAccess,
    },
    {
        info: "The secret does not have automatic rotation scheduled.",
        explanation: "Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerRotationEnabled,
    },
    {
        info: "The SNS topic does not have KMS encryption enabled.",
        explanation: "To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS). Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSEncryptedKMS,
    },
    {
        info: "The SNS Topic does not require publishers to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns:Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSTopicSSLPublishOnly,
    },
    {
        info: "The SQS Queue does not have server-side encryption enabled.",
        explanation: "Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSE,
    },
    {
        info: "The SQS queue does not require requests to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSLRequestsOnly,
    },
    {
        info: "The Timestream database does not use a Customer Managed KMS Key for at rest encryption.",
        explanation: "All Timestream tables in a database are encrypted at rest by default using an AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an periodic basis.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.timestream.TimestreamDatabaseCustomerManagedKey,
    },
    {
        info: "The VPC's default security group allows inbound or outbound traffic.",
        explanation: "When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) that have public IP addresses and allow ingress to all internet address or move laterally to non-public VPC-attached instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCDefaultSecurityGroupClosed,
    },
    {
        info: "A Network ACL or Network ACL entry has been implemented.",
        explanation: "Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoNACLs,
    },
    {
        info: "The subnet auto-assigns public IP addresses.",
        explanation: "Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) within the subnet in question that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCSubnetAutoAssignPublicIpDisabled,
    },
];
//# sourceMappingURL=data:application/json;base64,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