"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpMethod = exports.Connection = exports.HttpParameter = exports.Authorization = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const events_generated_1 = require("./events.generated");
/**
 * Authorization type for an API Destination Connection
 */
class Authorization {
    /**
     * Use API key authorization
     *
     * API key authorization has two components: an API key name and an API key value.
     * What these are depends on the target of your connection.
     */
    static apiKey(apiKeyName, apiKeyValue) {
        return new class extends Authorization {
            _bind() {
                return {
                    authorizationType: AuthorizationType.API_KEY,
                    authParameters: {
                        ApiKeyAuthParameters: {
                            ApiKeyName: apiKeyName,
                            ApiKeyValue: apiKeyValue,
                        },
                    },
                };
            }
        }();
    }
    /**
     * Use username and password authorization
     */
    static basic(username, password) {
        return new class extends Authorization {
            _bind() {
                return {
                    authorizationType: AuthorizationType.BASIC,
                    authParameters: {
                        BasicAuthParameters: {
                            Username: username,
                            Password: password,
                        },
                    },
                };
            }
        }();
    }
    /**
     * Use OAuth authorization
     */
    static oauth(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_events_OAuthAuthorizationProps(props);
        if (![HttpMethod.POST, HttpMethod.GET, HttpMethod.PUT].includes(props.httpMethod)) {
            throw new Error('httpMethod must be one of GET, POST, PUT');
        }
        return new class extends Authorization {
            _bind() {
                return {
                    authorizationType: AuthorizationType.OAUTH_CLIENT_CREDENTIALS,
                    authParameters: {
                        OAuthParameters: {
                            AuthorizationEndpoint: props.authorizationEndpoint,
                            ClientParameters: {
                                ClientID: props.clientId,
                                ClientSecret: props.clientSecret,
                            },
                            HttpMethod: props.httpMethod,
                            OAuthHttpParameters: {
                                BodyParameters: renderHttpParameters(props.bodyParameters),
                                HeaderParameters: renderHttpParameters(props.headerParameters),
                                QueryStringParameters: renderHttpParameters(props.queryStringParameters),
                            },
                        },
                    },
                };
            }
        }();
    }
}
exports.Authorization = Authorization;
_a = JSII_RTTI_SYMBOL_1;
Authorization[_a] = { fqn: "@aws-cdk/aws-events.Authorization", version: "1.146.0" };
/**
 * An additional HTTP parameter to send along with the OAuth request
 */
class HttpParameter {
    /**
     * Make an OAuthParameter from a string value
     *
     * The value is not treated as a secret.
     */
    static fromString(value) {
        return new class extends HttpParameter {
            _render(name) {
                return {
                    Key: name,
                    Value: value,
                };
            }
        }();
    }
    /**
     * Make an OAuthParameter from a secret
     */
    static fromSecret(value) {
        return new class extends HttpParameter {
            _render(name) {
                return {
                    Key: name,
                    Value: value,
                    IsSecretValue: true,
                };
            }
        }();
    }
}
exports.HttpParameter = HttpParameter;
_b = JSII_RTTI_SYMBOL_1;
HttpParameter[_b] = { fqn: "@aws-cdk/aws-events.HttpParameter", version: "1.146.0" };
/**
 * Define an EventBridge Connection
 *
 * @resource AWS::Events::Connection
 */
class Connection extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.connectionName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_events_ConnectionProps(props);
        const authBind = props.authorization._bind();
        const invocationHttpParameters = !!props.headerParameters || !!props.queryStringParameters || !!props.bodyParameters ? {
            BodyParameters: renderHttpParameters(props.bodyParameters),
            HeaderParameters: renderHttpParameters(props.headerParameters),
            QueryStringParameters: renderHttpParameters(props.queryStringParameters),
        } : undefined;
        let connection = new events_generated_1.CfnConnection(this, 'Connection', {
            authorizationType: authBind.authorizationType,
            authParameters: {
                ...authBind.authParameters,
                InvocationHttpParameters: invocationHttpParameters,
            },
            description: props.description,
            name: this.physicalName,
        });
        this.connectionName = this.getResourceNameAttribute(connection.ref);
        this.connectionArn = connection.attrArn;
        this.connectionSecretArn = connection.attrSecretArn;
    }
    /**
     * Import an existing connection resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param connectionArn ARN of imported connection
     */
    static fromEventBusArn(scope, id, connectionArn, connectionSecretArn) {
        const parts = core_1.Stack.of(scope).parseArn(connectionArn);
        return new ImportedConnection(scope, id, {
            connectionArn: connectionArn,
            connectionName: parts.resourceName || '',
            connectionSecretArn: connectionSecretArn,
        });
    }
    /**
     * Import an existing connection resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported connection properties
     */
    static fromConnectionAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_events_ConnectionAttributes(attrs);
        return new ImportedConnection(scope, id, attrs);
    }
}
exports.Connection = Connection;
_c = JSII_RTTI_SYMBOL_1;
Connection[_c] = { fqn: "@aws-cdk/aws-events.Connection", version: "1.146.0" };
class ImportedConnection extends core_1.Resource {
    constructor(scope, id, attrs) {
        const arnParts = core_1.Stack.of(scope).parseArn(attrs.connectionArn);
        super(scope, id, {
            account: arnParts.account,
            region: arnParts.region,
        });
        this.connectionArn = attrs.connectionArn;
        this.connectionName = attrs.connectionName;
        this.connectionSecretArn = attrs.connectionSecretArn;
    }
}
/**
 * Supported HTTP operations.
 */
var HttpMethod;
(function (HttpMethod) {
    /** POST */
    HttpMethod["POST"] = "POST";
    /** GET */
    HttpMethod["GET"] = "GET";
    /** HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** PUT */
    HttpMethod["PUT"] = "PUT";
    /** PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** DELETE */
    HttpMethod["DELETE"] = "DELETE";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * Supported Authorization Types.
 */
var AuthorizationType;
(function (AuthorizationType) {
    /** API_KEY */
    AuthorizationType["API_KEY"] = "API_KEY";
    /** BASIC */
    AuthorizationType["BASIC"] = "BASIC";
    /** OAUTH_CLIENT_CREDENTIALS */
    AuthorizationType["OAUTH_CLIENT_CREDENTIALS"] = "OAUTH_CLIENT_CREDENTIALS";
})(AuthorizationType || (AuthorizationType = {}));
function renderHttpParameters(ps) {
    if (!ps || Object.keys(ps).length === 0) {
        return undefined;
    }
    return Object.entries(ps).map(([name, p]) => p._render(name));
}
//# sourceMappingURL=data:application/json;base64,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