"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const crypto = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * (experimental) Specify a test that the canary should run.
 *
 * @experimental
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * (experimental) Specify a custom test with your own code.
     *
     * @param options The configuration options.
     * @returns `Test` associated with the specified Code object
     * @experimental
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
/**
 * (experimental) Runtime options for a canary.
 *
 * @experimental
 */
class Runtime {
    /**
     * @param name The name of the runtime version.
     * @experimental
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
/**
 * (experimental) `syn-1.0` includes the following:.
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-1.0
 * @experimental
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * (experimental) `syn-nodejs-2.0` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 81.0.4044.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-2.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_0 = new Runtime('syn-nodejs-2.0');
/**
 * (experimental) Define a new Canary.
 *
 * @experimental
 */
class Canary extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.stringValue({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_b = (_a = props.artifactsBucketLocation) === null || _a === void 0 ? void 0 : _a.bucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : this.createDefaultRole((_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_e = props.artifactsBucketLocation) === null || _e === void 0 ? void 0 : _e.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_f = props.startAfterCreation) !== null && _f !== void 0 ? _f : true,
            runtimeVersion: (_h = (_g = props.runtime) === null || _g === void 0 ? void 0 : _g.name) !== null && _h !== void 0 ? _h : Runtime.SYNTHETICS_NODEJS_2_0.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_j = props.failureRetentionPeriod) === null || _j === void 0 ? void 0 : _j.toDays(),
            successRetentionPeriod: (_k = props.successRetentionPeriod) === null || _k === void 0 ? void 0 : _k.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * (experimental) Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricDuration(options) {
        return this.metric('Duration', options);
    }
    /**
     * (experimental) Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricSuccessPercent(options) {
        return this.metric('SuccessPercent', options);
    }
    /**
     * (experimental) Measure the number of failed canary runs over a given time period.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricFailed(options) {
        return this.metric('Failed', options);
    }
    /**
     * @param metricName - the name of the metric
     * @param options - configuration options for the metric
     *
     * @returns a CloudWatch metric associated with the canary.
     * @default avg over 5 minutes
     */
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            metricName,
            namespace: 'CloudWatchSynthetics',
            dimensions: { CanaryName: this.canaryName },
            statistic: 'avg',
            ...options,
        }).attachTo(this);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: ['arn:aws:logs:::*'],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _a, _b, _c;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName,
            s3Key: (_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey,
            s3ObjectVersion: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _a, _b, _c, _d;
        return {
            durationInSeconds: String(`${(_b = (_a = props.timeToLive) === null || _a === void 0 ? void 0 : _a.toSeconds()) !== null && _b !== void 0 ? _b : 0}`),
            expression: (_d = (_c = props.schedule) === null || _c === void 0 ? void 0 : _c.expressionString) !== null && _d !== void 0 ? _d : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = this.node.uniqueId.toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
}
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FuYXJ5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2FuYXJ5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLGlDQUFpQztBQUNqQyw0REFBZ0U7QUFDaEUsd0NBQXdDO0FBQ3hDLHNDQUFzQztBQUN0QyxxQ0FBcUM7QUFJckMsaUVBQW1EOzs7Ozs7QUFLbkQsTUFBYSxJQUFJO0lBMkJmOzs7OztPQUtHO0lBQ0gsWUFBb0MsSUFBVSxFQUFrQixPQUFlO1FBQTNDLFNBQUksR0FBSixJQUFJLENBQU07UUFBa0IsWUFBTyxHQUFQLE9BQU8sQ0FBUTtJQUMvRSxDQUFDOzs7Ozs7OztJQTNCTSxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQTBCO1FBQzdDLElBQUksQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssTUFBTSxDQUFDLGVBQWUsQ0FBQyxPQUFlO1FBQzVDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsMENBQTBDLE9BQU8sR0FBRyxDQUFDLENBQUM7U0FDdkU7UUFDRCxJQUFJLE9BQU8sQ0FBQyxNQUFNLEdBQUcsRUFBRSxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsbURBQW1ELE9BQU8sR0FBRyxDQUFDLENBQUM7U0FDaEY7SUFDSCxDQUFDO0NBVUY7QUFuQ0Qsb0JBbUNDOzs7Ozs7QUFvQkQsTUFBYSxPQUFPOzs7OztJQTJCbEIsWUFBbUMsSUFBWTtRQUFaLFNBQUksR0FBSixJQUFJLENBQVE7SUFDL0MsQ0FBQzs7QUE1QkgsMEJBNkJDOzs7Ozs7Ozs7Ozs7O0FBakJ3QixzQkFBYyxHQUFHLElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDOzs7Ozs7O0FBVXhDLDZCQUFxQixHQUFHLElBQUksT0FBTyxDQUFDLGdCQUFnQixDQUFDLENBQUM7Ozs7OztBQTBIL0UsTUFBYSxNQUFPLFNBQVEsR0FBRyxDQUFDLFFBQVE7Ozs7SUE2QnRDLFlBQW1CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtCOztRQUNqRSxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDakUsWUFBWSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNoQztRQUVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxVQUFVLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7Z0JBQ3JELE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUU7YUFDekMsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxlQUFlLGVBQUcsS0FBSyxDQUFDLHVCQUF1QiwwQ0FBRSxNQUFNLG1DQUFJLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDckcsVUFBVSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXO1NBQzVDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLFNBQUcsS0FBSyxDQUFDLElBQUksbUNBQUksSUFBSSxDQUFDLGlCQUFpQixPQUFDLEtBQUssQ0FBQyx1QkFBdUIsMENBQUUsTUFBTSxDQUFDLENBQUM7UUFFeEYsTUFBTSxRQUFRLEdBQWMsSUFBSSxnQ0FBUyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDMUQsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxjQUFjLE9BQUMsS0FBSyxDQUFDLHVCQUF1QiwwQ0FBRSxNQUFNLENBQUM7WUFDOUYsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ25DLHdCQUF3QixRQUFFLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksSUFBSTtZQUMxRCxjQUFjLGNBQUUsS0FBSyxDQUFDLE9BQU8sMENBQUUsSUFBSSxtQ0FBSSxPQUFPLENBQUMscUJBQXFCLENBQUMsSUFBSTtZQUN6RSxJQUFJLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDdkIsUUFBUSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDO1lBQ3BDLHNCQUFzQixRQUFFLEtBQUssQ0FBQyxzQkFBc0IsMENBQUUsTUFBTSxFQUFFO1lBQzlELHNCQUFzQixRQUFFLEtBQUssQ0FBQyxzQkFBc0IsMENBQUUsTUFBTSxFQUFFO1lBQzlELElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztTQUM3QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUM7UUFDaEMsSUFBSSxDQUFDLFdBQVcsR0FBRyxRQUFRLENBQUMsU0FBUyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNoRSxDQUFDOzs7Ozs7OztJQVNNLGNBQWMsQ0FBQyxPQUF1QjtRQUMzQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzFDLENBQUM7Ozs7Ozs7O0lBU00sb0JBQW9CLENBQUMsT0FBdUI7UUFDakQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ2hELENBQUM7Ozs7Ozs7O0lBU00sWUFBWSxDQUFDLE9BQXVCO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNLLE1BQU0sQ0FBQyxVQUFrQixFQUFFLE9BQXVCO1FBQ3hELE9BQU8sSUFBSSx1QkFBTSxDQUFDO1lBQ2hCLFVBQVU7WUFDVixTQUFTLEVBQUUsc0JBQXNCO1lBQ2pDLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQzNDLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUcsT0FBTztTQUNYLENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDcEIsQ0FBQztJQUVEOztPQUVHO0lBQ0ssaUJBQWlCLENBQUMsTUFBZTtRQUN2QywyREFBMkQ7UUFDM0QsNElBQTRJO1FBQzVJLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztZQUNwQyxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7b0JBQ2hCLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixDQUFDO2lCQUNqQyxDQUFDO2dCQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxhQUFhLENBQUMsR0FBRyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sR0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7b0JBQ2hGLE9BQU8sRUFBRSxDQUFDLGNBQWMsRUFBRSxzQkFBc0IsQ0FBQztpQkFDbEQsQ0FBQztnQkFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztvQkFDaEIsT0FBTyxFQUFFLENBQUMsMEJBQTBCLENBQUM7b0JBQ3JDLFVBQVUsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLHNCQUFzQixFQUFFLHNCQUFzQixFQUFFLEVBQUU7aUJBQ2pGLENBQUM7Z0JBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixTQUFTLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQztvQkFDL0IsT0FBTyxFQUFFLENBQUMsc0JBQXNCLEVBQUUscUJBQXFCLEVBQUUsbUJBQW1CLENBQUM7aUJBQzlFLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELGNBQWMsRUFBRTtnQkFDZCxZQUFZLEVBQUUsTUFBTTthQUNyQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNLLFVBQVUsQ0FBQyxLQUFrQjs7UUFDbkMsTUFBTSxVQUFVLEdBQUc7WUFDakIsT0FBTyxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTztZQUMzQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUM7U0FDbEQsQ0FBQztRQUNGLE9BQU87WUFDTCxPQUFPLEVBQUUsVUFBVSxDQUFDLE9BQU87WUFDM0IsTUFBTSxFQUFFLFVBQVUsQ0FBQyxVQUFVO1lBQzdCLFFBQVEsUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxVQUFVO1lBQzNDLEtBQUssUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxTQUFTO1lBQ3ZDLGVBQWUsUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxhQUFhO1NBQ3RELENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSyxjQUFjLENBQUMsS0FBa0I7O1FBQ3ZDLE9BQU87WUFDTCxpQkFBaUIsRUFBRSxNQUFNLENBQUMsR0FBRyxZQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLFNBQVMscUNBQU0sQ0FBQyxFQUFFLENBQUM7WUFDbEUsVUFBVSxjQUFFLEtBQUssQ0FBQyxRQUFRLDBDQUFFLGdCQUFnQixtQ0FBSSxpQkFBaUI7U0FDbEUsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNLLGtCQUFrQjtRQUN4QixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBQ2hFLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxFQUFFLEVBQUU7WUFDckIsT0FBTyxJQUFJLENBQUM7U0FDYjthQUFNO1lBQ0wsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDL0M7SUFDSCxDQUFDO0NBQ0Y7QUEzTEQsd0JBMkxDO0FBRUQ7Ozs7R0FJRztBQUNILFNBQVMsUUFBUSxDQUFDLElBQVk7SUFDNUIsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ25FLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDekIsQ0FBQztBQUVELE1BQU0sU0FBUyxHQUFXLGdCQUFnQixDQUFDO0FBRTNDOzs7O0dBSUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxJQUFZO0lBQ2hDLElBQUksSUFBSSxDQUFDLE1BQU0sR0FBRyxFQUFFLEVBQUU7UUFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RUFBeUUsSUFBSSxDQUFDLE1BQU0sVUFBVSxJQUFJLElBQUksQ0FBQyxDQUFDO0tBQ3pIO0lBQ0QsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDekIsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RUFBeUUsSUFBSSxJQUFJLENBQUMsQ0FBQztLQUNwRztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCB7IE1ldHJpYywgTWV0cmljT3B0aW9ucyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENvZGUgfSBmcm9tICcuL2NvZGUnO1xuaW1wb3J0IHsgU2NoZWR1bGUgfSBmcm9tICcuL3NjaGVkdWxlJztcbmltcG9ydCB7IENmbkNhbmFyeSB9IGZyb20gJy4vc3ludGhldGljcy5nZW5lcmF0ZWQnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgVGVzdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjdXN0b20ob3B0aW9uczogQ3VzdG9tVGVzdE9wdGlvbnMpOiBUZXN0IHtcbiAgICBUZXN0LnZhbGlkYXRlSGFuZGxlcihvcHRpb25zLmhhbmRsZXIpO1xuICAgIHJldHVybiBuZXcgVGVzdChvcHRpb25zLmNvZGUsIG9wdGlvbnMuaGFuZGxlcik7XG4gIH1cblxuICAvKipcbiAgICogVmVyaWZpZXMgdGhhdCB0aGUgZ2l2ZW4gaGFuZGxlciBlbmRzIGluICcuaGFuZGxlcicuIFJldHVybnMgdGhlIGhhbmRsZXIgaWYgc3VjY2Vzc2Z1bCBhbmRcbiAgICogdGhyb3dzIGFuIGVycm9yIGlmIG5vdC5cbiAgICpcbiAgICogQHBhcmFtIGhhbmRsZXIgLSB0aGUgaGFuZGxlciBnaXZlbiBieSB0aGUgdXNlclxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgdmFsaWRhdGVIYW5kbGVyKGhhbmRsZXI6IHN0cmluZykge1xuICAgIGlmICghaGFuZGxlci5lbmRzV2l0aCgnLmhhbmRsZXInKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5hcnkgSGFuZGxlciBtdXN0IGVuZCBpbiAnLmhhbmRsZXInICgke2hhbmRsZXJ9KWApO1xuICAgIH1cbiAgICBpZiAoaGFuZGxlci5sZW5ndGggPiAyMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5hcnkgSGFuZGxlciBtdXN0IGJlIGxlc3MgdGhhbiAyMSBjaGFyYWN0ZXJzICgke2hhbmRsZXJ9KWApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBUZXN0IHByb3BlcnR5XG4gICAqXG4gICAqIEBwYXJhbSBjb2RlIFRoZSBjb2RlIHRoYXQgdGhlIGNhbmFyeSBzaG91bGQgcnVuXG4gICAqIEBwYXJhbSBoYW5kbGVyIFRoZSBoYW5kbGVyIG9mIHRoZSBjYW5hcnlcbiAgICovXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IGNvZGU6IENvZGUsIHB1YmxpYyByZWFkb25seSBoYW5kbGVyOiBzdHJpbmcpIHtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbVRlc3RPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29kZTogQ29kZSxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaGFuZGxlcjogc3RyaW5nLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBSdW50aW1lIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBTWU5USEVUSUNTXzFfMCA9IG5ldyBSdW50aW1lKCdzeW4tMS4wJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNZTlRIRVRJQ1NfTk9ERUpTXzJfMCA9IG5ldyBSdW50aW1lKCdzeW4tbm9kZWpzLTIuMCcpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmcpIHtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcnRpZmFjdHNCdWNrZXRMb2NhdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIENhbmFyeVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcnRpZmFjdHNCdWNrZXRMb2NhdGlvbj86IEFydGlmYWN0c0J1Y2tldExvY2F0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRpbWVUb0xpdmU/OiBjZGsuRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc2NoZWR1bGU/OiBTY2hlZHVsZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgc3RhcnRBZnRlckNyZWF0aW9uPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdWNjZXNzUmV0ZW50aW9uUGVyaW9kPzogY2RrLkR1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZmFpbHVyZVJldGVudGlvblBlcmlvZD86IGNkay5EdXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2FuYXJ5TmFtZT86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBydW50aW1lPzogUnVudGltZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0ZXN0OiBUZXN0O1xuXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIENhbmFyeSBleHRlbmRzIGNkay5SZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY2FuYXJ5SWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNhbmFyeVN0YXRlOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY2FuYXJ5TmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGFydGlmYWN0c0J1Y2tldDogczMuSUJ1Y2tldDtcblxuICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENhbmFyeVByb3BzKSB7XG4gICAgaWYgKHByb3BzLmNhbmFyeU5hbWUgJiYgIWNkay5Ub2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuY2FuYXJ5TmFtZSkpIHtcbiAgICAgIHZhbGlkYXRlTmFtZShwcm9wcy5jYW5hcnlOYW1lKTtcbiAgICB9XG5cbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuY2FuYXJ5TmFtZSB8fCBjZGsuTGF6eS5zdHJpbmdWYWx1ZSh7XG4gICAgICAgIHByb2R1Y2U6ICgpID0+IHRoaXMuZ2VuZXJhdGVVbmlxdWVOYW1lKCksXG4gICAgICB9KSxcbiAgICB9KTtcblxuICAgIHRoaXMuYXJ0aWZhY3RzQnVja2V0ID0gcHJvcHMuYXJ0aWZhY3RzQnVja2V0TG9jYXRpb24/LmJ1Y2tldCA/PyBuZXcgczMuQnVja2V0KHRoaXMsICdBcnRpZmFjdHNCdWNrZXQnLCB7XG4gICAgICBlbmNyeXB0aW9uOiBzMy5CdWNrZXRFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICAgIH0pO1xuXG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSA/PyB0aGlzLmNyZWF0ZURlZmF1bHRSb2xlKHByb3BzLmFydGlmYWN0c0J1Y2tldExvY2F0aW9uPy5wcmVmaXgpO1xuXG4gICAgY29uc3QgcmVzb3VyY2U6IENmbkNhbmFyeSA9IG5ldyBDZm5DYW5hcnkodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYXJ0aWZhY3RTM0xvY2F0aW9uOiB0aGlzLmFydGlmYWN0c0J1Y2tldC5zM1VybEZvck9iamVjdChwcm9wcy5hcnRpZmFjdHNCdWNrZXRMb2NhdGlvbj8ucHJlZml4KSxcbiAgICAgIGV4ZWN1dGlvblJvbGVBcm46IHRoaXMucm9sZS5yb2xlQXJuLFxuICAgICAgc3RhcnRDYW5hcnlBZnRlckNyZWF0aW9uOiBwcm9wcy5zdGFydEFmdGVyQ3JlYXRpb24gPz8gdHJ1ZSxcbiAgICAgIHJ1bnRpbWVWZXJzaW9uOiBwcm9wcy5ydW50aW1lPy5uYW1lID8/IFJ1bnRpbWUuU1lOVEhFVElDU19OT0RFSlNfMl8wLm5hbWUsXG4gICAgICBuYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIHNjaGVkdWxlOiB0aGlzLmNyZWF0ZVNjaGVkdWxlKHByb3BzKSxcbiAgICAgIGZhaWx1cmVSZXRlbnRpb25QZXJpb2Q6IHByb3BzLmZhaWx1cmVSZXRlbnRpb25QZXJpb2Q/LnRvRGF5cygpLFxuICAgICAgc3VjY2Vzc1JldGVudGlvblBlcmlvZDogcHJvcHMuc3VjY2Vzc1JldGVudGlvblBlcmlvZD8udG9EYXlzKCksXG4gICAgICBjb2RlOiB0aGlzLmNyZWF0ZUNvZGUocHJvcHMpLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jYW5hcnlJZCA9IHJlc291cmNlLmF0dHJJZDtcbiAgICB0aGlzLmNhbmFyeVN0YXRlID0gcmVzb3VyY2UuYXR0clN0YXRlO1xuICAgIHRoaXMuY2FuYXJ5TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHJlc291cmNlLnJlZik7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgbWV0cmljRHVyYXRpb24ob3B0aW9ucz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnRHVyYXRpb24nLCBvcHRpb25zKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY1N1Y2Nlc3NQZXJjZW50KG9wdGlvbnM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ1N1Y2Nlc3NQZXJjZW50Jywgb3B0aW9ucyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY0ZhaWxlZChvcHRpb25zPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdGYWlsZWQnLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcGFyYW0gbWV0cmljTmFtZSAtIHRoZSBuYW1lIG9mIHRoZSBtZXRyaWNcbiAgICogQHBhcmFtIG9wdGlvbnMgLSBjb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBtZXRyaWNcbiAgICpcbiAgICogQHJldHVybnMgYSBDbG91ZFdhdGNoIG1ldHJpYyBhc3NvY2lhdGVkIHdpdGggdGhlIGNhbmFyeS5cbiAgICogQGRlZmF1bHQgYXZnIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwcml2YXRlIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIG9wdGlvbnM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBtZXRyaWNOYW1lLFxuICAgICAgbmFtZXNwYWNlOiAnQ2xvdWRXYXRjaFN5bnRoZXRpY3MnLFxuICAgICAgZGltZW5zaW9uczogeyBDYW5hcnlOYW1lOiB0aGlzLmNhbmFyeU5hbWUgfSxcbiAgICAgIHN0YXRpc3RpYzogJ2F2ZycsXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBkZWZhdWx0IHJvbGUgZm9yIHRoZSBjYW5hcnlcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlRGVmYXVsdFJvbGUocHJlZml4Pzogc3RyaW5nKTogaWFtLklSb2xlIHtcbiAgICAvLyBDcmVhdGVkIHJvbGUgd2lsbCBuZWVkIHRoZXNlIHBvbGljaWVzIHRvIHJ1biB0aGUgQ2FuYXJ5LlxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1zeW50aGV0aWNzLWNhbmFyeS5odG1sI2Nmbi1zeW50aGV0aWNzLWNhbmFyeS1leGVjdXRpb25yb2xlYXJuXG4gICAgY29uc3QgcG9saWN5ID0gbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6TGlzdEFsbE15QnVja2V0cyddLFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIHJlc291cmNlczogW3RoaXMuYXJ0aWZhY3RzQnVja2V0LmFybkZvck9iamVjdHMoYCR7cHJlZml4ID8gcHJlZml4KycvKicgOiAnKid9YCldLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6UHV0T2JqZWN0JywgJ3MzOkdldEJ1Y2tldExvY2F0aW9uJ10sXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICBhY3Rpb25zOiBbJ2Nsb3Vkd2F0Y2g6UHV0TWV0cmljRGF0YSddLFxuICAgICAgICAgIGNvbmRpdGlvbnM6IHsgU3RyaW5nRXF1YWxzOiB7ICdjbG91ZHdhdGNoOm5hbWVzcGFjZSc6ICdDbG91ZFdhdGNoU3ludGhldGljcycgfSB9LFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIHJlc291cmNlczogWydhcm46YXdzOmxvZ3M6OjoqJ10sXG4gICAgICAgICAgYWN0aW9uczogWydsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJywgJ2xvZ3M6UHV0TG9nRXZlbnRzJ10sXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIHJldHVybiBuZXcgaWFtLlJvbGUodGhpcywgJ1NlcnZpY2VSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICBjYW5hcnlQb2xpY3k6IHBvbGljeSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgY29kZSBvYmplY3QgdGFrZW4gaW4gYnkgdGhlIGNhbmFyeSByZXNvdXJjZS5cbiAgICovXG4gIHByaXZhdGUgY3JlYXRlQ29kZShwcm9wczogQ2FuYXJ5UHJvcHMpOiBDZm5DYW5hcnkuQ29kZVByb3BlcnR5IHtcbiAgICBjb25zdCBjb2RlQ29uZmlnID0ge1xuICAgICAgaGFuZGxlcjogcHJvcHMudGVzdC5oYW5kbGVyLFxuICAgICAgLi4ucHJvcHMudGVzdC5jb2RlLmJpbmQodGhpcywgcHJvcHMudGVzdC5oYW5kbGVyKSxcbiAgICB9O1xuICAgIHJldHVybiB7XG4gICAgICBoYW5kbGVyOiBjb2RlQ29uZmlnLmhhbmRsZXIsXG4gICAgICBzY3JpcHQ6IGNvZGVDb25maWcuaW5saW5lQ29kZSxcbiAgICAgIHMzQnVja2V0OiBjb2RlQ29uZmlnLnMzTG9jYXRpb24/LmJ1Y2tldE5hbWUsXG4gICAgICBzM0tleTogY29kZUNvbmZpZy5zM0xvY2F0aW9uPy5vYmplY3RLZXksXG4gICAgICBzM09iamVjdFZlcnNpb246IGNvZGVDb25maWcuczNMb2NhdGlvbj8ub2JqZWN0VmVyc2lvbixcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBjYW5hcnkgc2NoZWR1bGUgb2JqZWN0XG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZVNjaGVkdWxlKHByb3BzOiBDYW5hcnlQcm9wcyk6IENmbkNhbmFyeS5TY2hlZHVsZVByb3BlcnR5IHtcbiAgICByZXR1cm4ge1xuICAgICAgZHVyYXRpb25JblNlY29uZHM6IFN0cmluZyhgJHtwcm9wcy50aW1lVG9MaXZlPy50b1NlY29uZHMoKSA/PyAwfWApLFxuICAgICAgZXhwcmVzc2lvbjogcHJvcHMuc2NoZWR1bGU/LmV4cHJlc3Npb25TdHJpbmcgPz8gJ3JhdGUoNSBtaW51dGVzKScsXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgdW5pcXVlIG5hbWUgZm9yIHRoZSBjYW5hcnkuIFRoZSBnZW5lcmF0ZWQgbmFtZSBpcyB0aGUgcGh5c2ljYWwgSUQgb2YgdGhlIGNhbmFyeS5cbiAgICovXG4gIHByaXZhdGUgZ2VuZXJhdGVVbmlxdWVOYW1lKCk6IHN0cmluZyB7XG4gICAgY29uc3QgbmFtZSA9IHRoaXMubm9kZS51bmlxdWVJZC50b0xvd2VyQ2FzZSgpLnJlcGxhY2UoJyAnLCAnLScpO1xuICAgIGlmIChuYW1lLmxlbmd0aCA8PSAyMSkge1xuICAgICAgcmV0dXJuIG5hbWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBuYW1lLnN1YnN0cmluZygwLCAxNSkgKyBuYW1lSGFzaChuYW1lKTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBUYWtlIGEgaGFzaCBvZiB0aGUgZ2l2ZW4gbmFtZS5cbiAqXG4gKiBAcGFyYW0gbmFtZSB0aGUgbmFtZSB0byBiZSBoYXNoZWRcbiAqL1xuZnVuY3Rpb24gbmFtZUhhc2gobmFtZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgY29uc3QgbWQ1ID0gY3J5cHRvLmNyZWF0ZUhhc2goJ3NoYTI1NicpLnVwZGF0ZShuYW1lKS5kaWdlc3QoJ2hleCcpO1xuICByZXR1cm4gbWQ1LnNsaWNlKDAsIDYpO1xufVxuXG5jb25zdCBuYW1lUmVnZXg6IFJlZ0V4cCA9IC9eWzAtOWEtel9cXC1dKyQvO1xuXG4vKipcbiAqIFZlcmlmaWVzIHRoYXQgdGhlIG5hbWUgZml0cyB0aGUgcmVnZXggZXhwcmVzc2lvbjogXlswLTlhLXpfXFwtXSskLlxuICpcbiAqIEBwYXJhbSBuYW1lIC0gdGhlIGdpdmVuIG5hbWUgb2YgdGhlIGNhbmFyeVxuICovXG5mdW5jdGlvbiB2YWxpZGF0ZU5hbWUobmFtZTogc3RyaW5nKSB7XG4gIGlmIChuYW1lLmxlbmd0aCA+IDIxKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBDYW5hcnkgbmFtZSBpcyB0b28gbGFyZ2UsIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAyMSBjaGFyYWN0ZXJzLCBidXQgaXMgJHtuYW1lLmxlbmd0aH0gKGdvdCBcIiR7bmFtZX1cIilgKTtcbiAgfVxuICBpZiAoIW5hbWVSZWdleC50ZXN0KG5hbWUpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBDYW5hcnkgbmFtZSBtdXN0IGJlIGxvd2VyY2FzZSwgbnVtYmVycywgaHlwaGVucywgb3IgdW5kZXJzY29yZXMgKGdvdCBcIiR7bmFtZX1cIilgKTtcbiAgfVxufVxuIl19