# Copyright 2020 Oliver Smith
# SPDX-License-Identifier: GPL-3.0-or-later
import glob
import os
import pytest
import sys

import pmb_test  # noqa
import pmb.build.newapkbuild
import pmb.config
import pmb.config.init
import pmb.helpers.logging


@pytest.fixture
def args(tmpdir, request):
    import pmb.parse
    sys.argv = ["pmbootstrap.py", "init"]
    args = pmb.parse.arguments()
    args.log = args.work + "/log_testsuite.txt"
    pmb.helpers.logging.init(args)
    request.addfinalizer(args.logfd.close)
    return args


def test_newapkbuild(args, monkeypatch, tmpdir):
    # Fake functions
    def confirm_true(*nargs):
        return True

    def confirm_false(*nargs):
        return False

    # Preparation
    monkeypatch.setattr(pmb.helpers.cli, "confirm", confirm_false)
    pmb.build.init(args)
    args.aports = tmpdir = str(tmpdir)
    func = pmb.build.newapkbuild

    # Show the help
    func(args, "main", ["-h"])
    assert glob.glob(tmpdir + "/*") == []

    # Test package
    pkgname = "testpackage"
    func(args, "main", [pkgname])
    apkbuild_path = tmpdir + "/main/" + pkgname + "/APKBUILD"
    apkbuild = pmb.parse.apkbuild(args, apkbuild_path)
    assert apkbuild["pkgname"] == pkgname
    assert apkbuild["pkgdesc"] == ""

    # Don't overwrite
    with pytest.raises(RuntimeError) as e:
        func(args, "main", [pkgname])
    assert "Aborted" in str(e.value)

    # Overwrite
    monkeypatch.setattr(pmb.helpers.cli, "confirm", confirm_true)
    pkgdesc = "testdescription"
    func(args, "main", ["-d", pkgdesc, pkgname])
    args.cache["apkbuild"] = {}
    apkbuild = pmb.parse.apkbuild(args, apkbuild_path)
    assert apkbuild["pkgname"] == pkgname
    assert apkbuild["pkgdesc"] == pkgdesc

    # There should be no src folder
    assert not os.path.exists(tmpdir + "/main/" + pkgname + "/src")
