# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['password_policy']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.9,<1.10']

setup_kwargs = {
    'name': 'password-policy',
    'version': '0.1.6',
    'description': 'Library for representing password policies in a machine-readable format. Also supports validating passwords against policies and for estimating the strength of passwords created under a given policy.',
    'long_description': "---\ntitle: Password Policy Library API\nlayout: page\n---\n\nThis library provides a machine readable representation of password composition policies. The ultimate goal for this library is to allow websites to upload policy descriptions and then have password generators download and use these descriptions to generate compliant passwords.\n\n## Policy Descriptions\n\nPolicies are defined using the following objects `PCP`, `PCPRule`, `PCPSubsetRequirement`, and `PCPCharsetRequirement`. There is also a utility function `SimplePCP` for creating simple policies.\n\n### SimplePCP\n\n`SimplePCP` is a shortcut function for creating PCP objects. It is the same as calling `PCP([PCPRule(*args, **kwargs)])`.\n\n| Argument | Type | Default | Description |\n| --- | --- | --- | --- |\n| min_length | int | 1 | The minimum number of characters that must be in the password. |\n| max_length | int | None | The maximum number of characters allowed in the password. |\n| max_consecutive | int | None | The maximum number of times the same character can appear in a row. |\n| prohibited_strings | set[str] | None | A set of strings that must not appear in the password. |\n| require | list[int] | None | A list of charsets that must appear in the password. |\n| require_subset | PCPSubsetRequirement | None | A list of charsets for which a subset must appear in the password. |\n| charset_requirements | dict[str,PCPCharsetRequirement] | None | A mapping between charsets and additional requirements for that charset. |\n\n### PCP\n\n`PCP` is the root class for all policies. It contains the following information.\n\n| Member | Type | Default | Description |\n| --- | --- | --- | --- |\n| charsets | dict[str,str] | `DEFAULT_CHARSETS` | The set of character allowed in the password, split into one or more disjoint character sets ('charset' for short). The key is the name of the charset and the value is a string containing the characters that make up the charset. |\n| rules | list[PCPRule] | N/A | The list of rules that make up the policy. As long as one rule matches, the policy is considered valid. At least one rule is required to be set. |\n\nTwo charsets are provided by the library:\n\n* `DEFAULT_CHARSETS`—Includes lowercase letters (*lower*), uppercase letters (*upper*), digits (*digits*), and symbols (*symbols*). This uses the charsets defined in python's `string` package.\n* `ALPHABET_CHARSETS`—Same as `DEFAULT_CHARSETS` except that lowercase and uppercase are merged into a single charset (*alphabet*).\n\nThe `PCP` class also provides several utility methods:\n\n* `validate()->None`—Validates that the policy is self consistent. For example, checking that it doesn't require more characters than it allows. Raises an exception if their are issues with the policy.\n* `dumps(**kwargs) -> str`—Dumps the policy to JSON. Tries to create the most succinct representation. Passed `**kwargs` to `json.dumps`, allowing formatted output.\n* `@staticmethod loads(s: str) -> PCP`— Loads a PCP object from the provided JSON.\n\n### PCPRule\n\n`PCPRule` specifies one or more requirements passwords must meet to satisfy the rule. All requirements must be met for the rule to be satisfied. Possible requirements are,\n\n| Member | Type | Default | Description |\n| --- | --- | --- | --- |\n| min_length | int | 1 | The minimum number of characters that must be in the password. |\n| max_length | int | None | The maximum number of characters allowed in the password. |\n| max_consecutive | int | None | The maximum number of times the same character can appear in a row. |\n| prohibited_strings | set[str] | None | A set of strings that must not appear in the password. |\n| require | list[int] | None | A list of charsets that must appear in the password. |\n| require_subset | PCPSubsetRequirement | None | A list of charsets for which a subset must appear in the password. |\n| charset_requirements | dict[str,PCPCharsetRequirement] | None | A mapping between charsets and additional requirements for that charset. |\n\n### PCPSubsetRequirement\n\nDescribes a subset requirement.\n\n| Member | Type | Default |Description |\n| --- | --- | --- | --- |\n| options | set[str] | None | The list of character sets to use. If not set, the list of all charsets will be used for the options when processing the rule.\n| count | int | N/A | The minimum number of options that must be in the password. Needs to be between 1 (inclusive) and the number of options (exclusive). Must be set if a subset requirement is used. |\n\n### PCPCharsetRequirement\n\nRequirements specific to the mapped charset.\n\n| Member | Type | Default |Description |\n| --- | --- | --- | --- |\n| min_required | int | None | Minimum characters required from the specified charset. |\n| max_allowed| int | None | Maximum characters from the charset allowed. |\n| max_consecutive | int | None | Maximum number of character from this charset allowed in a row. Note, the characters don't have to be the same, just from the same charset. |\n| required_locations| list[int] | None | Which locations in the password (0-indexed, allows reverse indexing) must contain a character from this charset. |\n|prohibited_locations| list[int] | None | Which locations in the password (0-indexed, allows reverse indexing) must *not* contain a character from this charset. |\n\n## Default character sets\n\n`DEFAULT_CHARSETS` contains four character sets:\n\n* `lower`—Lowercase alphabetic letters (a–z).\n* `upper`—Uppercase alphabetic letters (A–Z).\n* `digits`—Digits (0–9).\n* `symbols`—All ASCII symbols, including space.\n\n`ALPHABET_CHARSETS` contains three character sets:\n\n* `alphabet`—Lowercase alphabetic letters (a–z) and uppercase alphabetic letters (A–Z).\n* `digits`—Digits (0–9).\n* `symbols`—All ASCII symbols, including space.\n\n## Checking Passwords\n\nThe `check_password(password: str, pcp: PCP) -> bool` method can be used to check a password against a policy. It will return `True` if the password matches at least one of the rules in the policy and `False` otherwise.\n\n## Measuring Policy Strength\n\nThis library provides two methods for estimating the strength of passwords created using the given policy:\n\n* `check_machine_strength(pcp: PCP) -> int`—Estimates the number of passwords that a password generator could generate using the given policy, assuming it generates passwords equal to the smallest required `min_length`.\n* `check_human_strength(pcp: PCP, preferences: list[str]) -> int`—Estimates the strength of passwords that humans would create using the given policy. This method assume that humans will generate passwords equal to the smallest required `min_length`. Additionally, it assumes that users will select charsets based on the provided preferences (as long as they are allowed by the policy). If not set, preferences defaults to `['lower', 'upper', 'digits', 'symbols']`.\n\nFor both methods, the return value is half of the estimated number of passwords that can be generated. This is done because on average, the generated password would be guessed after searching half the search space.\n",
    'author': 'Scott Ruoti',
    'author_email': 'ruoti@utk.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://bitbucket.org/user-lab/password-policy/src/master/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1',
}


setup(**setup_kwargs)
