# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/08_losses.ipynb (unless otherwise specified).

__all__ = ['soft_label_cross_entropy_with_logits', 'soft_label_cross_entropy']


# Cell
import torch
import torch.nn.functional as F

from htools import add_docstring


# Cell
def soft_label_cross_entropy_with_logits(y_pred, y_true, reduction='mean'):
    """Compute cross entropy with soft labels. PyTorch's built in
    multiclass cross entropy functions require us to pass in integer
    indices, which doesn't allow for soft labels which are shaped like
    a one hot encoding. FastAI's label smoothing loss uniformly divides
    uncertainty over all classes, which again does not allow us to pass
    in our own soft labels.

    Parameters
    ----------
    y_pred: torch.FloatTensor
        Logits output by the model.
        Shape (bs, num_classes).
    y_true: torch.FloatTensor
        Soft labels, where values are between 0 and 1.
        Shape (bs, num_classes).
    reduction: str
        One of ('mean', 'sum', 'none'). This determines how to reduce
        the output of the function, similar to most PyTorch
        loss functions.

    Returns
    -------
    torch.FloatTensor: If reduction is 'none', this will have shape
        (bs, ). If 'mean' or 'sum', this will be be a tensor with a
        single value (no shape).
    """
    res = (-y_true * F.log_softmax(y_pred)).sum(-1)
    if reduction == 'none': return res
    return getattr(res, reduction)(0)


# Cell
@add_docstring(soft_label_cross_entropy_with_logits)
def soft_label_cross_entropy(y_pred, y_true, reduction='mean'):
    """Same as `soft_label_cross_entropy_with_logits` but operates on
    softmax output instead of logits. The version with logits is
    recommended for numerical stability. Below is the docstring for the logits
    version. The only difference in this version is that y_pred will not be
    logits.
    """
    res = -y_true * torch.log(y_pred)
    res = torch.where(torch.isnan(res), torch.zeros_like(res), res).sum(-1)
    if reduction == 'none': return res
    return getattr(res, reduction)(0)