# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.ipynb.

# %% auto 0
__all__ = ['save_kaggle_creds', 'file_extract', 'dl_kaggle']

# %% ../nbs/00_core.ipynb 3
from pathlib import Path  # For working with file paths
import json

# %% ../nbs/00_core.ipynb 4
def save_kaggle_creds(username:str, # The Kaggle API username.
                      key:str, # The Kaggle API key.
                      overwrite=False): # Overwrite existing credentials.
    """
    Save the Kaggle API credentials.
    """
    
    if username == "":
        print("Empty username.")
        return
    if key == "":
        print("Empty key.")
        return
    
    # Set the path to the kaggle.json file
    cred_path = Path('~/.kaggle/kaggle.json').expanduser()

    # Check if the file already exists
    if not cred_path.exists() or overwrite:
        # Create the directory if it does not exist
        cred_path.parent.mkdir(exist_ok=True)
        # Save the API key to the file
        with open(cred_path, "w") as write_file: 
            json.dump({"username": username, "key": key}, write_file)
        # Set the file permissions to be readable and writable by the current user
        cred_path.chmod(0o600)
    else:
        print("Credentials already present. Set `overwrite=True` to replace them.")

# %% ../nbs/00_core.ipynb 6
# Import the `zipfile` and `tarfile` modules from the standard library
import zipfile
import tarfile

# %% ../nbs/00_core.ipynb 7
def file_extract(fname, # The path of the archive file
                 dest=None): # The path of the destination directory
    """
    Extract the contents of the given archive file to the destination directory.

    Raises:
        Exception: if the archive file format is not recognized (supported formats are gz and zip)
    """
    
    # If destination directory is not provided, use the parent directory of the archive file
    if dest is None: dest = Path(fname).parent
    
    fname = str(fname)
    
    # Extract the archive file based on its format
    if fname.endswith('gz'):
        tarfile.open(fname, 'r:gz').extractall(dest)
    elif fname.endswith('zip'):
        zipfile.ZipFile(fname).extractall(dest)
    else:
        # Raise an exception if the archive format is not recognized
        raise Exception(f'Unrecognized archive: {fname}')

# %% ../nbs/00_core.ipynb 8
# Import the API module from the kaggle package
# from kaggle import api

# %% ../nbs/00_core.ipynb 9
def dl_kaggle(kaggle_dataset, # The Kaggle dataset id in the format `'<username>/<dataset_name>'`.
              archive_path, # The path to save the archive file.
              dataset_path, # The path to save the extracted dataset.
              delete_archive=True): # Delete the archive after extraction.
    """
    Download the Kaggle dataset and extract it to the specified directory. 
    """
    
    # Import the API module from the kaggle package
    from kaggle import api
    
    # If the dataset does not exist in the specified directory, download and extract it.
    if not dataset_path.exists():
        api.dataset_download_cli(kaggle_dataset, path=archive_path.parent)

        file_extract(fname=archive_path, dest=dataset_path)

        # Delete the archive if specified
        if delete_archive: archive_path.unlink()
    else:
        # Inform the user that the dataset already exists in the specified directory.
        print("Dataset already downloaded")
