# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dash_charts']

package_data = \
{'': ['*'], 'dash_charts': ['assets/*']}

install_requires = \
['Palettable>=3.3.0',
 'astor>=0.8.1',
 'attrs-strict>=0.2.2',
 'beautifulsoup4>=4.10.0',
 'calcipy>=0.11.0',
 'cerberus>=1.3.4',
 'dash-bootstrap-components>=1.0.0',
 'dash-extensions>=0.0.55',
 'dash-tabulator>=0.4.2',
 'dash[testing]>=2.0.0',
 'dataset>=1.5.2',
 'dominate>=2.6.0',
 'implements>=0.3.0',
 'jsonpickle>=2.1.0',
 'lxml>=4.7.1',
 'markdown>=3.3.6',
 'numpy>=1.22.2',
 'pandas>=1.3.0',
 'psutil>=5.9.0',
 'python-box>=5.4.1',
 'scipy>=1.6.1',
 'tqdm>=4.62.3']

setup_kwargs = {
    'name': 'dash-charts',
    'version': '0.1.1',
    'description': 'Python package for Plotly/Dash apps with support for multi-page, modules, and new charts such as Pareto with an Object Orient Approach',
    'long_description': '# dash_charts\n\nPython package for [Plotly/Dash](https://dash.plot.ly/) apps with support for multi-page, modules, and new charts such as Pareto with an Object Orient Approach\n\nIncludes base classes for building a custom chart or application, new charts such as a Pareto, and base classes for tabbed or multi-page applications. See full documentation at [https://kyleking.me/dash_charts/](https://kyleking.me/dash_charts/). ( TODO: Currently not online )\n\n<!-- TOC -->\n\n- [dash_charts](#dash_charts)\n    - [Nov2020-Mar2021 Updates](#nov2020-mar2021-updates)\n    - [Quick Start](#quick-start)\n        - [1. Install](#1-install)\n        - [2. Example Code](#2-example-code)\n        - [3. Resulting Pareto Chart](#3-resulting-pareto-chart)\n        - [4. Additional Notes](#4-additional-notes)\n    - [Design Principles](#design-principles)\n    - [Local Development](#local-development)\n    - [Example Charts and Tables](#example-charts-and-tables)\n        - [Pareto Chart](#pareto-chart)\n        - [Gantt Chart](#gantt-chart)\n        - [Time Vis Chart](#time-vis-chart)\n        - [Rolling Mean and STD Chart](#rolling-mean-and-std-chart)\n        - [Fitted Chart](#fitted-chart)\n        - [Real Time SQL Demo](#real-time-sql-demo)\n        - [Coordinate Chart](#coordinate-chart)\n        - [Marginal Chart](#marginal-chart)\n        - [Data Table Module](#data-table-module)\n        - [Upload Module](#upload-module)\n    - [Applications](#applications)\n        - [Tabbed Application](#tabbed-application)\n        - [Multi-Page Application](#multi-page-application)\n        - [PX Generic Application](#px-generic-application)\n    - [Other](#other)\n        - [Cache](#cache)\n        - [Static HTML Generation](#static-html-generation)\n    - [Coverage](#coverage)\n    - [External Links](#external-links)\n    - [Installation](#installation)\n    - [Usage](#usage)\n    - [Roadmap](#roadmap)\n    - [Contributing](#contributing)\n    - [License](#license)\n\n<!-- /TOC -->\n\n## Nov2020-Mar2021 Updates\n\n<!-- FIXME: Keep updates up to date! -->\nI am in the process of implementing breaking changes for a `0.1` version. The major change will be to refactor the various chart constructors to match the plotly express initialization arguments\n\nI plan on reaching >75% test coverage, adding type annotations, and improving the documentation as well\n\nI considered using a builder pattern, but the plotly.express approach can be implemented through an attributes class that accepts unhandled keyword arguments to the plots\n\n```py\n@attr.s(auto..)\nclass ChartSettings(BaseSettings):\n\n    x_label: str\n    y_label: str\n    etc: int\n\n\nclass SomeChart:\n    def __init__(self, **kwargs):\n        self.something = AttrClass(**kwargs)\n```\n\n~~Alternatively, the builder pattern might be useful to create new charts with slightly different views.~~ Would require serializing the class in order to copy rather than modify in memory\n\nCurrently blocked by pending changes to `calcipy` and `calcipy-template` to how doit tasks are configured\n\n## Quick Start\n\n### 1. Install\n\nWith Poetry install `dash_charts` with: `poetry add dash_charts --git https://github.com/KyleKing/dash_charts.git#main`\n\n### 2. Example Code\n\n<!-- CODE:tests/examples/readme.py -->\n\n```py\n"""Example Dash Application."""\n\nfrom typing import Optional\n\nimport dash\nfrom dash import html\nimport plotly.express as px\nfrom box import Box\nfrom implements import implements\n\nfrom dash_charts.pareto_chart import ParetoChart\nfrom dash_charts.utils_app import AppBase, AppInterface\nfrom dash_charts.utils_fig import min_graph\n\n# FIXME: the change to use Box/_ID needs to be implemented in the examples. This is causing failures in the test cases\n#   Still pending if this is the preferred approach\n_ID = Box({\n    \'chart\': \'pareto\',\n})\n"""Default App IDs."""\n\n\n@implements(AppInterface)\nclass ParetoDemo(AppBase):\n    """Example creating a simple Pareto chart."""\n\n    def __init__(self, app: Optional[dash.Dash] = None) -> None:\n        """Initialize app and initial data members. Should be inherited in child class and called with super().\n\n        Args:\n            app: Dash instance. If None, will create standalone app. Otherwise, will be part of existing app\n\n        """\n        self.name = \'Car Share Pareto Demo\'\n        self.data_raw = None\n        self.chart_main = None\n        self._id = _ID\n\n        super().__init__(app=app)\n\n    def generate_data(self) -> None:\n        """Format the car share data from plotly express for the Pareto. Called by parent class."""\n        self.data_raw = (px.data.carshare()\n                         .rename(columns={\'peak_hour\': \'category\', \'car_hours\': \'value\'}))\n        self.data_raw[\'category\'] = [f\'H:{cat:02}\' for cat in self.data_raw[\'category\']]\n\n    def create_elements(self) -> None:\n        """Initialize the charts, tables, and other Dash elements."""\n        self.chart_main = ParetoChart(title=\'Car Share Pareto\', xlabel=\'Peak Hours\', ylabel=\'Car Hours\')\n\n    def return_layout(self) -> dict:\n        """Return Dash application layout.\n\n        Returns:\n            dict: Dash HTML object\n\n        """\n        return html.Div([\n            html.Div([min_graph(\n                id=self._il[self._id.chart],\n                figure=self.chart_main.create_figure(df_raw=self.data_raw),\n            )]),\n        ])\n\n    def create_callbacks(self) -> None:\n        """Register the callbacks."""\n        pass  # Override base class. Not necessary for this example\n\n\nif __name__ == \'__main__\':\n    app = ParetoDemo()\n    app.create()\n    app.run(debug=True)\nelse:\n    app = instance()\n    app.create()\n    FLASK_HANDLE = app.get_server()\n\n```\n\n<!-- /CODE:tests/examples/readme.py -->\n\n### 3. Resulting Pareto Chart\n\n![MinimumExampleScreenCapture](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/pareto_readme.png)\n\n### 4. Additional Notes\n\n<!-- FIXME: Document use of the `calcipy_template` instead of manual configuration -->\n\nTO get the most out of the tools, you may want to add `calcipy` and add the below snippets to your `pyproject.toml` file\n\n```toml\n[tool.poetry.dev-dependencies.dash]\nextras = [ "testing",]\nversion = "*, ^1.19"\n\n[tool.poetry.dependencies.calcipy]\ngit = "https://github.com/kyleking/calcipy.git"\nbranch = "dev/development"\nrev = "01635ea"  # Always pin to a commit\ndevelop = true  # Optional: will reinstall each time\n\n[tool.poetry.dev-dependencies.calcipy]\ngit = "https://github.com/kyleking/calcipy.git"\nbranch = "dev/development"\nextras = [ "dev", "lint", "test", "commitizen_legacy"]\n```\n\nYou will also want at minimum the [dodo.py](https://raw.githubusercontent.com/KyleKing/dash_charts/blob/main/dodo.py) and folder structure from [dash_charts](https://github.com/KyleKing/dash_charts)\n\n## Design Principles\n\nI wanted to show that an object oriented programming approach to Dash projects could be really powerful for improving code reuse and improving readability in large projects. Below are a couple of design principles to keep that I try to keep in mind when implementing this package.\n\n- Try to encapsulate all application logic (callbacks, ids, etc.) in application classes or modules\n- For components that are reused between application, create a `module`. The layout, state, and callbacks can all be delegated to the module rather than cluttering the main application\'s class and allowing for better code reuse\n- Move any methods that do not require state (`self`) into standalone functions\n- If you can separate Dash (ids/layout/Callbacks) and Plotly (figures/tables) code, you can generate static HTML or PNGs more easily. Sometimes static output is useful when programmatically generating views rather than tweaking the interactive inputs each time.\n\nOverall, I find that this package really improves my Dash code and I hope others find it useful. Please feel free to submit PRs or open issues if you have any input. See [./DESIGN.md](https://github.com/KyleKing/dash_charts/blob/main/DESIGN.md) for additional information on design decisions and package architecture\n\n## Local Development\n\nInitial commands to clone code from Github, create Python virtual environment, and run first example application\n\n```sh\ngit clone https://github.com/KyleKing/dash_charts.git\ncd dash_charts\npoetry install\npoetry shell\npython tests/examples/ex_app_px.py\n```\n\nOther useful scripts for testing, documentation, and more:\n\n```sh\npoetry run ptw -- -m "not INTERACTIVE"\npoetry run doit run test\npoetry run doit\n```\n\n## Example Charts and Tables\n\nBelow are screenshots and links to the example code for each chart or table included in dash_charts\n\n### Pareto Chart\n\nCreate Pareto charts in Dash. Handles ordering the category, calculating the cumulative percentage, and configuring both YAxis.\n\nSee sample code in [tests/examples/ex_pareto_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_pareto_chart.py). Screenshot below:\n\n![ex_pareto_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_pareto_chart.png)\n\n### Gantt Chart\n\nCreate simple Gantt charts in Dash. Doesn\'t support more advanced features such as dependencies and resource assignment, but effectively shows tasks, progress, and projects in a clear way. You can toggle the different projects on/off and use zoom like a normal Plotly chart.\n\nSee sample code in [tests/examples/ex_gantt_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_gantt_chart.py). Screenshot below:\n\n![ex_gantt_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_gantt_chart.png)\n\n### Time Vis Chart\n\nCreate a TimeVis chart to plot general time visualizations (based on [daattali/timevis](https://github.com/daattali/timevis) from R). This chart differs from a Gantt chart by showing events that repeat in the same row.\n\nSee sample code in [tests/examples/ex_time_vis_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_time_vis_chart.py). Screenshot below:\n\n![ex_time_vis_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_time_vis_chart.png)\n\n### Rolling Mean and STD Chart\n\nEasily chart the rolling mean and standard deviation for a given scatter data set. Demonstrates using chart annotations.\n\nSee sample code in [tests/examples/ex_rolling_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_rolling_chart.py). Screenshot below:\n\n![ex_rolling_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_rolling_chart.png)\n\n### Fitted Chart\n\nFit arbitrary scatter data with a specified equation. On hover will show the rounded fit parameters. Useful for experimenting with new data.\n\nSee sample code in [tests/examples/ex_fitted_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_fitted_chart.py). Screenshot below:\n\n![ex_fitted_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_fitted_chart.png)\n\n### Real Time SQL Demo\n\nExample using a Rolling Chart to show real time data from a SQLite database. For the demo, a background thread populates the database as the Dash app runs in a separate thread.\n\nSee sample code in [tests/examples/ex_sqlite_realtime.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_sqlite_realtime.py). Screenshot below:\n\n![ex_sqlite_realtime.gif](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_sqlite_realtime.gif)\n\n### Coordinate Chart\n\nChart a discrete data set on a 2D plane with color for intensity. Below examples show how to use the `YearGrid()`, `MonthGrid()`, and `CircleGrid()` classes\n\nSee sample code in [tests/examples/ex_coordinate_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_coordinate_chart.py). Screenshot below:\n\n![ex_coordinate_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_coordinate_chart.png)\n\n### Marginal Chart\n\nExample creating a new chart from utils_fig.MarginalChart\n\nSee sample code in [tests/examples/ex_marginal_chart.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_marginal_chart.py). Screenshot below:\n\n![ex_marginal_chart.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_marginal_chart.png)\n\n### Data Table Module\n\nDisplay Dash data table from dataframe\n\nSee sample code in [tests/examples/ex_datatable.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_datatable.py). Screenshot below:\n\n![ex_datatable.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_datatable.png)\n\n### Upload Module\n\nUpload module for user-selected CSV, JSON, or SQLite files.\n\nSee sample code in [tests/examples/ex_upload.py](https://ra/www.gubusem/KyleKing/dash_charts/blob/main/tests/examples/ex_upload.py). Screenshot below:\n\n![ex_upload.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_upload.png)\n\n## Applications\n\nEvery app derives from `AppBase()` so that each tab or page can be run independently or part of a more complicated application\n\n### Tabbed Application\n\nUse the `AppWithTabs()` base class to quickly build applications with tabbed navigation. You can set tabs to the top/bottom/left/right, to be compact or not, etc.\n\nSee sample code in [tests/examples/ex_tabs.py](https://ra/www.gubusem/KyleKing/dash_charts/blob/main/tests/examples/ex_tabs.py). Screenshot below:\n\n![ex_tabs.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_tabs.png)\n\n### Multi-Page Application\n\nUse the `AppMultiPage()` base class to quickly build applications with tabbed navigation. You can set tabs to the top/bottom/left/right, to be compact or not, etc.\n\nSee sample code in [tests/examples/ex_multi_page.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_multi_page.py). Screenshot below:\n\n![ex_multi_page.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_multi_page.png)\n\n### PX Generic Application\n\nTo facilitate general data analysis from a JSON, CSV, or SQLite file, dash_charts includes apps for each px plot type and a tabbed app to bring them all together.\n\nSee sample code in [tests/examples/ex_app_px.py](https://ra/www.gubusem/KyleKing/dash_charts/blob/main/tests/examples/ex_app_px.py). Screenshot below:\n\n![ex_app_px.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_app_px.png)\n\n## Other\n\nOther notable components in `dash_charts`\n\n### Cache\n\nUtilities for utilizing a local cache file system for storing JSON data. Particularly useful if scraping or making many API calls to cache the responses locally.\n\nSee sample code in the relevant test file [tests/test_utils_json_cache.py](https:/raw./usercontentgithub.com/KyleKing/dash_charts/blob/main/tests/test_utils_json_cache.py)\n\n### Static HTML Generation\n\nutilities for creating static HTML output with all of the CustomCharts above or user-created. Also includes utilities for writing tables, code, markdown, and more to be added.\n\nSee sample code in [tests/examples/ex_utils_static.py](https://www.github.com/KyleKing/dash_charts/blob/main/tests/examples/ex_utils_static.py). Screenshot below:\n\n![ex_utils_static.png](https://raw.githubusercontent.com/KyleKing/dash_charts/main/.images/ex_utils_static.png)\n\n## Coverage\n\nLatest coverage table\n\n<!-- COVERAGE -->\n\n| File | Statements | Missing | Excluded | Coverage |\n| --: | --: | --: | --: | --: |\n| `dash_charts/__init__.py` | 2 | 0 | 0 | 100.0% |\n| `dash_charts/app_px.py` | 130 | 11 | 0 | 91.5% |\n| `dash_charts/components.py` | 13 | 0 | 0 | 100.0% |\n| `dash_charts/coordinate_chart.py` | 102 | 1 | 6 | 99.0% |\n| `dash_charts/custom_colorscales.py` | 3 | 0 | 0 | 100.0% |\n| `dash_charts/datatable.py` | 79 | 25 | 0 | 68.4% |\n| `dash_charts/equations.py` | 11 | 0 | 0 | 100.0% |\n| `dash_charts/gantt_chart.py` | 54 | 0 | 0 | 100.0% |\n| `dash_charts/modules_datatable.py` | 101 | 11 | 0 | 89.1% |\n| `dash_charts/modules_upload.py` | 132 | 60 | 0 | 54.5% |\n| `dash_charts/pareto_chart.py` | 43 | 0 | 2 | 100.0% |\n| `dash_charts/scatter_line_charts.py` | 45 | 0 | 3 | 100.0% |\n| `dash_charts/time_vis_chart.py` | 61 | 0 | 0 | 100.0% |\n| `dash_charts/utils_app.py` | 98 | 14 | 6 | 85.7% |\n| `dash_charts/utils_app_modules.py` | 26 | 3 | 4 | 88.5% |\n| `dash_charts/utils_app_with_navigation.py` | 119 | 9 | 6 | 92.4% |\n| `dash_charts/utils_callbacks.py` | 34 | 6 | 0 | 82.4% |\n| `dash_charts/utils_data.py` | 63 | 1 | 0 | 98.4% |\n| `dash_charts/utils_dataset.py` | 76 | 43 | 0 | 43.4% |\n| `dash_charts/utils_fig.py` | 77 | 2 | 4 | 97.4% |\n| `dash_charts/utils_helpers.py` | 17 | 8 | 7 | 52.9% |\n| `dash_charts/utils_json_cache.py` | 51 | 10 | 0 | 80.4% |\n| `dash_charts/utils_static.py` | 111 | 5 | 0 | 95.5% |\n| `dash_charts/utils_static_toc.py` | 22 | 1 | 0 | 95.5% |\n\nGenerated on: 2020-11-08T22:46:27.420973\n\n<!-- /COVERAGE -->\n\n## External Links\n\n[![codebeat badge](https://codebeat.co/badges/9d2b9a53-9203-4467-8cde-68f24e930389)](https://codebeat.co/projects/github-com-kyleking-dash_charts-main) [![Documentation Site Online Status Badge](https://img.shields.io/website?label=Doc%20Site&up_message=up&url=https%3A%2F%2Fgithub.com%2FKyleKing%2Fdash_charts%2F)](https://kyleking.me/dash_charts/) ![Commits Since last Release](https://img.shields.io/github/commits-since/KyleKing/dash_charts/latest) ![Last Commit Badge](https://img.shields.io/github/last-commit/kyleking/dash_charts)\n\nUseful Dash reference links\n\n- Official [Dash Docs](https://dash.plot.ly) / [Plotly Docs](https://plot.ly/python/)\n- Example Apps\n    - Pretty annotations from [Market Ahead](https://www.marketahead.com/p/FOX), a commercial Dash app\n    - Pleasant dark app from Dash-Bio, [Circos](https://github.com/plotly/dash-bio/blob/main/tests/dashbio_demos/app_circos.py)\n    - All [Dash Sample Apps](https://github.com/plotly/dash-sample-apps/tree/main/apps)\n- Code Conceptual inspiration\n    - [On Tidy data](https://www.jeannicholashould.com/tidy-data-in-python.html)\n    - [Using field for properties in a @DataClass](https://florimond.dev/blog/articles/2018/10/reconciling-dataclasses-and-properties-in-python/)\n- Best Practices\n    - IBM Design Language (note that v2 doesn\'t have documentation on visualization yet)\n        - [Process](https://www.ibm.com/design/v1/language/experience/data-visualization/process/)\n        - [Select Chart](https://www.ibm.com/design/v1/language/experience/data-visualization/chart-models/)\n            - Alt [Chart Decision Diagram](https://www.tatvic.com/blog/7-visualizations-learn-r/)\n        - [Principles (color conventions, etc.)](https://www.ibm.com/design/v1/language/experience/data-visualization/visualization/)\n        - [Color Library and Data Vis Swatches](https://www.ibm.com/design/v1/language/resources/color-library/)\n        - [Interaction](https://www.ibm.com/design/v1/language/experience/data-visualization/interaction/)\n        - [Style](https://www.ibm.com/design/v1/language/experience/data-visualization/style/)\n- Colors\n    - [Friendly Guide to Colors](https://lisacharlotterost.de/2016/04/22/Colors-for-DataVis/)\n    - [Viz-Palette Tool](https://projects.susielu.com/viz-palette)\n    - [AI Color Palette](http://colormind.io/) / [Coolors](https://coolors.co/2b303a-92dce5-eee5e9-7c7c7c-d64933)\n- Reference\n    - [Pandas CheatSheet for Data Manipulation](https://github.com/pandas-dev/pandas/blob/main/doc/cheatsheet/Pandas_Cheat_Sheet.pdf)\n\n<!-- TODO: See https://github.com/KyleKing/calcipy/issues/38 -->\n\n## Installation\n\n1. ...\n2. ...\n\n    ```sh\n    echo Hello World!\n    ```\n\n3. ...\n\n## Usage\n\n<!-- TODO: Show an example (screenshots, terminal recording, etc.) -->\n\nFor more examples, see [Scripts](https://github.com/kyleking/dash_charts/scripts) or [Tests](https://github.com/kyleking/dash_charts/tests)\n\n## Roadmap\n\nSee the `Open Issues` and `Milestones` for current status and [./docs/CODE_TAG_SUMMARY.md](./docs/CODE_TAG_SUMMARY.md) for annotations in the source code.\n\nFor release history, see the [./docs/CHANGELOG.md](./docs/CHANGELOG.md)\n\n## Contributing\n\nSee the Developer Guide, Contribution Guidelines, etc\n\n- [./docs/DEVELOPER_GUIDE.md](./docs/DEVELOPER_GUIDE.md)\n- [./docs/STYLE_GUIDE.md](./docs/STYLE_GUIDE.md)\n- [./docs/CONTRIBUTING.md](./docs/CONTRIBUTING.md)\n- [./docs/CODE_OF_CONDUCT.md](./docs/CODE_OF_CONDUCT.md)\n- [./docs/SECURITY.md](./docs/SECURITY.md)\n\n## License\n\n[LICENSE](https://github.com/kyleking/dash_charts/LICENSE)\n',
    'author': 'Kyle King',
    'author_email': 'dev.act.kyle@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kyleking/dash_charts',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
