"""Fragua Environment Class."""

from typing import Any, Callable, Dict, Iterable, List, Optional, Union

from fragua.builders.pipeline_builder import FraguaPipelineBuilder
from fragua.core.agent import FraguaAgent
from fragua.core.box import FraguaBox
from fragua.core.pipeline import FraguaPipeline
from fragua.core.registry import FraguaRegistry
from fragua.core.step import FraguaStep
from fragua.builders.step_builder import FraguaStepBuilder
from fragua.core.step_index import FraguaStepIndex
from fragua.core.warehouse import FraguaWarehouse
from fragua.core.set import FraguaSet
from fragua.utils.logger import get_logger

logger = get_logger("env_logger")


class FraguaEnvironment:
    """
    Execution environment for Fragua.

    An environment owns exactly one agent, one registry
    and one runtime warehouse.
    """

    name: str
    agent: FraguaAgent
    registry: FraguaRegistry
    warehouse: FraguaWarehouse
    step_index: FraguaStepIndex

    def __init__(self, name: str) -> None:
        self.name = name
        self.agent = FraguaAgent(name=f"{name}_agent")
        self.registry = self._initialize_registry()
        self.warehouse = FraguaWarehouse(name=f"{name}_warehouse")
        self.step_index = FraguaStepIndex()

    def _initialize_registry(self) -> FraguaRegistry:
        return FraguaRegistry(
            sets={
                "pipelines": FraguaSet("pipelines", step_enabled=False),
            }
        )

    # -------------------- Public API -------------------- #

    def add_sets(self, *sets: FraguaSet) -> None:
        """
        Register one or more FraguaSet instances into the environment.

        This method:
        - Registers each set in the FraguaRegistry
        - Automatically creates and indexes FraguaStepBuilder templates
        for callable items when step_enabled=True

        Args:
            sets: One or more FraguaSet instances.
        """
        for fragua_set in sets:
            # 1. Register set in registry
            if self.registry.get_set(fragua_set.name) is not None:
                raise ValueError(f"Registry set already exists: {fragua_set.name}")

            self.registry.add_set(fragua_set)

            # 2. Skip step generation if disabled
            if not fragua_set.step_enabled:
                continue

            # 3. Create StepBuilders for callable items only
            for item_name in fragua_set.list():
                fn = fragua_set.get_function(item_name)
                if fn is None:
                    # Skip pipelines or non-callables
                    continue

                builder = FraguaStepBuilder(
                    set_name=fragua_set.name,
                    function=item_name,
                )

                self.step_index.register(
                    name=item_name,
                    builder=builder,
                )

    def create_pipeline(self, name: str) -> FraguaPipelineBuilder:
        """
        Create a new pipeline builder.

        Args:
            name: Name of the pipeline.

        Returns:
            FraguaPipelineBuilder instance for constructing the pipeline.
        """
        if self.registry.get_set("pipelines") is None:
            raise RuntimeError("Pipelines registry set is not initialized")

        return FraguaPipelineBuilder(name)

    def add_pipelines(
        self,
        *pipelines: Dict[str, Any],
    ) -> None:
        """
        Register one or more FraguaPipeline instances
        in the pipelines registry set.
        """
        target_set = self.registry.get_set("pipelines")

        if target_set is None:
            raise ValueError("'Pipelines' set not found.")

        for pipeline in pipelines:
            if not isinstance(pipeline, dict):
                raise TypeError(
                    "register_pipelines only accepts declarative pipeline definition: Dict[str,Any]"
                )

            pipeline_name = pipeline["name"]

            registered = target_set.register(pipeline_name, pipeline)
            if not registered:
                raise ValueError(f"Pipeline '{pipeline_name}' is already registered")

    def replace_pipeline(
        self,
        definition: Dict[str, Any],
    ) -> None:
        """
        Replace an existing pipeline definition.

        The pipeline must already exist.
        """
        if not isinstance(definition, dict):
            raise TypeError("Pipeline definition must be a dictionary")

        if "name" not in definition:
            raise ValueError("Pipeline definition must include a 'name'")

        pipeline_set = self._get_registry_set("pipelines")
        name = definition["name"]

        if not pipeline_set.exists(name):
            raise ValueError(f"Pipeline '{name}' does not exist")

        pipeline_set.remove(name)
        pipeline_set.register(name, definition)

        logger.info("Replaced pipeline definition: %s", name)

    def update_pipeline(
        self,
        name: str,
        updates: Dict[str, Any],
    ) -> None:
        """
        Update an existing pipeline definition partially.
        """
        if not isinstance(updates, dict):
            raise TypeError("Updates must be a dictionary")

        pipeline_set = self._get_registry_set("pipelines")

        existing = pipeline_set.get(name)
        if existing is None:
            raise ValueError(f"Pipeline '{name}' does not exist")

        if not isinstance(existing, dict):
            raise TypeError(f"Registered pipeline '{name}' is not declarative")

        updated = dict(existing)

        if "name" in updates and updates["name"] != name:
            raise ValueError("Pipeline name cannot be changed")

        for key, value in updates.items():
            if key == "name":
                continue
            updated[key] = value

        pipeline_set.remove(name)
        pipeline_set.register(name, updated)

        logger.info("Updated pipeline definition: %s", name)

    def execute_pipeline(
        self,
        pipeline: Union[FraguaPipeline, str, Dict[str, Any]],
    ) -> FraguaBox:
        """
        Execute a pipeline from multiple possible representations.
        """
        if isinstance(pipeline, FraguaPipeline):
            logger.info(
                "Executing FraguaPipeline instance: %s",
                pipeline.name,
            )
            return self._run_pipeline(pipeline)

        if isinstance(pipeline, str):
            logger.info(
                "Executing pipeline by name: %s",
                pipeline,
            )
            compiled = self._resolve_pipeline(pipeline)
            return self._run_pipeline(compiled)

        if isinstance(pipeline, dict):
            logger.info(
                "Compiling pipeline from inline definition",
            )
            compiled = self._compile_pipeline(pipeline)
            logger.info(
                "Executing compiled pipeline: %s",
                compiled.name,
            )
            return self._run_pipeline(compiled)

        raise TypeError(
            "execute_pipeline expects a FraguaPipeline, "
            "a pipeline name (str), or a pipeline definition (dict)"
        )

    def create_transform_steps(
        self,
        *,
        step_names: Iterable[str],
        step_prefix: str,
        start_from: Optional[str],
    ) -> List[FraguaStep]:
        """
        Create a chain of transform steps.

        Args:
            step_names: Names of the steps to create from the step index.
            step_prefix: Prefix for naming saved results.
            start_from: Optional name of the step to start from.

        Returns:
            List of created FraguaStep instances.
        """
        steps: List[FraguaStep] = []
        previous_step_name: Optional[str] = start_from

        for index, step_name in enumerate(step_names, start=1):
            builder: Optional[FraguaStepBuilder] = self.step_index.get(step_name)
            if builder is None:
                raise ValueError(f"Step not found in StepIndex: {step_name}")

            save_as: str = f"{step_prefix}_step_{index}"
            builder = builder.with_params().with_save_as(save_as)

            if previous_step_name is not None:
                builder = builder.with_use(previous_step_name)

            step: FraguaStep = builder.build()
            steps.append(step)
            previous_step_name = save_as

        return steps

    def add_functions(
        self,
        *functions: Callable[..., Any],
        set_name: str,
    ) -> None:
        """
        Register one or more callable functions in a registry set.

        If the target set has step_enabled=True, corresponding
        FraguaStepBuilder templates are automatically indexed.
        """
        target_set = self.registry.get_set(set_name)

        if target_set is None:
            raise ValueError(f"{set_name} set not found.")

        for fn in functions:
            if not callable(fn):
                raise TypeError("register_functions only accepts callable objects")

            fn_name = fn.__name__

            registered = target_set.register(fn_name, fn)
            if not registered:
                raise ValueError(
                    f"Function '{fn_name}' is already registered in set '{set_name}'"
                )

            if target_set.step_enabled:
                builder = FraguaStepBuilder(
                    set_name=target_set.name,
                    function=fn_name,
                )

                self.step_index.register(
                    name=fn_name,
                    builder=builder,
                )

    # -------------------- Helpers -------------------- #
    def _run_pipeline(self, pipeline: FraguaPipeline) -> FraguaBox:
        """
        Execute a pipeline and return the result.

        Args:
            pipeline: FraguaPipeline instance or name of registered pipeline.

        Returns:
            FraguaBox containing the pipeline execution results.
        """
        box = self.agent.run_pipeline(
            pipeline=pipeline,
            registry=self.registry,
        )
        self.warehouse.store(box)
        return box

    def _compile_pipeline(self, definition: Dict[str, Any]) -> FraguaPipeline:
        """
        Compile a pipeline from a dictionary definition.

        Args:
            definition: Dictionary containing pipeline name and steps.

        Returns:
            Compiled FraguaPipeline ready for execution.
        """
        self._validate_pipeline_definition(definition)

        expanded_steps: List[Dict[str, Any]] = []

        for step in definition["steps"]:
            if self._is_macro(step):
                macro_steps: List[Dict[str, Any]] = self._expand_macro(step)
                expanded_steps.extend(macro_steps)
            else:
                expanded_steps.append(step)

        normalized_steps: List[Dict[str, Any]] = [
            self._normalize_step(step) for step in expanded_steps
        ]

        self._validate_step_dependencies(normalized_steps)
        self._validate_registry_bindings(normalized_steps)

        compiled_steps: List[FraguaStep] = [
            self._compile_step(step) for step in normalized_steps
        ]

        pipeline: FraguaPipeline = FraguaPipeline(name=definition["name"])
        pipeline.add(compiled_steps)

        return pipeline

    def _resolve_pipeline(self, pipeline: str) -> FraguaPipeline:
        """
        Resolve a pipeline name to a compiled FraguaPipeline.
        """
        pipeline_set = self._get_registry_set("pipelines")

        definition = pipeline_set.get(pipeline)
        if definition is None:
            raise ValueError(f"Pipeline not found: {pipeline}")

        if not isinstance(definition, dict):
            raise TypeError(
                f"Registered pipeline '{pipeline}' is not a pipeline definition"
            )

        logger.info("Compiling registered pipeline: %s", pipeline)
        return self._compile_pipeline(definition)

    def _validate_pipeline_definition(self, definition: Dict[str, Any]) -> None:
        if not isinstance(definition, dict):
            raise ValueError("Pipeline definition must be a dictionary.")

        if "name" not in definition:
            raise ValueError("Pipeline definition must include a 'name'.")

        if "steps" not in definition:
            raise ValueError("Pipeline definition must include 'steps'.")

        if not isinstance(definition["steps"], list):
            raise ValueError("'steps' must be a list.")

        if not definition["steps"]:
            raise ValueError("Pipeline must contain at least one step.")

    def _compile_step(self, step: Dict[str, Any]) -> FraguaStep:
        function_set: Optional[FraguaSet] = self.registry.get_set(step["set"])
        if function_set is None:
            raise ValueError("Function set not found.")

        if not function_set.step_enabled:
            return FraguaStep(
                set_name=step["set"],
                function=step["function"],
                params=step["params"],
                save_as=step["save_as"],
                use=step["use"],
            )

        builder: FraguaStepBuilder = FraguaStepBuilder(
            set_name=step["set"],
            function=step["function"],
        )

        if step["params"]:
            builder.with_params(**step["params"])

        if step["save_as"]:
            builder.with_save_as(step["save_as"])

        if step["use"]:
            builder.with_use(step["use"])

        return builder.build()

    def _normalize_step(self, raw_step: Dict[str, Any]) -> Dict[str, Any]:
        if not isinstance(raw_step, dict):
            raise ValueError("Each step must be a dictionary.")

        if "set" not in raw_step or "function" not in raw_step:
            raise ValueError("Each step must define 'set' and 'function'.")

        return {
            "set": raw_step["set"],
            "function": raw_step["function"],
            "params": raw_step.get("params", {}),
            "save_as": raw_step.get("save_as"),
            "use": raw_step.get("use"),
        }

    def _validate_step_dependencies(self, steps: List[Dict[str, Any]]) -> None:
        available_keys: set[str] = set()

        for step in steps:
            use: Optional[str] = step.get("use")
            save_as: Optional[str] = step.get("save_as")

            if use and use not in available_keys:
                raise ValueError(f"Step uses undefined result '{use}'.")

            if save_as:
                if save_as in available_keys:
                    raise ValueError(f"Duplicate save_as key '{save_as}'.")
                available_keys.add(save_as)

    def _get_registry_set(self, set_name: str) -> FraguaSet:
        target_set: Optional[FraguaSet] = self.registry.get_set(set_name)
        if target_set is None:
            raise ValueError(f"Unknown registry set: {set_name}")
        return target_set

    def _validate_registry_bindings(self, steps: List[Dict[str, Any]]) -> None:
        for step in steps:
            function_set: Optional[FraguaSet] = self.registry.get_set(step["set"])
            if function_set is None:
                raise ValueError(f"Registry set not found: {step['set']}")

            fn: Optional[Callable[..., Any]] = function_set.get_function(
                step["function"]
            )
            if fn is None:
                raise ValueError(
                    f"Function '{step['function']}' not found in set '{step['set']}'"
                )

    def _is_macro(self, step_def: Dict[str, Any]) -> bool:
        return "macro" in step_def

    def _expand_macro(self, macro_def: Dict[str, Any]) -> List[Dict[str, Any]]:
        steps_def = macro_def["steps"]

        if not isinstance(steps_def, list) or not steps_def:
            raise ValueError("Macro 'steps' must be a non-empty list")

        expanded: List[Dict[str, Any]] = []

        previous_save_as: Optional[str] = macro_def.get("start_from")
        prefix: str = macro_def.get("step_prefix", "step")
        final_save_as: Optional[str] = macro_def.get("save_as")

        for index, step_def in enumerate(steps_def, start=1):
            function = step_def["function"]
            params = step_def.get("params", {})

            save_as = f"{prefix}_{index}"
            is_last = index == len(steps_def)

            expanded.append(
                {
                    "set": macro_def.get("set", "transform"),
                    "function": function,
                    "params": params,
                    "use": previous_save_as,
                    "save_as": final_save_as if is_last and final_save_as else save_as,
                }
            )

            previous_save_as = save_as

        return expanded
