"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RateLimitedApiKey = exports.ApiKey = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const usage_plan_1 = require("./usage-plan");
/**
 * Base implementation that is common to the various implementations of IApiKey
 */
class ApiKeyBase extends core_1.Resource {
    /**
     * Permits the IAM principal all read operations through this key.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantRead(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: readPermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all write operations through this key.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: writePermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * Permits the IAM principal all read and write operations through this key.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [...readPermissions, ...writePermissions],
            resourceArns: [this.keyArn],
        });
    }
}
/**
 * An API Gateway ApiKey.
 *
 * An ApiKey can be distributed to API clients that are executing requests
 * for Method resources that require an Api Key.
 *
 * @stability stable
 */
class ApiKey extends ApiKeyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        const resource = new apigateway_generated_1.CfnApiKey(this, 'Resource', {
            customerId: props.customerId,
            description: props.description,
            enabled: props.enabled || true,
            generateDistinctId: props.generateDistinctId,
            name: this.physicalName,
            stageKeys: this.renderStageKeys(props.resources),
            value: props.value,
        });
        this.keyId = resource.ref;
        this.keyArn = core_1.Stack.of(this).formatArn({
            service: 'apigateway',
            account: '',
            resource: '/apikeys',
            sep: '/',
            resourceName: this.keyId,
        });
    }
    /**
     * Import an ApiKey by its Id.
     *
     * @stability stable
     */
    static fromApiKeyId(scope, id, apiKeyId) {
        class Import extends ApiKeyBase {
            constructor() {
                super(...arguments);
                this.keyId = apiKeyId;
                this.keyArn = core_1.Stack.of(this).formatArn({
                    service: 'apigateway',
                    account: '',
                    resource: '/apikeys',
                    sep: '/',
                    resourceName: apiKeyId,
                });
            }
        }
        return new Import(scope, id);
    }
    renderStageKeys(resources) {
        if (!resources) {
            return undefined;
        }
        return resources.map((resource) => {
            const restApi = resource;
            const restApiId = restApi.restApiId;
            const stageName = restApi.deploymentStage.stageName.toString();
            return { restApiId, stageName };
        });
    }
}
exports.ApiKey = ApiKey;
_a = JSII_RTTI_SYMBOL_1;
ApiKey[_a] = { fqn: "@aws-cdk/aws-apigateway.ApiKey", version: "1.122.0" };
/**
 * An API Gateway ApiKey, for which a rate limiting configuration can be specified.
 *
 * @stability stable
 * @resource AWS::ApiGateway::ApiKey
 */
class RateLimitedApiKey extends ApiKeyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        const resource = new ApiKey(this, 'Resource', props);
        if (props.apiStages || props.quota || props.throttle) {
            new usage_plan_1.UsagePlan(this, 'UsagePlanResource', {
                apiKey: resource,
                apiStages: props.apiStages,
                quota: props.quota,
                throttle: props.throttle,
            });
        }
        this.keyId = resource.keyId;
        this.keyArn = resource.keyArn;
    }
}
exports.RateLimitedApiKey = RateLimitedApiKey;
_b = JSII_RTTI_SYMBOL_1;
RateLimitedApiKey[_b] = { fqn: "@aws-cdk/aws-apigateway.RateLimitedApiKey", version: "1.122.0" };
const readPermissions = [
    'apigateway:GET',
];
const writePermissions = [
    'apigateway:POST',
    'apigateway:PUT',
    'apigateway:PATCH',
    'apigateway:DELETE',
];
//# sourceMappingURL=data:application/json;base64,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